/*
 * QUSOFT MICROSYSTMES
 * Moka
 * Copyright 2002 Frdric Brown
 */

/*
 *  This file is part of Moka API.
 *
 *  Moka API is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Moka API is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Moka API; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package moka.x;

import moka.lang.Long;
import moka.lang.Double;
import moka.lang.Character;
import moka.x.CaptionedComponent;

/**
 * TextField class is a text entry component.
 *
 * @author  Frdric Brown
 * @version 1.0, 2002-11-04
 * @since   MDK1.0a
 */
public class TextField extends CaptionedComponent {

	/** The TextField accepts any character.*/
	public static final short NO_INPUT_TYPE = 0;
	/** The TextField only accepts integers.*/
	public static final short INT_INPUT_TYPE = 1;
	/** The TextField only accepts integers and floats.*/
	public static final short FLOAT_INPUT_TYPE = 2;
	/** The TextField only accepts characters.*/
	public static final short CHAR_INPUT_TYPE = 3;

	/** The number of character this TextField should accept. Should be > 0.*/
	public short maxLength;

	/** The input type of this TextField*/
	public short inputType;

	/** The mask for this TextField. '\0' (default) to disable. Should be only read.*/
	public char mask;

	/*
	 * Constructs a new TextField.
	 */
	public TextField () {
		this("");
	}

	/*
	 * Constructs a new TextField specifying its caption.
	 *
	 * @param caption the TextField's caption.
	 */
	public TextField (String caption) {
		super(caption);
		this.maxLength = 255;
		this.inputType = TextField.NO_INPUT_TYPE;
		this.mask = 0;
	}

	/**
	 * Paints the textfield.
	 */
	public native void paint() {
		/*int x1 = this->width - 1;
		int y1 = this->height - 1;

		DrawLine(0, 0, x1, 0, A_NORMAL);
		DrawLine(0, y1, x1, y1, A_NORMAL);
		DrawLine(0, 1, 0, (y1-1), A_NORMAL);
		DrawLine(x1, 1, x1, (y1-1), A_NORMAL);*/

		if (this->font != Font_FONT_SYSTEM) {
			FontSetSys(this->font);
		}

		if (this->mask) {
			short len = strlen(this->caption->value);
			char buffer[len + 1];
			memset(buffer, this->mask, len);
			buffer[len] = '\0';
			DrawStr(2, 2, buffer, A_NORMAL);
		}
		else {
			DrawStr(2, 2, this->caption->value, A_NORMAL);
		}

		if (this->font != Font_FONT_SYSTEM) {
			FontSetSys(GEM_gem->systemFont);
		}

		DrawClipRect (&(WIN_RECT){0, 0,  this->width - 1, this->height - 1}, GEM_vScrRect, A_NORMAL);
	}

	/**
	 * This method is invoked by the GEM when the user interacts
	 * with the TextField.
	 */
	public void use () {
	    char buffer[this.maxLength + 1];
	    char mBuffer[this.maxLength + 1];
	    short ind = 0;
	    short key;
		short x = 0;
		short y = 0;
		short max = this.width / GEM.getFontWidth(this.font);
		Container ct = this.parent;

		if (!this.enabled) {
			return;
		}

		while (ct) {
			x = x + ct.x;
			y = y + ct.y;
			ct = ct.parent;
		}

		x = this.x + x + 2;
		y = this.y + y + 2;

		if (this.font != Font.FONT_SYSTEM) {
    		native {
				FontSetSys(this->font);
			}
		}

		native {
			if (strlen(this->caption->value) > this->maxLength) {
				this->setCaption_String(this, this->caption->substring_long_int_long_int(this, 0, this->maxLength));
			}
			strcpy(buffer, this->caption->value);
			ind = strlen(buffer);
			if (ind < max) {
				MoveTo(x, y);
				if (this->mask) {
					mBuffer[strlen(buffer)] = '\0';
					printf(memset(mBuffer, this->mask, strlen(buffer)));
				}
				else {
					printf(buffer);
				}
			}
		}

	    	while (true) {
	    		//key = ngetchx();
	    		key = native ( GKeyIn(NULL, 0) );
	    		if (key != KEY_ENTER) {
	    			switch(key) {
	    				case KEY_BACKSPACE: {
	    					//if i != 0, decrement ind
	    					if (ind) {
	    						buffer[ind] = '\0';
	    						buffer[--ind] = ' ';
	    					}
	    					break;
	    				}
	    				case KEY_CLEAR: {
	    					/*if (ind >= max) {
	    						ind = max - 1;
	    					}
	
	    					native {
	    						memset(buffer, ' ', ind);
	    					}
	    					buffer[ind] = '\0';*/
	    					native {
	    						ScrRectFill(&(SCR_RECT){{x, y, x + this->width - 4, y + this->height - 4}}, &(SCR_RECT){{x-this->x-2, y-this->y-2, x-this->x+this->parent->width-3, y-this->y+this->parent->height-3}}, A_REVERSE);
	    					}
	    					ind = 0;
	    					buffer[ind] = '\0';
	    					break;
	    				}
	    				default: {
	    					if (ind >= this.maxLength || ! this.checkInput(key, (char*)buffer, ind)) {
		    					break;
		    				}
		    				ind++;
		    				/*buffer[ind++] = key;
		    				buffer[ind] = '\0';*/
		    			}
		    		}
					native {
						if (ind < max) {
							MoveTo(x, y);
							if (this->mask) {
								mBuffer[strlen(buffer)] = '\0';
								memset(&mBuffer[ind], ' ', strlen(buffer) - ind);
								printf(memset(mBuffer, this->mask, ind));
							}
							else {
								printf(buffer);
							}
	
						}
					}
					buffer[ind] = '\0';
	    		}
	    		else {
	    			break;
	    		}
	    	}

	    	if (this.font != Font.FONT_SYSTEM) {
	    		native {
					FontSetSys(GEM_gem->systemFont);
				}
			}
	
	    	this.setCaption(new String(Character.copyValueOf((char*)buffer), false));
	
	    	super.use();
	}



    /**
     * Checks if the specified key is a valid input for the
     * input type of this TextField.
     *
     * @param key The key to verify.
     * @return true if the specified key is a valid input, false otherwize.
     */
    private boolean checkInput (short key, char* buffer, short ind) {
    	String str;
    	boolean result = true;

    	if (key < 13 && key > 255) {
    		return false;
    	}

    	buffer[ind] = key;
    	buffer[ind+1] = '\0';

    	str = new String(buffer);

    	switch (this.inputType) {
    		case TextField.INT_INPUT_TYPE : {
    			result = Long.isLong(str);
    			break;
    		}
    		case TextField.FLOAT_INPUT_TYPE : {
    			result = Double.isDouble(str);
    			break;
    		}
    		case TextField.CHAR_INPUT_TYPE : {
    			result = Character.isLetter((char)key);
    			break;
    		}
    	}

    	if (! result) {
    		buffer[ind] = '\0';
    	}

    	str.finalize();

    	return result;
    }
    
 	/**
 	 * Sets the mask of the TextField. Set to '\0' to disable mask.
 	 *
 	 * @param mask The mask of the TextField.
 	 */
	public void setMask (char mask) {
		this.mask = mask;
		
		this.repaint = true;
		if (this.visible && this.parent) {
			this.parent.onRefresh();
		}
	}
}