/*
 * QUSOFT MICROSYSTMES
 * Moka
 * Copyright 2002 Frdric Brown
 */

/*
 *  This file is part of Moka API.
 *
 *  Moka API is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Moka API is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Moka API; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package moka.x;

import moka.x.CaptionedComponent;
import moka.lang.System;

/**
 * TextArea class is a text entry component.
 *
 * @author  Frdric Brown
 * @version 1.5, 2004-11-14
 * @since   MDK1.0a
 */
public class TextArea extends CaptionedComponent {
	/** The number of character this TextField should accept. Should be > 0.*/
	public short maxLength;

	/** Should the words be warped ? Should be only read.*/
	public boolean warp;

	/** Is the vertical scroll bar should be visible ? Should be only read. Use setVScrollVisible to set value. */
	public boolean vScrollVisible;

    /** The index of the first line. Should be only read. Use setScrollLine to set value. */
    public int scrollLine;
    
    /** Is the text editable ? */
    public boolean editable;
    
    /* The total number of lines. */
    private int totalLines;
    /* The first char visible. */
    private int firstIndex;
    /* The last char visible. */
    private int lastIndex;

	/*
	 * Constructs a new TextArea.
	 */
	public TextArea () {
		this("");
	}

	/*
	 * Constructs a new TextArea specifying its caption.
	 *
	 * @param caption the TextArea's caption.
	 */
	public TextArea (String caption) {
		super(caption);
		this.maxLength = 2048;
		this.warp = true;
		/* Scroll support */
		this.vScrollVisible = false;
		this.scrollLine = 0;
		this.totalLines = 0;
		this.editable = true;
	}

	/**
	 * Sets if the words should be warped.
	 *
	 * @param b true to specify that the words should be warped, false to specify that the words should not be warped
	 */
	public void setWarp(boolean b) {
		this.warp = b;
	}

	/**
	 * Sets if the vertical scroll bar should be visible.
	 *
	 * @param b true to specify that the vertical scroll bar should be visible
	 */
	public void setVScrollVisible(boolean b) {
		this.vScrollVisible = b;

		this.repaint = true;
		if (this.visible && this.parent) {
			this.parent.onRefresh();
		}
	}

	/**
	 * Sets the scroll position. In other words, the first line to be
	 * displayed by the TextArea.
	 *
	 * @param num the number of the first line (should be >= 0 and < to the total number of lines).
	 */
	public void setScrollLine(int num) {
		this.scrollLine = num;

		this.repaint = true;
		if (this.visible && this.parent) {
			this.parent.onRefresh();
		}
	}

	/**
	 * Paints the TextArea.
	 */
	public native void paint() {
		short height;
		short nbL;
		short sh;
		short sLength;
		
		if (this->font != Font_FONT_SYSTEM) {
			FontSetSys(this->font);
		}

		if (this->warp) {
			this->printTextWarp_char_p_short_int_short_int(this, this->caption->value, 0, 0);
		}
		else {
			this->printTextNoWarp_char_p_short_int_short_int_short_int(this, this->caption->value, 0, 0, -1);
		}

		if (this->font != Font_FONT_SYSTEM) {
			FontSetSys(GEM_gem->systemFont);
		}

		if (this->vScrollVisible && !this->warp) {
			height = GEM_getFontHeight_short_int(this->font) + 1;
			nbL = this->height / height;
		
			if (this->totalLines && (this->totalLines >= nbL)) {
				sLength = nbL * (this->height - 38) / this->totalLines;
				sh = this->scrollLine * (this->height - 37) / this->totalLines;
			}
			else {
				sLength = (this->height - 38);
				sh = 0;
			}
			
			DrawLine(this->width-12, 1, this->width-12, this->height-12, A_NORMAL);
			
			Sprite8(this->width - 10, 2, 8, (unsigned char*)ICON_UPARROW, GEM_gem->port, SPRT_OR);
			DrawLine(this->width-11, 11, this->width-2, 11, A_NORMAL);
			
			DrawLine(this->width-11, this->height-23, this->width-2, this->height-23, A_NORMAL);
			Sprite8(this->width - 10, this->height-21, 8, (unsigned char*)ICON_DNARROW, GEM_gem->port, SPRT_OR);
			
			ScrRectFill (&(SCR_RECT){{this->width-12, this->height-12, this->width-2, this->height-2}}, GEM_vScrRect, A_NORMAL);
			
			ScrRectFill (&(SCR_RECT){{this->width-10, sh + 13, this->width-3, sh + 13 + sLength}}, GEM_vScrRect, A_NORMAL);
		}

		DrawClipRect (&(WIN_RECT){0, 0,  this->width - 1, this->height - 1}, GEM_vScrRect, A_NORMAL);
	}

	/**
     * Returns the next token from a specified string to a
     * specified buffer.
     *
	 * @param str The string
	 * @param buf The buffer that will receive the token
     * @return    a pointer after the token in str.
	 */
	private static char[] returnToken (char[] str, char[] buf) {
		//TAB, NL, FF, CR, SPACE
    	native {
    		char delim[] = {9, 10, 12, 13, 32};
    	}
    	int len = native.strlen(str);
    	int end;
    	int i;
    	int n;
    	char* pos = str;

		while (*pos) {
			for (i = 0; i < 5; i++) {
				if (*pos == delim[i]) {
					break;
				}
			}

			if (i == 5) {
				break;
			}

			pos ++;
		}
		
		str = pos;
		
		while (*pos) {
			for (i = 0; i < 5; i++) {
				if (*pos == delim[i]) {
					break;
				}
			}

			if (i < 5) {
				break;
			}

			pos ++;
		}

		end = (int)(pos - str) ;
		native.memcpy(buf, str, end);

		buf[end] = '\0';

    	return pos;
	}

	/**
	 * Prints the text in this TextArea while warping words.
	 *
	 * @param str The text to print
	 * @param x The x position
	 * @param y The y position
	 */
	private native void printTextWarp (char[] str, short x, short y) {
		short i;
		short n;
		short z;
		short pos = 0;
		short width = GEM_getFontWidth_short_int(this->font);
		short height = GEM_getFontHeight_short_int(this->font) + 1;
		short nbL = this->height / height;
		//short nbC = this->getNbC_(this);
		short nbC = (this->width - 3) / width;
		short len = strlen(str);
		short l;
		//TStringTokenizer* tok = TStringTokenizer_String(TString_char_p(str));
		//TString* s;
		char *s = malloc(this->maxLength + 1) ;
		char *ptr = str;

		for (i = 0; i < nbL && pos < len; i++) {
			for (n = 0; n < nbC && pos < len; n++) {
				if (str[pos] == ' ') {
					pos++;
					if (n < 1) {
						n--;
					}
				}
				else if (str[pos] == '\n') {
					pos++;
					i++;
					n = -1;
				}
				else {

					ptr = TextArea_returnToken_char_p_char_p(ptr, s);

					l = strlen(s);

					pos += l;

					if (l + n >= nbC && n > 0) {
						i++;
						n = 0;
					}

					for (z = 0; z < l && i < nbL; z++) {

						DrawChar(x + n * width + 2, y + i * height + 2, s[z], A_NORMAL);

						if (z + 1 < l) {
							if (n + 1 >= nbC)  {
								i++;
								n = 0;
							}
							else {
								n++;
							}
						}
					}
				}
			}
		}

		free(s);
	}

	/**
	 * Returns the number of characters per line this TextArea displays.
	 *
	 * @return the number of characters per line.
	 */
	private short getNbC() {
		short width = GEM.getFontWidth(this.font);
		if (this.vScrollVisible) {
			return (this.width - 5 - 12) / width;
		}
		else {
			return (this.width - 5) / width;
		}
	}

	/**
	 * Prints the text in this TextArea without warping words.
	 *
	 * @param str The text to print
	 * @param x The x position
	 * @param y The y position
	 * @param cur The current position
	 */
	private native void printTextNoWarp (char[] str, short x, short y, short cur) {
		short i;
		short n = 0;
		short z;
		short w;
		short pos = 0;
		short width = GEM_getFontWidth_short_int(this->font);
		short height = GEM_getFontHeight_short_int(this->font) + 1;
		short nbL = this->height / height;
		short nbC = this->getNbC_(this);
		short len = strlen(str);
		
		if (this->vScrollVisible && cur >= 0) {			
			DrawLine(this->width-12 + x - 1, 1 + y - 1, this->width-12 + x - 1, this->height-12 + y - 1, A_NORMAL);
			
			Sprite8(this->width - 10 + x - 1, 2 + y - 1, 8, (unsigned char*)ICON_UPARROW, LCD_MEM, SPRT_OR);
			DrawLine(this->width-11 + x - 1, 11 + y - 1, this->width-2 + x - 1, 11 + y - 1, A_NORMAL);
			
			DrawLine(this->width-11 + x - 1, this->height-23 + y - 1, this->width-2 + x - 1, this->height-23 + y - 1, A_NORMAL);
			Sprite8(this->width - 10 + x - 1, this->height-21 + y - 1, 8, (unsigned char*)ICON_DNARROW, LCD_MEM, SPRT_OR);
			ScrRectFill (&(SCR_RECT){{this->width-12 + x - 1, this->height-12 + y - 1, this->width-2 + x - 1, this->height-2 + y - 1}}, ScrRect, A_NORMAL);
		}
		
		this->totalLines = 0;
		
		for (i = 0; (i < this->scrollLine) && (pos < len); i++) {
			for (n = 0; n < nbC && pos < len; n++) {
				if (str[pos ++] == '\n') {
					break;
				}
			}
			this->totalLines ++;
		}
		
		this->firstIndex = pos;

		for (i = 0; i < nbL && pos < len; i++) {
			for (n = 0; n < nbC && pos < len; n++) {
				if (str[pos] == '\n') {
					if (pos == cur) {
						DrawChar(x + n * width + 2, y + i * height + 2, ' ', A_REVERSE);
					}
					pos++;
					n = nbC;
				}
				else if (pos != cur) {
					DrawChar(x + n * width + 2, y + i * height + 2, str[pos++], A_NORMAL);
				}
				else {
					DrawChar(x + n * width + 2, y + i * height + 2, str[pos++], A_REVERSE);
				}
			}
			
			this->totalLines ++;
		}

		if (pos == 0) {
			DrawChar ( x + n * width + 2 , y + i * height + 2 , ' ' , A_REVERSE ) ;

		}
		else if (pos == len && pos == cur) {

			if (n >= nbC) {
				n = 0;
			}
			else {
				i--;
			}
			DrawChar ( x + n * width + 2 , y + i * height + 2 , ' ' , A_REVERSE ) ;
		}
		
/*
		this->lastIndex = this->firstIndex;

		for (i = 0; i < nbL; i++) {
			for (n = 0; n < nbC; n++) {
				this->lastIndex ++;
			}
		}
*/
		
		this->lastIndex = pos;

		for (z = i; z < nbL; z++) {
			for (w = 0; w < nbC; w++) {
				this->lastIndex ++;
			}
		}
		
		for (i = i; (pos < len); i++) {
			for (n = 0; n < nbC && pos < len; n++) {
				if (str[pos++] == '\n') {
					break;
				}
			}
			this->totalLines ++;
		}
	}

	/**
	 * This method is invoked by the GEM when the user interacts
	 * with the TextArea.
	 */
	public void use () {
	    /*char buffer[this.maxLength + 1];
	    char btemp[this.maxLength + 1];*/
	    char[] buffer;
	    char[] btemp;

	    short width = GEM.getFontWidth(this.font);
		short height = GEM.getFontHeight(this.font) + 1;
		short nbL = this.height / height;
	    short nbC = this.getNbC();

	    short ind = 0;
	    short key;
		short x = 0;
		short y = 0;
		Container ct = this.parent;

		int n; /* Temp var used in a for */
		int iBegin; /* Temp var used for cursor positionning */
		int iEnd; /* Temp var used for cursor positionning */
		int maxPos; /* Temp var used for cursor positionning */
		int iTmp; /* Temp var used for cursor positionning */

		if (this.enabled) {
			
			while (ct) {
				x = x + ct.x;
				y = y + ct.y;
				ct = ct.parent;
			}
			
			x = this.x + x;
			y = this.y + y;
			
			if (this.vScrollVisible && (GEM.gem.cursorX >= (this.width - 12 + x))) {
				if (GEM.gem.cursorY <= 11 + y) {
					if (this.scrollLine > 0) {
						this.setScrollLine(this.scrollLine - 1);
					}
				}
				else if (GEM.gem.cursorY <= (this.height - 12 + y)) {
					if (GEM.gem.cursorY >= (this.height - 23 + y)) {
						if (this.scrollLine < (this.totalLines - nbL)) {
							this.setScrollLine(this.scrollLine + 1);
						}
					}
				}
			}
			else if (this.editable) {
				
				x ++;
				y ++;
				
				buffer = new char[this.maxLength + 1];
				btemp = new char[this.maxLength + 1];
		
				if (this.font != Font.FONT_SYSTEM) {
		    		native {
						FontSetSys(this->font);
					}
				}
		
				native {
					if (strlen(this->caption->value) > this->maxLength) {
						this->setCaption_String(this, this->caption->substring_long_int_long_int(this, 0, this->maxLength));
					}
					strcpy(buffer, this->caption->value);
					ind = strlen(buffer);

					this->printTextNoWarp_char_p_short_int_short_int_short_int(this, buffer, x, y, ind);
					
					if (this->totalLines - nbL > 0) {
						this->scrollLine = this->totalLines - nbL;
					}
					else {
						this->scrollLine = 0;
					}
					
					ScrRectFill(&(SCR_RECT){{x, y, x + this->width - 3, y + this->height - 3}}, ScrRect, A_REVERSE);
					this->printTextNoWarp_char_p_short_int_short_int_short_int(this, buffer, x, y, ind);										
				}
		
			   	while (true) {
			    	key = native ( GKeyIn(NULL, 0) );
		
			    	if (key != KEY_ENTER) {
						if (key == KEY_BACKSPACE) {
							if (ind > 0) {
								if (ind < native ( strlen (buffer) )) {
									native {
										strcpy(&buffer[ind-1], &buffer[ind]);
									}
									ind--;
								}
								else {
									buffer[--ind] = '\0';
								}
								if (ind < this.firstIndex) {
									this.scrollLine --;
								}
							}
						}
						else if (key == System_KEY_DIAMOND + KEY_BACKSPACE) {
							
							maxPos = native.strlen (buffer) ;
							if (ind < maxPos) {
								native {
									strcpy(btemp, &buffer[ind+1]);
									strcpy(&buffer[ind], btemp);
								}
							}
						}
						else if (key == System_KEY_LEFT) {
							if (ind) {
								ind--;
								if (ind < this.firstIndex) {
									this.scrollLine --;
								}
							}
						}
						else if (key == System_KEY_RIGHT) {
							maxPos = native.strlen(buffer) ;
							
							if (ind < maxPos) {
								ind++;
								if (ind >= this.lastIndex) {
									this.scrollLine ++;
								}
							}
						}
						else if (key == System.KEY_UP) {
		
							/* ind -= nbC;*/
							iTmp = ind - nbC;
							
							if (iTmp < 0) {
								iTmp = 0;
							}
							
							//Backward until the begining of the line, count chars
							iBegin = ind;
							for (n = 0; (iBegin > 0) && (buffer[--iBegin] != '\n'); n++) ;
							
							iEnd = iBegin;
							
							if (iEnd > 0) {
								iEnd --;
							}
							
							//Forward until the end of the line, stop at the character after '\n'
							while ((iEnd > 0) && (buffer[iEnd] != '\n')) iEnd --;
							
							if (iEnd > 0) {
								iEnd ++;
							}
							
							if (iBegin < iTmp ) {
								ind = iTmp;	
							}
							else {
								maxPos = native.strlen(buffer);
								
								for (iTmp = 0; (iTmp < maxPos) && (buffer[iTmp] != '\n'); iTmp++) ;
								
								if (iTmp < ind) {
									//Ajust the displacement if the line is spanned
									ind = iEnd + ((iBegin - iEnd) / nbC ) * nbC;
									
									//Forward until the char in the same position than the one on the previous line ; break if '\n' is encountered
									while ((n-- > 0) && (buffer[ind] != '\n')) ind ++;
								}
								else {
									ind = 0;
								}
							}
							
							if (ind < this.firstIndex) {
								this.scrollLine --;
							}
						}
						else if (key == System.KEY_DOWN) {
							//ind += nbC;
		
							iTmp = ind + nbC;
							
							maxPos = native.strlen(buffer);
							if (iTmp > maxPos) {
								iTmp = maxPos;
							}
		
							//Backward until the begining of the line, count chars
							iBegin = ind;
							for (n = 0; (iBegin > 0) && (buffer[--iBegin] != '\n'); n++) ;
							
							if (iBegin > 0) {
								iBegin ++;
							}
							
							iEnd = ind;
							//Forward until the end of the line, stop at the character after '\n'
							while ((iEnd < maxPos) && (buffer[iEnd] != '\n')) iEnd ++;
							
							if (iEnd >= iTmp) {
								ind = iTmp;
							}
							else if ((iEnd + n) >= maxPos) {
								ind = maxPos;
							}
							else {
								if (ind >= (iBegin + (nbC * ((iEnd - iBegin) / nbC)))) {
									iEnd ++;
									
									//Ajust the displacement if the line is spanned
									n %= nbC;
									
									//Forward until the char in the same position than the one on the previous line ; break if '\n' is encountered
									while ((n-- > 0) && (buffer[iEnd] != '\n')) iEnd ++;
								}
								ind = iEnd;
							}
							
							if (ind >= this.lastIndex) {
								this.scrollLine ++;
							}
						}
						else if (key == System.KEY_DIAMOND + System.KEY_LEFT) {
							if (buffer[ind] != '\n') {
								for (n = 0; (ind > 0) && (buffer[--ind] != '\n'); n++) ;
								
								if (buffer[ind] == '\n') {
									ind ++;
								}
								
								ind += (n / nbC) * nbC;
							}
						}
						else if (key == System.KEY_DIAMOND + System.KEY_RIGHT) {
							if (buffer[ind] != '\n') {
								for (n = 0; (ind > 0) && (buffer[--ind] != '\n'); n++) ;
								
								if (buffer[ind] == '\n') {
									ind ++;
								}
								
								ind += (n / nbC) * nbC;
								
								maxPos = ind + nbC - 1;
								
								iTmp = native.strlen(buffer) ;
								
								if (maxPos > iTmp) {
									maxPos = iTmp;
								}
								
								while ((ind < maxPos) && (buffer[ind] != '\n')) ind++;
							}
						}
						else if (key == System.KEY_DIAMOND + System.KEY_UP) {
							this.scrollLine = 0;
							
							ind = 0;
						}
						else if (key == System.KEY_DIAMOND + System.KEY_DOWN) {
							this.scrollLine = this.totalLines - nbL;
							
							ind = native ( strlen(buffer) );
						}
						else if (key == KEY_CLEAR) {
							ind = 0;
							buffer[ind] = '\0';
							
							this.scrollLine = 0;
						}
						else {
							if (ind < this.maxLength && (key > 13 && key < 255 || key == System_KEY_DIAMOND + KEY_ENTER)) {
								native {
									strcpy(btemp, &buffer[ind]);
									strcpy(&buffer[ind+1], btemp);
								}
								if (key != System_KEY_DIAMOND + KEY_ENTER) {
									buffer[ind++] = key;
									if (ind >= this.lastIndex) {
										this.scrollLine ++;
									}
								}
								else {
									if (ind >= this.lastIndex - nbC) {
										this.scrollLine ++;
									}
									buffer[ind++] = '\n';
								}
								
								
							}
						}
						native {
							//if (ind < max) {
								ScrRectFill(&(SCR_RECT){{x, y, x + this->width - 3, y + this->height - 3}}, ScrRect, A_REVERSE);
								this->printTextNoWarp_char_p_short_int_short_int_short_int(this, buffer, x, y, ind);
								//}
						}
			    	}
			    	else {
			    		break;
			    	}
			    }
		
				if (this.font != Font.FONT_SYSTEM) {
					native {
						FontSetSys(GEM_gem->systemFont);
					}
				}
				
				this.setCaption(new String(Character.copyValueOf((char*)buffer), false));
				
				System.free(buffer);
	    		System.free(btemp);
			}
		}
		
		super.use();
	}
}