/*
 * QUSOFT MICROSYSTMES
 * Moka
 * Copyright 2002 Frdric Brown
 */
 
/*
 *  This file is part of Moka API.
 *
 *  Moka API is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Moka API is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Moka API; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package moka.x;

import moka.x.Container;
import moka.util.Vector;

/**
 * A ScollPanePanel is a scollable area.
 *
 * @author  Frdric Brown
 * @version 1.0, 2003-12-07
 * @since   MDK2.11
 */
public class ScrollPane extends Container {
	/**The virtual x position in the scrollable area. Should be only read.*/
	public short sX;
	/**The virtual y position in the scrollable area. Should be only read.*/
	public short sY;
	/** A flag specifying if the Scroll Pane will scroll automatically to the next component instead of scrolling progressively. Set to false by default.*/
	public boolean autoScroll;
	
	/** Is the vertical scroll bar should be visible. Should be only read.*/
	public boolean vScrollVisible;
	/** Is the horizontal scroll bar should be visible. Should be only read.*/
	public boolean hScrollVisible;
	
	/** A vector containing the items visible in the scrollable area. Should be only read.*/
	public Vector items;
	
	/**
	 * Constructs a newly allocated ScrollPane.
	 */
	public ScrollPane () {
		super();
		this.sX = 0;
		this.sY = 0;
		this.vScrollVisible = true;
		this.hScrollVisible = true;
		this.autoScroll = false;
		this.items = new Vector();
		this.width = 50;
		this.height = 50;
	}

	/**
	 * Frees the memory of system ressources used by this ScrollPane
	 * object.
	 */
	public void finalize() {
		this.items.finalize();

		super.finalize();
	}

	/**
	 * Sets if the vertical scroll bar should be visible.
	 *
	 * @param b true to specify that the vertical scroll bar should be visible
	 */
	public void setVScrollVisible(boolean b) {
		this.vScrollVisible = b;
		
		this.repaint = true;
		if (this.visible && this.parent) {
			this.parent.onRefresh();
		}
	}
	
	/**
	 * Sets if the horizontal scroll bar should be visible.
	 *
	 * @param b the horizontal scroll bar should be visible
	 */
	public void setHScrollVisible(boolean b) {
		this.hScrollVisible = b;
		
		this.repaint = true;
		if (this.visible && this.parent) {
			this.parent.onRefresh();
		}
	}

	/**
	 * Sets the virtual X coordinate.
	 *
	 * @param sX the virtual X coordinate
	 */
	public void setSX (short sX) {
		this.sX = sX;
		
		this.repaint = true;
		if (this.visible && this.parent) {
			this.parent.onRefresh();
		}
	}

	/**
	 * Sets the virtual Y coordinate.
	 *
	 * @param sY the virtual Y coordinate
	 */
	public void setSY (short sY) {
		this.sY = sY;
		
		this.repaint = true;
		if (this.visible && this.parent) {
			this.parent.onRefresh();
		}
	}

	/**
	 * Paints the foreground of the panel.
	 */
	public native void paintForeground () {		
		short bh = this->height - 36;
		short bw = this->width - 36;
		short vh = this->getVHeight_(this);
		short vw = this->getVWidth_(this);
		short sah = this->height - 12;
		short saw = this->width - 12;

		if (!this->hScrollVisible) {
			sah += 0;
		}
		
		if (!this->vScrollVisible) {
			saw += 0;
		}
		
		short sh = ((this->sY + sah) * bh) / vh + 12 - sah * bh / vh;
		short sw = ((this->sX + saw) * bw) / vw + 12 - saw * bw / vw;
		
		DrawClipRect (&(WIN_RECT){0, 0,  this->width - 1, this->height - 1}, GEM_vScrRect, A_NORMAL);
		
		if (this->vScrollVisible) {
			/* Erase content under bar */
			ScrRectFill (&(SCR_RECT){{this->width-11, 1, this->width-2, this->height-13}}, GEM_vScrRect, A_REVERSE);
			
			DrawLine(this->width-12, 1, this->width-12, this->height-12, A_NORMAL);
			
			Sprite8(this->width - 10, 2, 8, (unsigned char*)ICON_UPARROW, GEM_gem->port, SPRT_OR);
			DrawLine(this->width-11, 11, this->width-2, 11, A_NORMAL);
			
			DrawLine(this->width-11, this->height-23, this->width-2, this->height-23, A_NORMAL);
			Sprite8(this->width - 10, this->height-21, 8, (unsigned char*)ICON_DNARROW, GEM_gem->port, SPRT_OR);
			
			/* Show scroll control */
			//DrawLine(this->width-11, sh, this->width-2, sh, A_NORMAL);
			ScrRectFill (&(SCR_RECT){{this->width-10, sh + 1, this->width-3, sh + (sah * bh) / vh - 1}}, GEM_vScrRect, A_NORMAL);
		}
		
		if (this->hScrollVisible) {
			/* Erase content under bar */
			ScrRectFill (&(SCR_RECT){{1, this->height-11, this->width-13, this->height-2}}, GEM_vScrRect, A_REVERSE);
			
			DrawLine(1, this->height-12, this->width-12, this->height-12, A_NORMAL);
			
			Sprite8(2, this->height - 10, 8, (unsigned char*)ICON_LARROW, GEM_gem->port, SPRT_OR);
			DrawLine(11, this->height - 11, 11, this->height - 2, A_NORMAL);
			
			DrawLine(this->width-23, this->height - 11, this->width-23, this->height-2, A_NORMAL);
			Sprite8(this->width - 21, this->height - 10, 8, (unsigned char*)ICON_RARROW, GEM_gem->port, SPRT_OR);
			
			/* Show scroll control */
			//DrawLine(sw, this->height-12, sw, this->height-2, A_NORMAL);
			ScrRectFill (&(SCR_RECT){{sw + 1, this->height-10, sw + (saw * bw) / vw - 1, this->height-3}}, GEM_vScrRect, A_NORMAL);
		}
		
		if (this->hScrollVisible || this->vScrollVisible) {
			/* Draw lower right corner */
			ScrRectFill (&(SCR_RECT){{this->width-12, this->height-12, this->width-2, this->height-2}}, GEM_vScrRect, A_NORMAL);
		}
	}
	
	/**
	 * Paints the children of this container to the virtual screen.
	 *
	 * param clip The SCR_RECT clip
	 */
	public void paintChildren (SCR_RECT* clip) {
		Component c;
		
		for (short i = 0; i < this.items.size; i++) {
			c = (Component) this.items.array[i];
			if (c.visible) {
				native {
					BitmapPut (c->x - this->sX, c->y - this->sY, c->img, clip, A_REPLACE);
				}
			}
		}
	}
	
	/**
	 * Makes the list of visible items.
	 */
	private void makeList() {
		Component c = this.child;
		
		this.items.clear();
		
		while (c) {
			if (c.x >= this.sX && c.y >= this.sY && c.x < (this.width - 9 + this.sX) && c.y < (this.height - 9 + this.sY)) {
				this.items.push(c);
			}
			
			c = c.next;
		}
	}
	
	/**
	 * Called when a PaintEvent occurs to paint the image of the container and
	 * stores it in its virtual memory.  The container is painted, then all its
	 * children are and finally its foreground is.  A PaintEvent occurs when the
	 * aspect of the container is altered or any of its children is.
	 */
	public void onPaint () {
		this.makeList();
		super.onPaint();
	}
	
	/**
	 * Returns the virtual width.
	 *
	 * @return the virtual width
	 */
	private short getVWidth () {
		short w = 0;
		Component c = this.child;
		
		while (c) {
			if ((c.x + c.width + 1) > w) {
				w = c.x + c.width + 1;
			}
			c = c.next;
		}
		
		return w;
	}

	/**
	 * Returns the virtual height.
	 *
	 * @return the virtual height
	 */
	private short getVHeight () {
		short h = 0;
		Component c = this.child;
		
		while (c) {
			if ((c.y + c.height + 1) > h) {
				h = c.y + c.height + 1;
			}
			c = c.next;
		}
		
		return h;
	}
	
	/**
	 * This method is invoked by the GEM when the user interacts
	 * with the container.
	 */
	public void use () {
		Component c;
		short x = 0;
		short y = 0;
		short vWidth = this.getVWidth();
		short vHeight = this.getVHeight();
		short xSave;
		short ySave;
		short nearX;
		short nearY;
		Container ct = this;
		boolean scrolled = false;

		if (!this.enabled) {
			return;
		}

		while (ct) {
			x = x + ct.x;
			y = y + ct.y;
			ct = ct.parent;
		}
		
		if (this.vScrollVisible && GEM.gem.cursorX >= (this.width - 12 + x)) {
			if (GEM.gem.cursorY <= 11 + y) {
				if (this.autoScroll) {
					nearY = -1;
					nearX = -1;
					c = this.child;
					while (c) {
						if ((c.y < this.sY) && (c.y > nearY)) {
							nearY = c.y;
							nearX = c.x;
						}
						
						c = c.next;
					}
					
					if (nearX >= 0) {
						this.sY = nearY;
						this.sX = nearX;
					}
				}
				else {
					this.sY -= GEM.gem.cursorSpeedY;
				}
			}
			else if (GEM.gem.cursorY <= (this.height - 12 + y)) {
				if (GEM.gem.cursorY >= (this.height - 23 + y)) {
					if (this.autoScroll) {
						nearY = native.MAXINT;
						nearX = -1;
						c = this.child;
						while (c) {
							if ((c.y > this.sY) && (c.y < nearY)) {
								nearY = c.y;
								nearX = c.x;
							}
							
							c = c.next;
						}
						
						if (nearX >= 0) {
							this.sY = nearY;
							this.sX = nearX;
						}
					}
					else {
						this.sY += GEM.gem.cursorSpeedY;	
					}
				}
				else {
					this.sY = ((GEM.gem.cursorY - y - 12) / (float)(this.height - 35)) * (vHeight - 1);
				}
			}

			scrolled = true;
		}
		else if (this.hScrollVisible && GEM.gem.cursorY >= (this.height - 12 + y)) {
			if (GEM.gem.cursorX <= 11 + x) {
				if (this.autoScroll) {
					nearY = -1;
					nearX = -1;
					c = this.child;
					while (c) {
						if ((c.x < this.sX) && (c.x > nearX)) {
							nearY = c.y;
							nearX = c.x;
						}
						
						c = c.next;
					}
					
					if (nearY >= 0) {
						this.sY = nearY;
						this.sX = nearX;
					}
				}
				else {
					this.sX -= GEM.gem.cursorSpeedX;
				}

			}
			else if (GEM.gem.cursorX <= (this.width - 12 + x)) {
				if (GEM.gem.cursorX >= (this.width - 23 + x)) {
					if (this.autoScroll) {
						nearY = -1;
						nearX = native.MAXINT;
						c = this.child;
						while (c) {
							if ((c.x > this.sX) && (c.x < nearX)) {
								nearY = c.y;
								nearX = c.x;
							}
							
							c = c.next;
						}
						
						if (nearY >= 0) {
							this.sY = nearY;
							this.sX = nearX;
						}
					}
					else {
						this.sX += GEM.gem.cursorSpeedX;
					}
				}
				else {
					this.sX = ((GEM.gem.cursorX - x - 12) / (float)(this.width - 35)) * (vWidth - 1);
				}
				
			}	
			
			scrolled = true;
		}
		
		if (scrolled) {
			if (this.sY < 0) {
				this.sY = 0;
			}
			
			if (this.sX < 0) {
				this.sX = 0;
			}
			
			if (this.sY > vHeight - this.height + 13) {
				this.sY = vHeight - this.height + 13;
			}
			
			if (this.sX > vWidth - this.width + 13) {
				this.sX = vWidth - this.width + 13;
			}

			this.repaint = true;
			if (this.visible && this.parent) {
				this.parent.onRefresh();
			}
			
			if (this.listen) {
				this.listen.eventTriggered(this);
			}

			return;
		}
		
		x -= this.sX;
		y -= this.sY;
		
		for (short i = this.items.size - 1; i >= 0; i--) {
			c = (Component) this.items.array[i];
			if (c.visible && c.isCursorOver(x, y)) {
				
				c.x -= xSave = this.sX;
				c.y -= ySave = this.sY;
				
				
				c.use();

				c.x += xSave;
				c.y += ySave;
				
				this.repaint = true;
				if (this.visible && this.parent) {
					this.parent.onRefresh();
				}
				
				return;	
			}
		}
		
		if (this.listen) {
			this.listen.eventTriggered(this);
		}
	}
}