/*
 * QUSOFT MICROSYSTMES
 * Moka
 * Copyright 2002 Frdric Brown
 */
 
/*
 *  This file is part of Moka API.
 *
 *  Moka API is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Moka API is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Moka API; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package moka.x;

import moka.x.GEM;
import moka.x.Frame;
import moka.x.Button;
import moka.x.Label;
import moka.x.TextField;

/**
 * <code>OptionPane</code> makes it easy to pop up a standard dialog box that
 * prompts users for a value or informs them of something.
 *
 * @author  Frdric Brown
 * @version 1.0, 2002-11-13
 * @since   MDK1.0a
 */
public class OptionPane extends Frame {
	/** The dialog used when calling one of the "show" methods.*/
 	private static OptionPane dialog = null;
 	
    /** Type used for <code>showConfirmDialog</code>. */
    public static final short YES_NO_OPTION = 0;
    /** Type used for <code>showConfirmDialog</code>. */
    public static final short YES_NO_CANCEL_OPTION = 1;
    /** Type used for <code>showConfirmDialog</code>. */
    public static final short OK_CANCEL_OPTION = 2;
 	
    /** Default return value. */
    public static final short DEFAULT_OPTION = -1;
    /** Return value from class method if YES is chosen. */
    public static final short YES_OPTION = 0;
    /** Return value from class method if NO is chosen. */
    public static final short NO_OPTION = 1;
    /** Return value from class method if CANCEL is chosen. */
    public static final short CANCEL_OPTION = 2;
    /** Return value form class method if OK is chosen. */
    public static final short OK_OPTION = 0;
 	
 	/** The type of the option pane.*/
 	//private short type; 
 	
 	/** The selected option of this option pane.*/
 	private short option;
 	
 	/** The input value of this option pane.*/
 	private String value;
 	
 	/** The "ok" button.*/
 	private Button btnOk;

 	/** The "yes" button.*/
 	private Button btnYes;
 	
 	/** The "no" button.*/
 	private Button btnNo;
 	
 	/** The "cancel" button.*/
 	private Button btnCancel;
 	
 	/** The textfield used in input dialogs.*/
 	private TextField text;
 	
 	/** The label used in message and confirm dialogs.*/
 	private Label message;
 	
 	/**
 	 * Constructs a newly allocated OptionPane dialog.
 	 */
 	public OptionPane () {
 		super();
 		this.minimizeButtonVisible = false;
 		this.modal = true;
 		this.defaultCloseOperation = Frame.HIDE_ON_CLOSE;
 		
 		this.setBounds(0s, 0s, 150s, 75s);
 		this.setFont(Font.FONT_6x8);
 		//GEM.gem.add(this);
 		
 		this.btnOk = new Button("Ok");
 		this.btnOk.setBounds(2s, 57s, 25s, 16s);
 		this.btnOk.mnemonic = 'O';
 		this.btnOk.listen = this;
 		this.add(this.btnOk);
 		this.btnYes = new Button("Yes");
 		this.btnYes.setBounds(2s, 57s, 30s, 16s);
 		this.btnYes.mnemonic = 'Y';
 		this.btnYes.listen = this;
 		this.add(this.btnYes);
 		this.btnNo = new Button("No");
 		this.btnNo.setBounds(34s, 57s, 25s, 16s);
 		this.btnNo.mnemonic = 'N';
 		this.btnNo.listen = this;
 		this.add(this.btnNo);
 		this.btnCancel = new Button("Cancel");
 		this.btnCancel.setBounds(61s, 57s, 52s, 16s);
 		this.btnCancel.mnemonic = 'C';
 		this.btnCancel.listen = this;
 		this.add(this.btnCancel);
 		
 		this.text = new TextField();
 		this.text.setBounds(2s, 32s, 146s, 16s);
 		this.text.setFont(Font.FONT_6x8);
 		this.add(this.text);
 		this.message = new Label();
 		this.message.setFont(Font.FONT_6x8);
 		this.message.setBounds(2s, 14s, 146s, 14s);
 		this.add(this.message);
 		
 		this.option = OptionPane.DEFAULT_OPTION;
 		this.value = null;
 	}
 	
 	/**
 	 * Called at the end of the program to free the
 	 * ressources used by the static members of this class.
 	 */
 	finally {
 		if (OptionPane.dialog) {
 			OptionPane.dialog.dispose();
 		}
 	}
 	
 	/**
 	 * Reinitializes the components of the OptionPane.
 	 */
 	private static void prepare () {
 		if (! OptionPane.dialog) {
 			OptionPane.dialog = new OptionPane();
 		}
 		
 		OptionPane.dialog.repaint = true;
 		
 		OptionPane.dialog.btnOk.visible = false;
 		OptionPane.dialog.btnYes.visible = false;
 		OptionPane.dialog.btnNo.visible = false;
 		OptionPane.dialog.btnCancel.visible = false;
 		OptionPane.dialog.text.visible = false;
	}

    /**
     * Brings up an information-message dialog titled "Message". The
     * dialog will be centered on the screen.
     *
     * @param message The information-message to be displayed.
     */
 	public static void showMessageDialog (String message) {
 		OptionPane.showMessageDialog ((Container)null, "Message", message);
 	}
 	
    /**
     * Brings up an information-message dialog. The
     * dialog will be centered on its parent.
     *
     * @param parent The parent of the dialog.
     * @param title The title of the dialog.
     * @param message The information-message to be displayed.
     */ 
 	public static void showMessageDialog (Container parent, String title, String message) {
		OptionPane.prepare();
		OptionPane.dialog.setCaption(title);
		OptionPane.dialog.message.setCaption(message);
		OptionPane.dialog.btnOk.visible = true;
		
		/*if (OptionPane.dialog.parent) {
			OptionPane.dialog.parent.remove(OptionPane.dialog);
		}*/
		
		if (!parent) {
			GEM.gem.add(OptionPane.dialog);
		}
		else {
			parent.add(OptionPane.dialog);
		}
		
		OptionPane.dialog.center();
		
		OptionPane.dialog.setVisible(true);
		
		OptionPane.dialog.parent.remove(OptionPane.dialog);
 	}
 	
    /**
     * Shows a question-message dialog requesting input from the user. The 
     * dialog will be centered on the screen.
     *
     * @param message The question-message to be displayed.
     */
 	public static String showInputDialog (String message) {
 		return OptionPane.showInputDialog ((Container)null, "Input", message);
 	}

    /**
     * Shows a question-message dialog requesting input from the user. The 
     * dialog will be centered on its parent.
     *
     * @param parent The parent of the dialog.
     * @param title The title of the dialog.
     * @param message The question-message to be displayed.
     */ 	
 	public static String showInputDialog (Container parent, String title, String message) {
		OptionPane.prepare();
		OptionPane.dialog.setCaption(title);
		OptionPane.dialog.message.setCaption(message);
		OptionPane.dialog.text.setCaption("");
		OptionPane.dialog.btnOk.visible = true;
		OptionPane.dialog.btnCancel.visible = true;
		OptionPane.dialog.text.visible = true;
		
		/*if (OptionPane.dialog.parent) {
			OptionPane.dialog.parent.remove(OptionPane.dialog);
		}*/
		
		if (!parent) {
			GEM.gem.add(OptionPane.dialog);
		}
		else {
			parent.add(OptionPane.dialog);
		}
		
		OptionPane.dialog.center();
		
 		OptionPane.dialog.value = null;
		
		OptionPane.dialog.setVisible(true);
		
		OptionPane.dialog.parent.remove(OptionPane.dialog);
		
		return OptionPane.dialog.value;
 	}
 	
    /**
     * Brings up a dialog with the options <i>Yes</i>,
     * <i>No</i> and <i>Cancel</i>; with the
     * title, <b>Select an Option</b>. The
     * dialog will be centered on the screen.
     *
     * @param message The message to be displayed.
     */
 	public static short showConfirmDialog (String message) {
 		return OptionPane.showConfirmDialog ((Container)null, "Confirm", message, OptionPane.YES_NO_CANCEL_OPTION);
 	}
 	
    /**
     * Brings up a dialog with the specified options (a combination of
     *  <i>Yes</i>, <i>No</i> and <i>Cancel</i>); with the
     * specified title. The 
     * dialog will be centered on its parent.
     *
     * @param parent The parent of the dialog.
     * @param title The title of the dialog.
     * @param message The message to be displayed.
     * @param option The options to be displayed.
     */
 	public static short showConfirmDialog (Container parent, String title, String message, short option) {
		OptionPane.prepare();
		OptionPane.dialog.setCaption(title);
		OptionPane.dialog.message.setCaption(message);
		
		switch (option) {
			case OptionPane.OK_CANCEL_OPTION: {
				OptionPane.dialog.btnOk.visible = true;
			}
			case OptionPane.YES_NO_CANCEL_OPTION: {
				OptionPane.dialog.btnCancel.visible = true;
				if (option == OptionPane.OK_CANCEL_OPTION) {
					break;
				}
			}
			case OptionPane.YES_NO_OPTION: {
				OptionPane.dialog.btnYes.visible = true;
				OptionPane.dialog.btnNo.visible = true;
			}
		}
		
		/*if (OptionPane.dialog.parent) {
			OptionPane.dialog.parent.remove(OptionPane.dialog);
		}*/
		
		if (!parent) {
			GEM.gem.add(OptionPane.dialog);
		}
		else {
			parent.add(OptionPane.dialog);
		}
		
		OptionPane.dialog.center();
		
		if (option == OptionPane.YES_NO_OPTION) {
			OptionPane.dialog.option = OptionPane.NO_OPTION;
		}
		else {
			OptionPane.dialog.option = OptionPane.CANCEL_OPTION;
		}
		
		OptionPane.dialog.setVisible(true);
		
		OptionPane.dialog.parent.remove(OptionPane.dialog);
		
		return OptionPane.dialog.option;
 	}
 	
    /**
     * Invoked when an event is triggered.
     *
     * @param sender The object who triggered the event.
     */
 	public void eventTriggered (Object sender) {
 		if (sender == this.btnOk) {
 			if (this.text.visible) {
 				this.value = this.text.caption.toString();
 			}
 			else {
 				this.option = OptionPane.OK_OPTION;
 			}
 		}
 		else if (sender == this.btnYes) {
 			this.option = OptionPane.YES_OPTION;
 		}
 		else if (sender == this.btnNo) {
 			this.option = OptionPane.NO_OPTION;
 		}
 		else if (sender == this.btnCancel) {
 			if (this.text.visible) {
 				this.value = null;
 			}
 			else {
 				this.option = OptionPane.CANCEL_OPTION;
 			}
 		}
 		else {
 			return;
 		}
 		
 		this.setVisible(false);
 	}
}