/*
 * QUSOFT MICROSYSTMES
 * Moka
 * Copyright 2003 Frdric Brown
 */

/*
 *  This file is part of Moka API.
 *
 *  Moka API is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Moka API is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Moka API; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package moka.x;

import moka.x.Label;
import moka.x.MenuItem;
import moka.util.Vector;
import moka.lang.Character;

/**
 *  Menu class represents a pull-down menu component that is deployed from a menu bar.
 *
 * Each item in a menu must belong to the MenuItem class.
 *
 * @author  Frdric Brown
 * @version 1.0, 2003-10-12
 * @since   MDK2.1
 */
public class Menu extends Label {
	/** A vector containing the items of this Menu.*/
	public Vector items;

	/*
	 * Constructs a newly allocated Menu.
	 */
	public Menu () {
		this("Menu");
	}

	/**
	 * Frees the memory of system ressources used by this Menu
	 * object.
	 */
	public void finalize() {
		Object o;//An Object reference

		while (this.items.size) {//While the list is not empty
			o = this.items.array[0];//Assing the item 0 of the list to o
			this.items.remove(0);//Remove the first item
			o.finalize();//Finalize o
		}

		this.items.finalize();

		super.finalize();
	}

	/*
	 * Constructs a new Menu specifying its caption.
	 *
	 * @param caption the Menu's caption.
	 */
	public Menu (String caption) {
		super(caption);

		this.items = new Vector();
	}

	/**
	 * Paints the Menu.
	 */
	public native void paint() {
		Label->paint_(this);

		DrawLine(this->width - 1, 0, this->width - 1, this->height - 2, A_SHADE_PS);
	}

	/**
	 * Adds the specified MenuItem to this Menu.
	 * Same as this Menu items.add(i);
	 *
	 * @param i the MenuItem to add to this Menu.
	 */
	public void add (MenuItem i) {
		this.items.push(i);
	}

	/**
	 * Removes the specified Menu from this MenuBar.
	 * Same as this Menu items.remove(i);
	 *
	 * @param m the Menu to remove from this MenuBar
	 */
	public void remove (MenuItem i) {
		this.items.remove(i);
	}

	/**
	 * Called when a PaintEvent occurs to paint the image of the container and
	 * stores it in its virtual memory.  The container is painted, then all its
	 * children are and finally its foreground is.  A PaintEvent occurs when the
	 * aspect of the container is altered or any of its children is.
	 */
	public void onPaint () {
		if (this.parent instanceof MenuBar) {
			((MenuBar)this.parent).update();
		}

		super.onPaint();
	}

	/**
	 * Sets the visibility of the component.
	 *
	 * @param b true if the component should be visible, false otherwize.
	 */
	public void setVisible (boolean b) {
		super.setVisible(b);

		if (this.parent instanceof MenuBar) {
			((MenuBar)this.parent).update();
		}
	}

	/**
	 * This method is invoked by the GEM when the user interacts
	 * with the ComboBox.
	 */
	public native void use () {
		short i;
		TContainer* ct = this->parent;
		short y = this->y;
		short x = this->x;
		short height = GEM_getFontHeight_short_int(this->font) + 4;
		short x0;
		short y0;
		short x1;
		short y1;
		short max;
		short key;
		short h = GEM_getFontHeight_short_int(this->font);
		short w = GEM_getFontWidth_short_int(this->font);
		TMenuItem* item;
		void* begin;
		void* end;
		long int pos;
		long int len;
		long int maxLen = 0;
		short boxWidth;

		if (!this->enabled) {
			return;
		}

		if (this->items->size < 1) {
			CaptionedComponent->use_((TCaptionedComponent*)this);
			return;
		}


		while (ct) {
			y = y + ct->y;
			x = x + ct->x;
			ct = ct->parent;
		}

		for (i = 0; i < this->items->size; i++) {
			len = strlen(((TMenuItem*)this->items->array[i])->caption->value);

			if (len > maxLen) maxLen = len;
		}

		if (this->items->size * height + y + this->height <= GEM_USABLE_HEIGHT) {
			x0 = x;
			y0 = y + this->height;
			x1 = x + this->width - 1;
			y1 = y0 + this->items->size * height + 2;

		}
		else if (this->items->size * height - y - 2 < 0) {
			x0 = x;
			y0 = y - 2 - this->items->size * height;
			x1 = x + this->width - 1;
			y1 = y-2;
		}
		else {
			x0 = x;
			y0 = 0;
			x1 = x + this->width - 1;
			y1 = this->items->size * height + 1;
		}

		boxWidth = (maxLen * w + 4);

		if (boxWidth < GEM_LCD_WIDTH) {
			x1 = x0 + boxWidth;
		}
		else {
			x1 = GEM_LCD_WIDTH - 1;
		}

		if ((x0 + boxWidth) >= GEM_LCD_WIDTH) {
			if (x + this->width - 1 - boxWidth >= 0 && (x + this->width - 1 < GEM_LCD_WIDTH)) {
				x0 = x + this->width - 1 - boxWidth;
				x1 = x + this->width - 1;

			}
			else {
				x0 = 0;
				x1 = boxWidth;
			}

		}

		ScrRectFill(&(SCR_RECT){{x, y-1, x+this->width-3, y+this->height-1}}, ScrRect, A_XOR);
		ScrRectFill(&(SCR_RECT){{x0 + 1, y0 + 1, x1 - 1, y1 - 1}}, ScrRect, A_REVERSE);

		if (this->font != Font_FONT_SYSTEM) {
			FontSetSys(this->font);
		}

		max = (y1 - y0) / height;

		for (i = 0; i < this->items->size && i < max; i++) {

			item = (TMenuItem*)this->items->array[i];

			DrawStr(x0 + 2, y0 + 2 + i * height, item->caption->value, A_NORMAL);

			begin = item->caption->value;
			len = strlen(item->caption->value);
			end = memchr(begin, item->mnemonic, len);


			if (!end) {
				end = memchr(begin, Character_toLowerCase_char(item->mnemonic), len);
			}

			if (end) {
				pos = (long int)((char*)end - (char*)begin);
				DrawLine(x0 + 2 + w * pos, y0 + 3 + i * height + h , x0 + 2 + w * (pos + 1), y0 + 3 + i * height + h, A_NORMAL);
			}
		}

		if (this->font != Font_FONT_SYSTEM) {
			FontSetSys(GEM_gem->systemFont);
		}

		//DrawClipRect (&(WIN_RECT){x0, y0,  x1, y1}, ScrRect, A_NORMAL);

		DrawLine ( x0 , y0 , x1 , y0 , A_NORMAL ) ;
		DrawLine ( x0 , y0 , x0 , y1 , A_NORMAL ) ;
		DrawLine ( x0 , y1 , x1 , y1 , A_NORMAL ) ;
		DrawLine ( x1 , y0 , x1 , y1 , A_NORMAL ) ;

		key = GEM_gem->useCursor_(GEM_gem);

		if (key == KEY_ENTER) {
			if (GEM_gem->cursorY >= y0 && GEM_gem->cursorY <= y1 && GEM_gem->cursorX >= x0 && GEM_gem->cursorX <= x1) {
				i = (GEM_gem->cursorY - y0) / (GEM_getFontHeight_short_int(this->font) + 4);
				if (i < this->items->size && ((TMenuItem*)this->items->array[i])->listen) {
						((TMenuItem*)this->items->array[i])->listen->eventTriggered_Object(((TMenuItem*)this->items->array[i])->listen, this->items->array[i]);
				}
			}
		}
		else {
			for (i = 0; i < this->items->size; i++) {
				if (((TMenuItem*)this->items->array[i])->mnemonic == key && ((TMenuItem*)this->items->array[i])->listen) {
						((TMenuItem*)this->items->array[i])->listen->eventTriggered_Object(((TMenuItem*)this->items->array[i])->listen, this->items->array[i]);
				}
			}
		}

		CaptionedComponent->use_((TCaptionedComponent*)this);

		GEM_gem->repaint = TRUE;
	}
}