/*
 * QUSOFT MICROSYSTMES
 * Moka
 * Copyright 2002 Frdric Brown
 */

/*
 *  This file is part of Moka API.
 *
 *  Moka API is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Moka API is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Moka API; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package moka.x;

import moka.x.CaptionedComponent;
import moka.util.Vector;
import moka.util.Enumeration;

/**
 * List class is a list selection component.
 *
 * @author  Frdric Brown
 * @version 1.0, 2003-03-07
 * @since   MDK1.0a
 */
public class List extends CaptionedComponent {

	/** A vector containing the items of this list. Should be only read.*/
	public Vector items;

	/** The selected item's index. Should be only read.*/
	public short index;

	/**
	 * Constructs a new List.
	 */
	public List () {
		super();
		this.index = -1;
		this.items = new Vector();
	}

	/**
	 * Paints the List.
	 */
	public native void paint() {
		short i;
		short n;
		short height = GEM_getFontHeight_short_int(this->font) + 1;
		short max = this->height / height;
		TString* str;

		DrawClipRect (&(WIN_RECT){0, 0,  this->width - 1, this->height - 1}, GEM_vScrRect, A_NORMAL);

		if (this->font != Font_FONT_SYSTEM) {
			FontSetSys(this->font);
		}

		for (i = 0; i < this->items->size && i < max; i++) {
			str = this->items->array[i]->toString_(this->items->array[i]);

			DrawStr(2, 2 + i * height, str->value, A_NORMAL);

			if (i == this->index) {
				for (n = 2 + i * height - 1; n < (2 + (i + 1) * height); n++) {
					DrawLine(1, n, this->width - 2, n, A_XOR);
				}
			}

			str->finalize_(str);
		}

		if (this->font != Font_FONT_SYSTEM) {
			FontSetSys(GEM_gem->systemFont);
		}
	}

	/**
	 * Frees the memory of system ressources used by this List
	 * object.
	 */
	public void finalize() {
		this.items.finalize();

		super.finalize();
	}

	/**
	 * This method is invoked by the GEM when the user interacts
	 * with the List.
	 */
	public void use () {
		short i;
		Container ct = this.parent;
		short y = this.y;

		if (!this.enabled) {
			return;
		}

		while (ct) {
			y = y + ct.y;
			ct = ct.parent;
		}

		i = (GEM.gem.cursorY - y) / (GEM.getFontHeight(GEM.gem.systemFont) + 1);

		if (i < this.items.size && i != this.index) {
			this.setSelectedIndex(i);
		}

		super.use();
	}

	/**
	 * Sets the selected index.
	 *
	 * @param index the index of the one cell to select
	 */
	public void setSelectedIndex (short index) {
		this.index = index;

		this.repaint = true;
		if (this.visible && this.parent) {
			this.parent.onRefresh();
		}
	}

	/**
	 * Returns the item identified by the selected index.
	 *
	 * @return The item identified by the selected index
	 */
	public Object getSelectedItem () {
		return this.items.array[this.index];//this.items.get(this.index)
	}

	/**
	 * Updates the display of the list.
	 */
	public void update () {
		this.repaint = true;
		if (this.visible && this.parent) {
			this.parent.onRefresh();
		}
	}

	/**
	 * Use the items in the specified Vector as the data
	 * for this List.
	 *
	 * @param data The Vector that contain the desired objects
	 */
	public void setListData (Vector data) {
		Enumeration enu = data.elements();

		this.items.size = 0;//this.items.clear()

		while (enu.hasMoreElements()) {
			this.items.push(enu.nextElement());
		}

		enu.finalize();

		this.repaint = true;
		if (this.visible && this.parent) {
			this.parent.onRefresh();
		}
	}

	/**
	 * Disposes of all ressources used by this List's
	 * elements.
	 *
	 * @since MDK1.01a
	 */
	public void clear () {
		Object o;//An Object reference

		while (this.items.size) {//While the list is not empty
			o = this.items.array[0];//Assing the item 0 of the list to o
			this.items.remove(0);//Remove the first item
			o.finalize();//Finalize o
		}
	}
}