/*
 * QUSOFT MICROSYSTMES
 * Moka
 * Copyright 2002 Frdric Brown
 */

/*
 *  This file is part of Moka API.
 *
 *  Moka API is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Moka API is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Moka API; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package moka.x;

import moka.x.Component;
import moka.io.Graphics;

/**
 * TextArea class is a text entry component.
 *
 * @author  Frdric Brown
 * @version 1.0, 2003-03-06
 * @since   MDK1.0a
 */
public class Image extends Component {
	/** The bitmap displayed by this Image. Should be only read.*/
	public BITMAP* bmp;

	/*
	 * Constructs a new Image.
	 */
	public Image () {
		super();
		this.bmp = null;
	}

	/*
	 * Constructs a new Image specifying its bitmap.
	 *
	 * @param bmp the Image's BITMAP
	 */
	public Image (BITMAP* bmp) {
		super();
		this.bmp = null;
		this.setBmp(bmp);
	}

	/*
	 * Constructs a new Image specifying its PIC file name.
	 * If the file string object risk to be used later, use a copy of the string object
	 * instead of the original, the string being finalized before the method returns.
	 *
	 * @param file The Image's PIC file name
	 */
	public Image (String file) {
		super();
		this.bmp = null;
		this.setBmp(file);
	}

	/**
	 * Paints the Image.
	 */
	public native void paint() {
		if (this->bmp) {
			BitmapPut (0, 0, this->bmp, GEM_vScrRect, A_REPLACE);
		}
	}

	/**
	 * Frees the memory of system ressources used by this Image
	 * object.
	 */
	public void finalize() {
		if (this.bmp) {
			native {
				free(this->bmp);
			}
		}

		super.finalize();
	}

	/**
	 * Sets the bitmap displayed by this Image.
	 *
	 * @param bmp The Image's BITMAP
	 */
	public void setBmp(BITMAP* bmp) {
		if (this.bmp) {
			native {
				free(this->bmp);
			}
		}

		if (! bmp) {
			return;
		}

		this.bmp = Graphics.cloneBitmap(bmp);

		this.repaint = true;
		if (this.visible && this.parent) {
			this.parent.onRefresh();
		}
	}

	/**
	 * Sets the bitmap displayed by this Image.
	 * If the file string object risk to be used later, use a copy of the string object
	 * instead of the original, the string being finalized before the method returns.
	 *
	 * @param file The Image's PIC file name
	 */
	public void setBmp(String file) {
		if (! file) {
			return;
		}

		if (this.bmp) {
			native {
				free(this->bmp);
			}
		}

		this.bmp = Graphics.bitmapFromVar(file);

		this.repaint = true;
		if (this.visible && this.parent) {
			this.parent.onRefresh();
		}
	}
}