/*
 * QUSOFT MICROSYSTMES
 * Moka
 * Copyright 2002 Frdric Brown
 */

/*
 *  This file is part of Moka API.
 *
 *  Moka API is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Moka API is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Moka API; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package moka.x;

import moka.io.Keyboard;
import moka.x.Component;
import moka.x.Container;
import moka.x.Frame;
import moka.x.Font;
import moka.event.EventListener;
import moka.event.Interrupt;
import moka.util.Timer;
import moka.io.RowRead;

/**
 * The GEM (Graphic Environment Manager) class implements the methods used
 * to manage the Graphic Environment.
 *
 * @author  Frdric Brown
 * @version 1.2, 2004-03-14
 * @since   MDK1.0a
 */
public class GEM extends Container {

	/** The state of the lcd before x was started.*/
	private void * lcd;

	/** Is an interrupt occured ?*/
	//private boolean interupted;

	/** The array of interrupts.*/
	private Interrupt * array;

	/** The number of interrupts.*/
	private int size;

	/** The actual interrupts' index.*/
	private int index;

	/** The capacity of the interrupts' array.*/
	private int capacity;

	/** A keyboard object used when moving the cursor.*/
	//private Keyboard kb;

	/** The difference between the cursor horizontal coordinate and the dragged component horizontal coordinate.*/
	private short difX;

	/** The difference between the cursor vertical coordinate and the dragged component vertical coordinate.*/
	private short difY;

	/** The context of the dragged component.*/
	private SCR_RECT dragRect;

	/** The image behind the cursor.*/
	private void * cursorImg;

	/** The image of the component being dragged.*/
	//private void * dragImg;

	/** The component being dragged.*/
	private Component dragCpn;

	/** The virtual screen buffer.*/
	protected void * port;

	/** The instance of the Graphic Environment Manager. Should be only read.*/
	public static GEM gem;

	/** Height of the screen in pixels (100 on TI-89, 128 on TI-92 Plus or V200).*/
	public static short LCD_HEIGHT = ( System.CALCULATOR ? 128 : 100 );
	/** Number of bytes in the visible part of a screen line (20 on TI-89, 30 on TI-92 Plus or V200).*/
	public static short LCD_LINE_BYTES = ( System.CALCULATOR ? 30 : 20 );
	/** Width of the screen in pixels (160 on TI-89, 240 on TI-92 Plus or V200).*/
	public static short LCD_WIDTH = ( System.CALCULATOR ? 240 : 160 );
	/** The usable height of the screen (94 on TI-89, 122 on TI-92 Plus or V200)*/
	public static short USABLE_HEIGHT = GEM.LCD_HEIGHT - 6 ;
	/** The virtual screen SCR_RECT.*/
	public static SCR_RECT * vScrRect;

	/** Is the Graphic Environment started ? Should be only read.*/
	public boolean started;

	/** The cursor x position. Should be only read.*/
	public short cursorX;
	/** The cursor y position. Should be only read.*/
	public short cursorY;
	/** Is the cursor visible ? Should be only read.*/
	public short cursorVisible;
	/** The horizontal speed of the cursor.*/
	public short cursorSpeedX;
	/** The vertical speed of the cursor.*/
	public short cursorSpeedY;

	/** The ssytem font.  Should be only read.*/
	public short systemFont;

	/** Is the user interface should be updated ?*/
	public boolean updateUI = true;

	/** The event listener notified when the Graphic Environment starts.*/
	public EventListener startListen;

	/** The event listener notified when a frame is minimized.*/
	public EventListener minListen;

	/*
	 * Constructs a newly allocated Graphic Environment Manager.
	 * This constructor is called at the startup of the application
	 * to initialize GEM.gem (The system default Graphical Environment
	 * Manager).
	 */
	public GEM () {
		super();
		this.next = null;
		this.prev = null;
		this.x = 0;
		this.y = 0;
		this.width = GEM.LCD_WIDTH;
		this.height = GEM.USABLE_HEIGHT;
		this.started = false;
		this.cursorX = 25;
		this.cursorY = 25;
		this.cursorVisible = false;
		this.cursorSpeedX = 3;
		this.cursorSpeedY = 3;
		this.port = native ( malloc(LCD_SIZE) );
		this.lcd = native ( malloc(LCD_SIZE) );
		this.cursorImg = native ( malloc(LCD_SIZE) );
		//this.dragImg = null;
		this.dragCpn = null;
		//this.kb = new Keyboard();
		this.systemFont = native ( FontGetSys() );
		//this.desktop = null;
		//this.desktop = false;
		this.startListen = null;
		this.minListen = null;

		this.size = 0;
		this.index = -1;
		this.capacity = 5;
		this.array = native ( malloc( 5 * sizeof(TInterrupt*)) );
	}

    /**
     * Free the memory and system ressources used by this GEM.
     */
	public native void finalize () {
		free(this->array);
		free(this->port);
		free(this->lcd);
		free(this->cursorImg);
		if (this->img) {
			free(this->img);
		}
		//this->kb->finalize_(this->kb);
		this->clean_(this);
	}

	/**
	 * Called when an UpdateEvent occurs.  An update event occurs when a
	 * visible component contained in this container has changed its
	 * appearance.
	 */
	public void onRefresh () {
		this.onPaint();
	}

	/**
	 * Disposes of all ressources used by the Graphic Environment.
	 */
	protected void clean () {
		Component c;
		Component cn = this.child;

		while (cn) {
			c = cn;
			cn = cn.next;
			c.clean();
		}

		native {
			free(this);
		}
	}

	/**
	 * Starts this Graphic Environment Manager as if setStarted(true) was called.
	 */
	/*public void startx () {
		this.setStarted(true);
	}*/

	/**
	 * Starts or stops this GEM. While the GEM is started, the execution flow become even driven instead of
	 * linear i.e. that instructions are executed when events or interrupts occur instead
	 * of being executed sequentially. When the GEM is stopped, the execution flow continues were it was.
	 *
	 * @param b true if the Graphic Environment should be started, false if the
	 *			Graphic Environment should be stopped.
	 */
	public void setStarted (boolean b) {
		this.started = b;
		if (b) {
			native {
				LCD_save(this->lcd);
			}

			this.run();
		}
		else {
			native {
				LCD_restore(this->lcd);
			}
		}
	}

	/**
	 * Called when a PaintEvent occurs to refresh the screen. All
	 * the components are painted.  A PaintEvent occurs when the
	 * aspect of any component is altered.
	 */
	public native void onPaint () {
		BOOL b = this->cursorVisible;
		if (this->started && this->updateUI) {
			if (b) {
				this->setCursorVisible_BOOL(this, FALSE);
			}

			LCD_restore(this->lcd);
			this->repaintChildren_(this);

			this->paintChildren_SCR_RECT_p(this, ScrRect);

			if (b) {
				this->setCursorVisible_BOOL(this, TRUE);
			}

			this->repaint = FALSE;
		}
	}

	/**
	 * Sets the visibility of the cursor.
	 *
	 * @param b true if the cursor should be visible, false otherwise.
	 */
	public native void setCursorVisible (boolean b) {
		short int sprtX;
		short int sprtY;

		this->updateUI = FALSE;

		this->cursorVisible = b;

		if (b) {

			if (this->dragCpn) {
				//BitmapPut (this->cursorX - this->difX, this->cursorY - this->difY, this->dragImg, &this->dragRect, A_REPLACE);
				Sprite8(this->cursorX - this->difX, this->cursorY - this->difY, 8, (unsigned char*)ICON_UPLEFTCORNER, LCD_MEM, SPRT_OR);
				sprtX = this->cursorX - this->difX + this->dragCpn->width - 8;
				sprtY = this->cursorY - this->difY + this->dragCpn->height - 8;
				if (sprtX < GEM_LCD_WIDTH && sprtY < GEM_USABLE_HEIGHT)
					Sprite8(sprtX, sprtY, 8, (unsigned char*)ICON_DNRIGHTCORNER, LCD_MEM, SPRT_OR);

				sprtX = this->cursorX - this->difX + this->dragCpn->width - 8;
				sprtY = this->cursorY - this->difY;
				if (sprtX < GEM_LCD_WIDTH && sprtY < GEM_USABLE_HEIGHT)
					Sprite8(sprtX, sprtY, 8, (unsigned char*)ICON_UPRIGHTCORNER, LCD_MEM, SPRT_OR);

				sprtX = this->cursorX - this->difX;
				sprtY = this->cursorY - this->difY + this->dragCpn->height - 8;
				if (sprtX < GEM_LCD_WIDTH && sprtY < GEM_USABLE_HEIGHT)
					Sprite8(sprtX, sprtY, 8, (unsigned char*)ICON_DNLEFTCORNER, LCD_MEM, SPRT_OR);
			}
			else {
				LCD_save(this->cursorImg);
			}

			Sprite8(this->cursorX, this->cursorY, 8, (unsigned char*)CURSOR_ARROW, LCD_MEM, SPRT_OR);

			Sprite8(this->cursorX, this->cursorY, 8, (unsigned char*)CURSOR_MASK_ARROW, LCD_MEM, SPRT_AND);
		}
		else {
			LCD_restore(this->cursorImg);
		}

		this->updateUI = TRUE;
	}

	/**
	 * Sets the current screen state as background.
	 *
	 * @since MDK2.11
	 */
	public native void setBackground() {
		LCD_save(this->lcd);
		this->repaint = TRUE;
		this->onRefresh_(this);
	}

	/**
	 * Gives the user the control of the cursor.
	 *
	 * @return The last key read from the keyboard.
	 */
	public short useCursor () {
		//Keyboard kb = this.kb;
		short key = 0;
		short x0 = this.cursorX;
		short y0 = this.cursorY;
		short pasX = this.cursorSpeedX;
		short pasY = this.cursorSpeedY;

		//this.interupted = false;

		this.setCursorVisible(true);

		do {
			if (this.repaint) {
				this.onPaint();
			}
			Keyboard.flush();

			Timer.wait(1);
			//key = Keyboard.getKey(1);
			//kb.checkKeys();
			RowRead.enable();

			if (RowRead.key_ANY) {
				key = 1;

				if (RowRead.key_Right) {
					x0 = x0 + pasX;
					key = 0;
				}
				if (RowRead.key_Left) {
					x0 = x0 - pasX;
					key = 0;
				}
				if (RowRead.key_Up) {
					y0 = y0 - pasY;
					key = 0;
				}
				if (RowRead.key_Down) {
					y0 = y0 + pasY;
					key = 0;
				}

				if (x0 > GEM.LCD_WIDTH) {
					x0 = GEM.LCD_WIDTH - 1;
				}
				else if (x0 < 0) {
					x0 = 0;
				}

				if (y0 > GEM.USABLE_HEIGHT) {
					y0 = GEM.USABLE_HEIGHT - 1;
				}
				else if (y0 < 0) {
					y0 = 0;
				}

				this.setCursorVisible(false);
				this.cursorX = x0;
				this.cursorY = y0;
				this.setCursorVisible(true);

				if (RowRead.key_ENTER) {
					key = Keyboard.KEY_ENTER;
					break;
				}

				if (RowRead.key_2nd && (RowRead.key_F4 || RowRead.key_APPS)) {
					break;
				}

				if (key) {
					RowRead.disable();
					key = System.read();
					//key = Keyboard.getKey();
					RowRead.enable();

					if (key >= 'a' && key <= 'z') {
						key -= ('a' - 'A');
					}

					break;
				}
			}
			RowRead.disable();

			this.onInterrupt();
		}
		while (this.started);

		RowRead.disable();

		native {
			GKeyFlush();
		}

		this.setCursorVisible(false);

		return key;
	}

	/**
	 * Called when an InterruptEvent occurs.  This method
	 * checks the current interrupt then increment the interrupts' index.
	 * An InterruptEvent occurs regulary when the GEM is in
	 * waiting state (waiting for an event).
	 */
	public void onInterrupt () {

		if (this.index < 0) {
			return;
		}

		if (this.index >= this.size) {
			this.index = 0;
		}

		if (this.array[this.index].check()) {
			//this.interupted = true;
			this.setCursorVisible(false);
			Interrupt.busy = true;
			this.array[this.index].run();
			Interrupt.busy = false;
			this.setCursorVisible(true);
		}

		this.index++;
	}

	/**
	 * Removes the specified component from the GEM.
	 *
	 * @param c the component to remove from the GEM.
	 */
	/*public void remove (Component c) {
		super.remove(c);
		if (!this.child) {
			this.setStarted(false);
		}
	}*/

	/**
	 * Registers the specified interrupt.
	 *
	 * @param item the interrupt to be registered.
	 */
	public native void register (Interrupt item) {
 		TInterrupt** array;

 		if (this->size < this->capacity) {
 			this->array[this->size] = item;
 		}
 		else {
 			array = malloc(sizeof(TInterrupt*) * (this->capacity + 10));
 			this->capacity += 10;

 			memcpy(array, this->array, (this->size) * sizeof(TInterrupt*));

 			array[this->size] = item;

 			free(this->array);

 			this->array = array;
 		}

		this->size++;
	}

	/**
	 * Unregisters the specified interrupt.
	 *
	 * @param item the interrupt to be unregistered.
	 */
	public native void unregister (Interrupt item) {
    	long int i;

    	for (i = 0; i < this->size; i++) {
    		if (this->array[i] == item) {
    			break;
    		}
    	}

   		memmove(this->array + i, this->array + i + 1, (this->size - i - 1) * sizeof(TInterrupt*));
   		this->size--;
   		if (this->index == this->size) {
   			this->index = 0;
   		}
	}

	/**
	 * This method is looping when the GEM is started.
	 */
	public void run () {
		char key;

		if (this.startListen) {
			this.startListen.eventTriggered(this);
		}

		native {
			xStartFunc ( MOKAMACRO_MAINPARMS ) ;
		}

		while (this.started && this.child) {

			key = this.useCursor();

			RowRead.enable();

			if (RowRead.key_2nd && RowRead.key_APPS) {
				RowRead.disable();
				if (this.child) {
					this.toFront(this.child);
				}
			}
			else if (RowRead.key_2nd && RowRead.key_F4) {
				RowRead.disable();
				if (this.tail && (this.tail instanceof Frame)) {
					((Frame)this.tail).onClose();
				}
			}
			else if (key == native ( KEY_ENTER )) {
				RowRead.disable();
				Interrupt.busy = true;
				this.use();
				Interrupt.busy = false;
			}
			else {
				RowRead.disable();
				Interrupt.busy = true;
				this.checkMnemonic(key);
				Interrupt.busy = false;
			}

		}

		if (!this.child) {
			this.setStarted(false);
		}
	}

	/**
	 * This method is called when a Frame is showed in modal mode.
	 *
	 * @param frame The frame to show in modal mode.
	 */
	public void showModal (Frame frame) {
		char key;
		boolean b = Interrupt.busy;

		Interrupt.busy = true;

		this.toFront(frame);

		while (frame.modal && frame.visible && this.started && this.child) {
			key = this.useCursor();

			RowRead.enable();

			if (RowRead.key_2nd && RowRead.key_F4) {
				RowRead.disable();
				frame.onClose();
			}
			else if (key == native ( KEY_ENTER )) {
				RowRead.disable();
				frame.use();
			}
			else {
				RowRead.disable();
				frame.checkMnemonic(key);
			}
			/*if (!this.child) {
				this.setStarted(false);
			}*/
		}

		Interrupt.busy = b;
	}

	/**
	 * Called when a DragEvent occurs to drag and drop a component.
	 * A DragEvent occurs when a component needs to be moved by the
	 * user.
	 *
	 * @param sender the object who triggered the DragEvent.
	 */
	public void onDrag (Component sender) {
		//Keyboard kb = this.kb;
		short x0 = this.cursorX;
		short y0 = this.cursorY;
		short pasX = this.cursorSpeedX;
		short pasY = this.cursorSpeedY;
		short x = 0;
		short y = 0;
		Container ct = sender.parent;
		boolean b;
		boolean cont;

		while (ct) {
			x = x + ct.x;
			y = y + ct.y;
			ct = ct.parent;
		}

		native {
			this->dragRect = (SCR_RECT){{x, y, sender->parent->x + sender->parent->width - 1, sender->parent->y + sender->parent->height - 1}};
		}

		this.difX = x0 - sender.x - x;
		this.difY = y0 - sender.y - y;

		//this.interupted = false;

		b = this.updateUI;
		this.updateUI = true;

		sender.setVisible(false);

		native.LCD_save(this->cursorImg);

		//this.dragImg = sender.img;
		this.dragCpn = sender;

		this.setCursorVisible(true);

		do {
			Timer.wait(1);
			//kb.checkSysKeys();
			RowRead.enable();
			if (RowRead.key_ANY) {

				if (RowRead.key_Right) {
					x0 = x0 + pasX;
				}
				if (RowRead.key_Left) {
					x0 = x0 - pasX;
				}
				if (RowRead.key_Up) {
					y0 = y0 - pasY;
				}
				if (RowRead.key_Down) {
					y0 = y0 + pasY;
				}

				if (x0 > sender.parent.width + x) {
					x0 = sender.parent.width + x - 1;
				}
				//else if (x0 < sender.parent.x) {
				else if (x0 - this.difX - x < sender.parent.x) {
					x0 = sender.parent.x + this.difX + x;
				}

				if (y0 > sender.parent.height + y) {
					y0 = sender.parent.height + y - 1;
				}
				//else if (y0 < sender.parent.y) {
				else if (y0 - this.difY - y < sender.parent.y) {
					y0 = sender.parent.y + this.difY + y;
				}

				if (x0 != this.cursorX || y0 != this.cursorY) {
					this.setCursorVisible(false);
					this.cursorX = x0;
					this.cursorY = y0;
					this.setCursorVisible(true);
				}
			}

			cont = RowRead.key_ENTER;

			RowRead.disable();
			//this.onInterrupt();
		}
		while (this.started && cont);

		native {
			GKeyFlush();
		}

		this.setCursorVisible(false);

		//this.dragImg = null;
		this.dragCpn = null;

		sender.move((short)x0 - this.difX - x, (short)y0 - this.difY - y);

		this.updateUI = b;

		sender.setVisible(true);
	}

	/**
	 * Sets the system font.
	 *
	 * @param font the new system font.
	 */
	public void setFont (short font) {
		this.systemFont = font;

		native {
			FontSetSys(font);
		}

		this.paint();
	}

	/**
	 * Returns the specified font height.
	 *
	 * @param font the specified font
	 * @return the specified font height
	 */
	public static short getFontHeight (short font) {

		if (font == Font.FONT_SYSTEM) {
			font = GEM.gem.systemFont;
		}

		switch (font) {
			case Font.FONT_4x6: {
				return 6;

			}
			case Font.FONT_6x8: {
				return 8;

			}
			case Font.FONT_8x10: {
				return 10;

			}
		}

		return 0;
	}

	/**
	 * Returns the specified font width.
	 *
	 * @param font the specified font
	 * @return the specified font width
	 */
	public static short getFontWidth (short font) {

		if (font == Font.FONT_SYSTEM) {
			font = GEM.gem.systemFont;
		}

		switch (font) {
			case Font.FONT_4x6: {
				return 4;

			}
			case Font.FONT_6x8: {
				return 6;

			}
			case Font.FONT_8x10: {
				return 8;

			}
		}

		return 0;
	}

	/**
	 * Returns the current system font height.
	 *
	 * @return the current system font height
	 */
	public short getFontHeight () {
		return GEM.getFontHeight(this.systemFont);
	}

	/**
	 * Returns the current system font width.
	 *
	 * @return the current system font width
	 */
	public short getFontWidth () {
		return GEM.getFontWidth(this.systemFont);
	}

	/**
	 * Called when a MinimizedEvent is triggered. A MinimizeEvent
	 * is triggered when a user wants to minimize a frame.
	 */
	public void onMinimize (Frame sender) {
		if (this.minListen) {
			this.minListen.eventTriggered(sender);
		}
	}
}