/*
 * QUSOFT MICROSYSTMES
 * Moka
 * Copyright 2002 Frdric Brown
 */

/*
 *  This file is part of Moka API.
 *
 *  Moka API is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Moka API is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Moka API; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package moka.x;

import moka.x.GEM;
import moka.x.Captioned;
import moka.x.Container;
import moka.x.MenuBar;

/**
 * Class Frame is a floating window.
 *
 * @author  Frdric Brown
 * @version 1.1, 2002-11-06
 * @since   MDK1.0a
 */
 public class Frame extends Container implements Captioned {
	/** Indicate that the default close operation exits the application.*/
	public static final short EXIT_ON_CLOSE = 0;
	/** Indicate that the default close operation disposes of this frame.*/
	public static final short DISPOSE_ON_CLOSE = 1;
	/** Indicate that the default close operation hides the frame.*/
	public static final short HIDE_ON_CLOSE = 2;
	/** Indicate that the default close operation does nothing.*/
	public static final short DO_NOTHING_ON_CLOSE = 3;

	/** Is this frame movable ?*/
	public boolean movable;

	/** Is this frame modal ?  Should be only read.*/
	public boolean modal;

	/** The default close operation for this frame.*/
	public short defaultCloseOperation;

	/** Is the close button visible ?  Should be only read.*/
	public boolean closeButtonVisible;

	/** Is the minimize button visible ?  Should be only read.*/
	public boolean minimizeButtonVisible;

	/*
	 * Constructs a new frame.
	 */
	public Frame () {
		this("");
	}

	/*
	 * Constructs a new frame specifying its caption.
	 *
	 * @param caption the frame's caption.
	 */
	public Frame (String caption) {
		super();
		this.caption = caption;
		this.font = Font.FONT_SYSTEM;
		this.modal = false;
		this.defaultCloseOperation = Frame.DISPOSE_ON_CLOSE;
		this.visible = false;
		this.movable = true;
		this.closeButtonVisible = true;
		this.minimizeButtonVisible = false;
	}

	/**
	 * Adds the specified MenuBar to this Frame.
	 *
	 * @param m the MenuBar to add to this Frame.
	 */
	public void addMenuBar (MenuBar m) {
		m.setBounds(1, 13, GEM.LCD_WIDTH - 2, 15);
		
		this.add((Component)m);
	}

    /**
     * Free the memory and system ressources used by this Frame.
     */
	public void finalize () {
		this.caption.finalize();
		super.finalize();
	}

	/**
	 * This method is invoked by the GEM when the user interacts
	 * with the frame.
	 */
	public void use () {
		short x = 0;
		short y = 0;
		Container ct = this.parent;

		while (ct) {
			x = x + ct.x;
			y = y + ct.y;
			ct = ct.parent;
		}

		if (this.isCursorOver(x, y)) {
			this.parent.toFront(this);
		}

		if (this.closeButtonVisible && this.isCursorOverCloseButton(x, y)) {
			this.onClose();
			return;
		}

		if (this.minimizeButtonVisible && this.isCursorOverMinimizeButton(x, y)) {
			this.onMinimize();
			return;
		}

		if (this.movable && this.isCursorOverTitleBar(x, y)) {
			this.onDrag();
			return;
		}

		//this.parent.toFront(this);

		super.use();
	}

	/**
	 * Indicates if the cursor is over the close button of this frame.
	 *
	 * @param x the reference's horizontal coordinate
	 * @param y	the reference's vertical coordinate
	 * @return true if the cursor is over the close button, false otherwize.
	 */
	public boolean isCursorOverCloseButton (short x, short y) {
		return GEM.gem.cursorX >= this.x + x + this.width - 12 && GEM.gem.cursorY >= this.y + y && GEM.gem.cursorX < this.x + x + this.width && GEM.gem.cursorY < this.y + y + 12;
	}

	/**
	 * Indicates if the cursor is over the minimize button of this frame.
	 *
	 * @param x the reference's horizontal coordinate
	 * @param y	the reference's vertical coordinate
	 * @return true if the cursor is over the minimize button, false otherwize.
	 */
	public boolean isCursorOverMinimizeButton (short x, short y) {
		return GEM.gem.cursorX >= this.x + x + this.width - 24 && GEM.gem.cursorY >= this.y + y && GEM.gem.cursorX < this.x + x + this.width - 12 && GEM.gem.cursorY < this.y + y + 12;
	}

	/**
	 * Indicates if the cursor is over the title bar of this frame.
	 *
	 * @param x the reference's horizontal coordinate
	 * @param y	the reference's vertical coordinate
	 * @return true if the cursor is over the title bar, false otherwize.
	 */
	public boolean isCursorOverTitleBar (short x, short y) {
		return GEM.gem.cursorX >= this.x + x && GEM.gem.cursorY >= this.y + y && GEM.gem.cursorX < this.x + x + this.width && GEM.gem.cursorY < this.y + y + 12;
	}

	/**
	 * Called when a CloseEvent occurs.  The method will perform a task
	 * according to the default close operation of this frame.  A CloseEvent
	 * occurs when the user uses the close button of the frame.
	 */
	public void onClose () {
		switch (this.defaultCloseOperation) {
			case 0: {
				GEM.gem.setStarted(false);
				break;
			}
			case 1: {
				this.dispose();
				break;
			}
			case 2: {
				this.setVisible(false);
				break;
			}
		}
	}

	/**
	 * Called when a MinimizeEvent occurs.  The method will minimize the window
	 * to the task bar of the desktop.  A MinimizeEvent
	 * occurs when the user uses the minimize button of the frame.
	 */
	public void onMinimize () {
		GEM.gem.onMinimize(this);
	}

	/**
	 * Called when a DragEvent occurs.  The method will notify the GEM
	 * with a DragEvent.  A DragEvent
	 * occurs when the user click on the title bar.
	 */
	public void onDrag () {
		GEM.gem.onDrag(this);
	}

	/**
	 * Disposes of all ressources used by this frame and
	 * all its children.
	 */
	public void dispose () {
		if (this.parent) {
			this.parent.remove(this);
		}

		this.clean();
	}

	/**
	 * Sets the visibility of the frame.
	 *
	 * @param b true if the frame should be visible, false otherwize.
	 */
	public void setVisible (boolean b) {
		if (b && !this.parent) {
			GEM.gem.add(this);
		}

		super.setVisible(b);

		if (b && this.modal) {
			GEM.gem.showModal(this);
		}
	}

	/**
	 * Sets frame is modal or not.
	 *
	 * @param b true if the frame should be modal, false otherwize.
	 */
	public void setModal (boolean b) {
		this.modal = b;

		if (b && this.visible) {
			GEM.gem.showModal(this);
		}
	}

	/**
	 * Sets the visibility of the close button.
	 *
	 * @param b true if the close button should be visible, false otherwize.
	 */
	public void setCloseButtonVisible (boolean b) {
		this.closeButtonVisible = b;

		this.repaint = true;
		if (this.visible && this.parent) {
			this.parent.onRefresh();
		}
	}

	/**
	 * Sets the visibility of the minimize button.
	 *
	 * @param b true if the minimize button should be visible, false otherwize.
	 */
	public void setMinimizeButtonVisible (boolean b) {
		this.minimizeButtonVisible = b;

		this.repaint = true;
		if (this.visible && this.parent) {
			this.parent.onRefresh();
		}
	}

	/**
	 * Paints the frame.
	 */
	public native void paint () {
		int x1 = this->width - 1;

		DrawLine(1, 12, (x1-1), 12, A_NORMAL);

		if (this->font != Font_FONT_SYSTEM) {
			FontSetSys(this->font);
		}

		DrawStr(2, 2, this->caption->value, A_NORMAL);

		if (this->font != Font_FONT_SYSTEM) {
			FontSetSys(GEM_gem->systemFont);
		}

		if (this->closeButtonVisible) {
			DrawLine((x1-11), 1, (x1-11), 11, A_NORMAL);
			DrawStr((x1-9), 2, "x", A_NORMAL);
		}

		if (this->minimizeButtonVisible) {
			DrawLine((x1-22), 1, (x1-22), 11, A_NORMAL);
			DrawStr((x1-20), 2, "-", A_NORMAL);
		}
	}

	/**
	 * Paints the foreground of the frame.
	 */
	public native void paintForeground () {
		/*int x1 = this->width - 1;
		int y1 = this->height - 1;

		DrawLine(0, 0, x1, 0, A_NORMAL);
		DrawLine(0, y1, x1, y1, A_NORMAL);
		DrawLine(0, 1, 0, (y1-1), A_NORMAL);
		DrawLine(x1, 1, x1, (y1-1), A_NORMAL);*/

		DrawClipRect (&(WIN_RECT){0, 0,  this->width - 1, this->height - 1}, GEM_vScrRect, A_NORMAL);
	}

	/**
	 * Indicates if the cursor is over this frame.
	 *
	 * @param x the reference's horizontal coordinate
	 * @param y	the reference's vertical coordinate
	 * @return true if the cursor is over the frame, false otherwize.
	 */
	public boolean isCursorOver (short x, short y) {
		if (this.modal) {
			return true;
		}

		return super.isCursorOver(x, y);
	}
}