/*
 * QUSOFT MICROSYSTMES
 * Moka
 * Copyright 2002 Frdric Brown
 */

/*
 *  This file is part of Moka API.
 *
 *  Moka API is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Moka API is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Moka API; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package moka.x;

import moka.x.Component;

/**
 * Container is a component that contains child components.
 *
 * @author  Frdric Brown
 * @version 1.0, 2002-10-31
 * @since   MDK1.0a
 */
public class Container extends Component {

	/** The first component in this Container.*/
	protected Component child;

	/** The last component in this Container.*/
	protected Component tail;

	/**
	 * Constructs a newly allocated Container.
	 */
	public Container () {
		super();
		this.child = null;
		this.tail = null;
	}

	/**
	 * Called when a PaintEvent occurs to paint the image of the container and
	 * stores it in its virtual memory.  The container is painted, then all its
	 * children are and finally its foreground is.  A PaintEvent occurs when the
	 * aspect of the container is altered or any of its children is.
	 */
	public void onPaint () {
		this.repaint = false;
		this.repaintChildren();
		native {
			SCR_RECT rect = (SCR_RECT){{0, 0, this->width - 1, this->height - 1}};

			PortSet(GEM_gem->port, 239, 127);
			ScrRectFill (&rect, &rect, A_REVERSE);
		}

		this.paint();

		this.paintChildren((SCR_RECT*)&rect);

		this.paintForeground();

		native {
			if (this->img) {
				free (this->img);
			}

			this->img = malloc(BitmapSize(&rect));
			BitmapGet (&rect, this->img);
			PortSet ((void *) 0x4C00, 239, 127);
		}
	}

	/**
	 * Called when an UpdateEvent occurs.  An update event occurs when a
	 * visible component contained in this container has changed its
	 * appearance.
	 */
	public void onRefresh () {
		this.repaint = true;
		if (this.visible && this.parent) {
			this.parent.onRefresh();
		}
	}

	/**
	 * Sends a PaintEvent to all the children of this component
	 * that need to be repainted.
	 */
	public void repaintChildren () {
		Component c;

		c = this.child;
		while (c) {
			if (c.visible && c.repaint) {
				c.onPaint();
			}
			c = c.next;
		}
	}

	/**
	 * This method is called to draw the foreground of the
	 * container. This method is called after all the children
	 * of this container are painted.
	 */
	public void paintForeground () {}


	/**
	 * Paints the children of this container to the virtual screen.
	 *
	 * param clip The SCR_RECT clip
	 */
	public void paintChildren (SCR_RECT* clip) {
		Component c;

		c = this.child;
		while (c) {
			if (c.visible) {
				native {
					BitmapPut (c->x, c->y, c->img, clip, A_REPLACE);
				}
			}
			c = c.next;
		}
	}

	/**
	 * This method is invoked by the GEM when the user interacts
	 * with the container.
	 */
	public void use () {
		Component c;
		short x = 0;
		short y = 0;
		Container ct = this;

		if (!this.enabled) {
			return;
		}

		while (ct) {
			x = x + ct.x;
			y = y + ct.y;
			ct = ct.parent;
		}

		c = this.tail;

		while (c) {
			if (c.visible && c.isCursorOver(x, y)) {
				c.use();
				return;
			}

			c = c.prev;
		}

		if (this.listen) {
			this.listen.eventTriggered(this);
		}
	}

	/**
	 * Checks if the mnemonic of this container correspond
	 * to the current mnemonic. If it is the case, an event
	 * is triggered.
	 */
	public void checkMnemonic(char m) {
		Component c = this.tail;

		if (this.mnemonic == m) {
			if (this.listen) {
				this.listen.eventTriggered(this);
			}
			return;
		}

		while (c) {
			if (c.visible) {
				c.checkMnemonic(m);
			}
			c = c.prev;
		}
	}

	/**
	 * Adds the specified component to this container.
	 *
	 * @param c the component to add to this container.
	 */
	public void add (Component c) {
		if (!this.child) {
			this.child = c;
		}
		if (this.tail) {
			this.tail.next = c;
		}
		c.prev = this.tail;
		c.next = null;
		this.tail = c;
		c.parent = this;
		if (c.visible) {
			this.repaint = true;
			if (this.visible && this.parent) {
				this.parent.onRefresh();
			}
		}
	}

	/**
	 * Removes the specified component from this container.
	 *
	 * @param c the component to remove from this container
	 */
	public void remove (Component c) {
		Component cpn = this.child;

		while (cpn) {
			if (cpn == c) {
				if (c.prev) {
					c.prev.next = c.next;
				}
				if (c.next) {
					c.next.prev = c.prev;
				}
				if (this.child == c) {
					this.child = c.next;
				}
				if (this.tail == c) {
					this.tail = c.prev;
				}
				c.parent = null;
				c.next = null;
				c.prev = null;
				break;
			}
			cpn = cpn.next;
		}

		if (c.visible) {
			this.repaint = true;
			if (this.visible && this.parent) {
				this.parent.onRefresh();
			}
		}
	}

	/**
	 * Empties the container (removes all its children).
	 */
	public void empty () {
		Component c;
		Component cn = this.child;

		while (cn) {
			c = cn;
			cn = cn.next;
			c.parent = null;
			c.next = null;
			c.prev = null;
		}
		this.child = null;
		this.tail = null;

		this.repaint = true;
		if (this.visible && this.parent) {
			this.parent.onRefresh();
		}
	}

	/**
	 * Called to brings a component to the front
	 * of this container.
	 *
	 * @param c The component to bring to front.
	 * @return This container if it is the component to bring to front or if a child of
	 * 			this container is, null otherwize.
	 */
	public Component toFront (Component c) {
		Component cpn = this.child;

		if (this == c) {
			return this;
		}

		while (cpn) {
			if (cpn == cpn.toFront(c)) {
				if (cpn != this.tail) {
					this.remove(cpn);
					this.add(cpn);

					if (c.visible) {
						this.repaint = true;
						if (this.visible && this.parent) {
							this.parent.onRefresh();
						}
					}
				}
				return this;
			}
			cpn = cpn.next;
		}

		return null;
	}

	/**
	 * Disposes of all ressources used by this container and
	 * all its children.
	 */
	protected void clean () {
		Component c;
		Component cn = this.child;

		while (cn) {
			c = cn;
			cn = cn.next;
			c.clean();
		}

		this.finalize();
	}
}