/*
 * QUSOFT MICROSYSTMES
 * Moka
 * Copyright 2002 Frdric Brown
 */
 
/*
 *  This file is part of Moka API.
 *
 *  Moka API is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Moka API is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Moka API; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package moka.x;

import moka.x.GEM;
import moka.event.EventListener;

/**
 * Component is the base class for user-interface objects.
 *
 * @author  Frdric Brown
 * @version 1.01, 2003-11-02
 * @since   MDK1.0a
 */
public class Component extends EventListener {
	
	/** The image of the component to be displayed.*/
	private BITMAP * img;
	
	/** Is the component need to be repainted ?*/
	private boolean repaint;
	
	/** The next component in the container of this Component.*/
	protected Component next;
	
	/** The previous component in the container of this Component.*/
	protected Component prev;
	
	/** The parent Container of this Component.*/
	protected Container parent;
	
	/** The height of this Component. Should be only read.*/
	public short height;
	/** The width of this Component. Should be only read.*/
	public short width;
	/** The horizontal coordinate of this Component. Should be only read.*/
	public short x;
	/** The vertical coordinate of this Component. Should be only read.*/
	public short y;
	
	/** Is the Component visible ? Should be only read.*/
	public boolean visible;
	
	/** Is the Component enabled ? Should be only read.*/
	public boolean enabled;
	
	/** The mnemonic associated with this component.*/
	public char mnemonic;
	
	/** The event listener for this component.*/
	public EventListener listen;
	
	/**
	 * Constructs a newly allocated Component.
	 */
	public Component () {
		this.img = null;
		this.repaint = true;
		this.parent = null;
		this.mnemonic = '\0';
		this.height = 25;
		this.width = 25;
		this.x = 0;
		this.y = 0;
		this.visible = true;
		this.enabled = true;
		this.listen = this;
	}
	
    /**
     * Free the memory and system ressources used by this Component.
     */
	public native void finalize () {
		if (this->img) {
			free(this->img);
		}
		free(this);
	}

    /**
     * Invoked when an event is triggered.
     *
     * @param sender The object who triggered the event.
     */
	public void eventTriggered(Object sender) {
		
	}

	/**
	 * Called when a PaintEvent occurs to paint the image of the component and
	 * stores it in its virtual memory.  A PaintEvent occurs when the aspect of
	 * the component is altered.
	 */
	public native void onPaint () {
		this->repaint = FALSE;
		SCR_RECT rect = (SCR_RECT){{0, 0, this->width - 1, this->height - 1}};

		PortSet(GEM_gem->port, 239, 127);
		ScrRectFill (&rect, &rect, A_REVERSE);
		
		this->paint_(this);
		
		if (this->img) {
			free (this->img);
		}
		
		this->img = malloc(BitmapSize(&rect));
		
		BitmapGet (&rect, this->img);
		PortSet ((void *) 0x4C00, 239, 127);
	}

	/**
	 * This method is called to draw the component.  Subclasses of Component
	 * are encouraged to overrides the <code>paint</code> method.
	 */
	public void paint () {}
	
	/*
	 * Sets the bounds (position and size) of the component.
	 *
	 * @param x the horizontal coordinate.
	 * @param y the vertical coordinate.
	 * @param width the width of the component.
	 * @param height the height of the component.
	 */
	public void setBounds (short x, short y, short width, short height) {
		this.x = x;
		this.y = y;
		this.width = width;
		this.height = height;
		
		this.repaint = true;
		if (this.visible && this.parent) {
			this.parent.onRefresh();
		}
	}
	
	/*
	 * Sets the horizontal coordinate of the component.
	 *
	 * @param x the horizontal coordinate.
	 */
	public void setX (short x) {
		this.x = x;
		
		if (this.visible && this.parent) {
			this.parent.onRefresh();
		}
	}

	/*
	 * Sets the vertical coordinate of the component.
	 *
	 * @param y the vertical coordinate.
	 */
	public void setY (short y) {
		this.y = y;
		
		if (this.visible && this.parent) {
			this.parent.onRefresh();
		}
	}
	
	/*
	 * Sets the width of the component.
	 *
	 * @param width the width of the component.
	 */
	public void setWidth (short width) {
		this.width = width;
		
		this.repaint = true;
		if (this.visible && this.parent) {
			this.parent.onRefresh();
		}
	}
	
	/*
	 * Sets the height of the component.
	 *
	 * @param height the height of the component.
	 */
	public void setHeight (short height) {
		this.height = height;
		
		this.repaint = true;
		if (this.visible && this.parent) {
			this.parent.onRefresh();
		}
	}
	
	/*
	 * Sets the position of the component.
	 *
	 * @param x the horizontal coordinate.
	 * @param y the vertical coordinate.
	 */
	public void move (short x, short y) {
		this.x = x;
		this.y = y;
		
		if (this.visible && this.parent) {
			this.parent.onRefresh();
		}
	}
	
	/**
	 * Sets the visibility of the component.
	 * 
	 * @param b true if the component should be visible, false otherwize.
	 */
	public void setVisible (boolean b) {		
		this.visible = b;
		
		if (this.parent) {
			this.parent.onRefresh();
		}
	}
	
	/**
	 * Sets if the component is enabled or not.
	 * 
	 * @param b true if the component should be enabled, false otherwize.
	 */
	public void setEnabled (boolean b) {		
		this.enabled = b;
		
		this.repaint = true;
		if (this.visible && this.parent) {
			this.parent.onRefresh();
		}
	}
	
	/**
	 * Sets the mnemonic for this component.
	 * 
	 * @param mnemonic.
	 */
	public void setMnemonic (char mnemonic) {
		this.mnemonic = mnemonic;
		
		this.repaint = true;
		if (this.visible && this.parent) {
			this.parent.onRefresh();
		}
	}
	
	/**
	 * This method is invoked by the GEM when the user interacts
	 * with the component.
	 */
	public void use () {
		if (this.enabled && this.listen) {
			this.listen.eventTriggered(this);
		}
	}
	
	/**
	 * Indicates if the cursor is over this component.
	 *
	 * @param x the reference's horizontal coordinate
	 * @param y	the reference's vertical coordinate
	 * @return true if the cursor is over the component, false otherwize.
	 */
	public boolean isCursorOver (short x, short y) {				
		return GEM.gem.cursorX >= this.x + x && GEM.gem.cursorY >= this.y + y && GEM.gem.cursorX < this.x + x + this.width && GEM.gem.cursorY < this.y + y + this.height;
	}
	
	/*
	 * Centers the component in its parent.
	 */
	public void center() {
		this.x = (this.parent.width - this.width) / 2;
		this.y = (this.parent.height - this.height) / 2;
		
		if (this.visible && this.parent) {
			this.parent.onRefresh();
		}
	}
	
	/*
	 * Sends the component to the top of its parent.
	 */
	public void top() {
		this.y = 0;
		
		if (this.visible && this.parent) {
			this.parent.onRefresh();
		}
	}
	
	/*
	 * Sends the component to the bottom of its parent.
	 */
	public void bottom() {
		this.y = this.parent.height - this.height ;
		
		if (this.visible && this.parent) {
			this.parent.onRefresh();
		}
	}
	
	/*
	 * Sends the component to the left of its parent.
	 */
	public void left() {
		this.x = 0 ;

		if (this.visible && this.parent) {
			this.parent.onRefresh();
		}
	}
	
	/*
	 * Sends the component to the right of its parent.
	 */
	public void right() {
		this.x = this.parent.width - this.width ;
		
		if (this.visible && this.parent) {
			this.parent.onRefresh();
		}
	}
	
	/**
	 * Called to brings a component to the front
	 * of a container.
	 * 
	 * @param c the component to bring to front.
	 * @return this component if it is the component to bring to front, null otherwize.
	 */
	public Component toFront (Component c) {
		if (c == this) {
			return c;
		}
		
		return null;
	}
	
	/**
	 * Disposes of all ressources used by this component.
	 */
	protected void clean () {
		this.finalize();
	}
	
	/**
	 * Checks if the mnemonic of this component correspond
	 * to the current mnemonic. If it is the case, an event
	 * is triggered.
	 */
	public void checkMnemonic(char m) {
		if (this.listen && this.mnemonic == m) {
			//this.listen.eventTriggered(this);
			this.use();
		}
	}
}