/*
 * QUSOFT MICROSYSTMES
 * Moka
 * Copyright 2002 Frdric Brown
 */

/*
 *  This file is part of Moka API.
 *
 *  Moka API is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Moka API is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Moka API; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package moka.x;

import moka.x.Image;
import moka.x.Color;
import moka.x.Container;
import moka.io.Graphics;

/**
 * Canvas class is a drawing component.
 *
 * @author  Frdric Brown
 * @version 1.0, 2003-03-06
 * @since   MDK1.0a
 */
public class Canvas extends Image {

	/** The color of the pen use to draw in this Canvas. Can be either Color.BLACK_COLOR, Color.WHITE_COLOR or Color.INVERT_COLOR.*/
	public short color;

	/*
	 * Constructs a new Canvas.
	 */
	public native Canvas () {
		Image__((TImage*)this);
		SCR_RECT rec = (SCR_RECT){{0, 0,  GEM_LCD_WIDTH - 1, GEM_LCD_HEIGHT - 1}};
		PortSet(GEM_gem->port, GEM_LCD_WIDTH - 1, GEM_LCD_HEIGHT - 1);

		clrscr();

		this->bmp = (BITMAP*)malloc(BitmapSize(&rec));

		BitmapGet(&rec, this->bmp);

		PortSet ((void *) 0x4C00, 239, 127);

		this->color = Color_BLACK_COLOR;
	}

	/*
	 * Constructs a new Canvas specifying its bitmap.
	 *
	 * @param bmp the Image's BITMAP
	 */
	public Canvas (BITMAP* bmp) {
		super(bmp);
		this.color = Color.BLACK_COLOR;
	}

	/*
	 * Constructs a new Canvas specifying its PIC file name.
	 * If the file string object risk to be used later, use a copy of the string object
         * instead of the original, the string being finalized before the method returns.
	 *
	 * @param file The Canvas's PIC file name
	 */
	public Canvas (String file) {
		super(file);
		this.color = Color.BLACK_COLOR;
	}

	/**
	 * Draws a pixel in the canvas.
	 *
	 * @param x The horizontal coodinate
	 * @param y The vertical coodinate
	 */
	public void drawPix (short dx, short dy) {
		native {
			short x = this->x;
			short y = this->y;
			TContainer* ct = this->parent;
			SCR_RECT rec = (SCR_RECT){0, 0,  GEM_LCD_WIDTH - 1, GEM_LCD_HEIGHT - 1};

			while (ct) {
				x += ct->x;
				y += ct->y;
				ct = ct->parent;
			}

			PortSet(GEM_gem->cursorImg, GEM_LCD_WIDTH - 1, GEM_LCD_HEIGHT - 1);

			DrawPix(x + dx, y + dy, this->color);

			PortSet(GEM_gem->port, GEM_LCD_WIDTH - 1, GEM_LCD_HEIGHT - 1);

			BitmapPut (0, 0, this->bmp, ScrRect, A_REPLACE);

			DrawPix(dx, dy, this->color);

			free(this->bmp);

			this->bmp = (BITMAP*)malloc(BitmapSize(&rec));

			BitmapGet(&rec, this->bmp);

			PortSet ((void *) 0x4C00, 239, 127);
		}

		this.repaint = true;
		if (this.visible && this.parent) {
			this.parent.onRefresh();
		}
	}

	/**
	 * Gets the status of a pixel on the canvas.
	 *
	 * @param x The x coordinate
	 * @param y The y coordinate
	 * @return true if the pixel is on, false otherwize
	 */
	public native boolean getPix (short dx, short dy) {
		short x = this->x;
		short y = this->y;
		TContainer* ct = this->parent;

		while (ct) {
			x += ct->x;
			y += ct->y;
			ct = ct->parent;
		}

		return GetPix(x + dx, y + dy);
	}

	/**
	 * Saves the area of the canvas.
	 */
	public void save () {
		native {
			SCR_RECT rec;
			BITMAP* bmp;
			short x = this->x;
			short y = this->y;
			TContainer* ct = this->parent;

			while (ct) {
				x += ct->x;
				y += ct->y;
				ct = ct->parent;
			}

			rec = (SCR_RECT){x, y,  this->width - 1, this->height - 1};
			bmp = (BITMAP*)malloc(BitmapSize(&rec));

			BitmapGet (&rec, bmp);
		}


		this.setBmp(bmp);
	}


	/**
	 * Erases the the canvas.
	 */
	public void erase () {
		native {
			SCR_RECT rec = (SCR_RECT){0, 0,  GEM_LCD_WIDTH - 1, GEM_LCD_HEIGHT - 1};
			BITMAP* bmp = (BITMAP*)malloc(BitmapSize(&rec));
			PortSet(GEM_gem->port, GEM_LCD_WIDTH - 1, GEM_LCD_HEIGHT - 1);

			clrscr();

			BitmapGet(&rec, bmp);

			PortSet ((void *) 0x4C00, 239, 127);
		}

		this.setBmp(bmp);
	}

	/**
	 * Saves the bitmap displayed in the canvas to a PIC file.
	 * If the file string object risk to be used later, use a copy of the string object
         * instead of the original, the string being finalized before the method returns.
	 *
	 * @param file The PIC file name
	 */
	public void saveToFile(String file) {
		Graphics.bitmapToVar(this.bmp, file);
	}

	/**
	 * This method is invoked by the GEM when the user interacts
	 * with the Canvas.
	 */
	public void use () {
		short x = this.x;
		short y = this.y;
		Container ct = this.parent;

		if (!this.enabled) {
			return;
		}

		while (ct) {
			x = x + ct.x;
			y = y + ct.y;
			ct = ct.parent;
		}

		this.drawPix(GEM.gem.cursorX - x, GEM.gem.cursorY - y);

		super.use();
	}
}