/*
 * QUSOFT MICROSYSTMES
 * Moka
 * Copyright 2002 Frdric Brown
 */

/*
 *  This file is part of Moka API.
 *
 *  Moka API is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Moka API is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Moka API; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
 package moka.util;

 import moka.lang.Object;
 import moka.util.Enumeration;
 import moka.util.Entry;

 /**
 * The <code>Vector</code> class implements a growable array of
 * objects. Like an array, it contains components that can be
 * accessed using an integer index. However, the size of a
 * <code>Vector</code> can grow or shrink as needed to accommodate
 * adding and removing items after the <code>Vector</code> has been created.<p>
 *
 * Each vector tries to optimize storage management by maintaining a
 * <code>capacity</code> and a <code>capacityIncrement</code>. The
 * <code>capacity</code> is always at least as large as the vector
 * size; it is usually larger because as components are added to the
 * vector, the vector's storage increases in chunks the size of
 * <code>capacityIncrement</code>. An application can increase the
 * capacity of a vector before inserting a large number of
 * components; this reduces the amount of incremental reallocation. <p>
 *
 * @author  Frdric Brown
 * @version 1.0, 2002-08-29
 * @since   MDK1.0a
 */
public class Vector {
	/** The actual array of objects.*/
	private Object * array;

	/** The actual capacity of the Vector.  Should be only read.*/
 	public int capacity;
 	/** The capacity increment of the Vector.  Sould be >= 1.*/
 	public int capacityIncrement;
 	/** The actual size of the Vector.  Should be only read.*/
 	public int size;

 	/**
 	 * Creates a new Vector with an initial capacity of 10 and
 	 * a capacity increment of 5.
 	 */
 	public Vector () {
 		this(10, 5);
 	}

 	/**
 	 * Creates a new Vector with the specified capacity and
 	 * capacity increment.
 	 *
 	 * @param initialCapacity The initial capacity.
 	 * @param capacityIncrement The capacity increment.
 	 */
 	public Vector (int initialCapacity, int capacityIncrement) {
 		this.capacity = initialCapacity;
 		this.capacityIncrement = capacityIncrement;
 		this.size = 0;
 		this.array = native ( malloc(sizeof(TObject*) * initialCapacity) );
 	}

 	/**
 	 * Creates a new Vector adding the elements of
 	 * the specified enumeration.
 	 *
 	 * @param enum the elements to add to the vector.
 	 */
 	public Vector (Enumeration enu) {
		while (enu.hasMoreElements()) {
			this.push(enu.nextElement());
		}

		enu.finalize();
 	}

    /**
     * Returns an enumeration of the components of this vector. The
     * returned <tt>Enumeration</tt> object will generate all items in
     * this vector. The first item generated is the item at index <tt>0</tt>,
     * then the item at index <tt>1</tt>, and so on.
     *
     * @return  an enumeration of the components of this vector.
     */
    public Enumeration elements() {
    	Enumeration enu = new Enumeration();
    	Entry e;
    	int i;

    	if (this.size) {
    		e = new Entry((Object)this.array[0], (Entry)null, (Entry)null);
    		enu.first = e;
    	}

    	for (i = 1; i < this.size; i++) {
    		e.next = new Entry((Object)this.array[i], e, (Entry)null);
    		e = e.next;
    	}

    	enu.current = enu.first;

    	return enu;
    }

    /**
     * Inserts the specified element at the specified position in this vector.
     * Shifts the element currently at that position
     * (if any) and any subsequent elements to the right (adds one to their
     * indices).
     *
     * @param index index at which the specified element is to be inserted.
     * @param element element to be inserted.
     */
 	public native void add (int index, Object o) {
 		TObject** array;

 		if (this->size < this->capacity) {
 			memmove(this->array + index + 1, this->array + index, (this->size - index) * sizeof(TObject*));

 			this->array[index] = o;
 		}
 		else {
 			array = malloc(sizeof(TObject*) * (this->capacity + this->capacityIncrement));
 			this->capacity += this->capacityIncrement;

 			memcpy(array, this->array, (index) * sizeof(TObject*));

 			array[index] = o;

 			memcpy(array + index + 1, this->array + index, (this->size - index) * sizeof(TObject*));

 			free(this->array);

 			this->array = array;
 		}

 		this->size++;
 	}

    /**
     * Pushes an item onto the top of this stack.
     *
     * @param   item   the item to be pushed onto this stack.
     */
 	public native void push (Object item) {
 		TObject** array;

 		if (this->size < this->capacity) {
 			this->array[this->size] = item;
 		}
 		else {
 			array = malloc(sizeof(TObject*) * (this->capacity + this->capacityIncrement));
 			this->capacity += this->capacityIncrement;

 			memcpy(array, this->array, (this->size) * sizeof(TObject*));

 			array[this->size] = item;

 			free(this->array);

 			this->array = array;
 		}

		this->size++;
 	}

    /**
     * Enqueues an item at the end of this queue.
     *
     * @param   item   the item to be enqueued.
     */
 	public native void enqueue (Object item) {
 		TObject** array;

 		if (this->size < this->capacity) {
 			this->array[this->size] = item;
 		}
 		else {
 			array = malloc(sizeof(TObject*) * (this->capacity + this->capacityIncrement));
 			this->capacity += this->capacityIncrement;

 			memcpy(array, this->array, (this->size) * sizeof(TObject*));

 			array[this->size] = item;

 			free(this->array);

 			this->array = array;
 		}

		this->size++;
 	}

    /**
     * Removes all of the elements from this vector.  This
     * vector will be empty after this call returns.
     */
    public void clear () {
    	this.size = 0;
    }

    /**
     * Frees the memory used by this object.
     */
    public native void finalize () {
    	free(this->array);
    	free(this);
    }

    /**
     *
     * Returns <tt>true</tt> if this vector contains the specified element.
     *
     * @param o element whose presence in this vector is to be tested.
     * @return <tt>true</tt> if this vector contains the specified element.
     */
    public boolean contains (Object o) {
    	int i;

    	for (i = 0; i < this.size; i++) {
    		if (this.array[i] == o) {
    			break;
    		}
    	}

    	return (i < this.size);
    }

    /**
     * Returns the element at the specified position in this vector.
     *
     * @param index index of element to return.
     * @return the element at the specified position in this vector.
     */
    public Object get (int index) {
    	return this.array[index];
    }

    /**
     * Returns the index in this vector of the first occurrence of the specified
     * element, or -1 if this list does not contain this element.
     *
     * @param o element to search for.
     * @return the index in this vector of the first occurrence of the specified
     * 	       element, or -1 if this vector does not contain this element.
     */
    public int indexOf (Object o) {
    	int i;

    	for (i = 0; i < this.size; i++) {
    		if (this.array[i] == o) {
    			break;
    		}
    	}

    	if (i < this.size) {
    		return i;
    	}
    	else {
    		return -1;
    	}
    }

    /**
     * Returns the index in this vector of the last occurrence of the specified
     * element, or -1 if this list does not contain this element.
     *
     * @param o element to search for.
     * @return the index in this vector of the last occurrence of the specified
     * 	       element, or -1 if this vector does not contain this element.
     */
    public int lastIndexOf (Object o) {
    	int i;

    	for (i = this.size - 1; i >= 0; i--) {
    		if (this.array[i] == o) {
    			break;
    		}
    	}

    	if (i >= 0) {
    		return i;
    	}
    	else {
    		return -1;
    	}
    }

    /**
     * Removes the element at the specified position in this vector.
     * Shifts any subsequent elements to the left (subtracts one
     * from their indices).  Returns the element that was removed from the
     * vector.
     *
     * @param index the index of the element to removed.
     * @return the element previously at the specified position.
     */
    public native Object remove(int index) {
    	TObject* o = this->array[index];

    	memmove(this->array + index, this->array + index + 1, (this->size - index - 1) * sizeof(TObject*));
    	this->size--;

    	return o;
    }

    /**
     * Removes the first occurrence in this vector of the specified element.
     * If this vector does not contain the element, it is
     * unchanged.
     *
     * @param o element to be removed from this vector, if present.
     * @return <tt>true</tt> if this vector contained the specified element.
     */
    public native boolean remove (Object o) {
    	long int i;

    	for (i = 0; i < this->size; i++) {
    		if (this->array[i] == o) {
    			break;
    		}
    	}

    	if (i < this->size) {
    		memmove(this->array + i, this->array + i + 1, (this->size - i - 1) * sizeof(TObject*));
    		this->size--;
    		return TRUE;
    	}
    	else {
    		return FALSE;
    	}
    }

    /**
     * Looks at the object at the top of this stack without removing it
     * from the stack.
     *
     * @return     the object at the top of this stack (the last item
     *             of the <tt>Vector</tt> object).
     */
    public Object peek() {
    	return this.array[this.size - 1];
    }

    /**
     * Removes the object at the top of this stack and returns that
     * object as the value of this function.
     *
     * @return     The object at the top of this stack (the last item
     *             of the <tt>Vector</tt> object).
     */
    public Object pop () {
    	Object o = this.array[this.size - 1];

		this.size--;

    	return o;
    }

    /**
     * Removes the object at the beginning of this queue and returns that
     * object as the value of this function.
     *
     * @return     The object at the beginning of this queue (the first item
     *             of the <tt>Vector</tt> object).
     */
    public native Object dequeue () {
    	TObject* o = this->array[0];

		this->size--;

		memmove(this->array, this->array + 1, (this->size) * sizeof(TObject*));

    	return o;
    }

    /**
     * Replaces the element at the specified position in this vector with the
     * specified element.
     *
     * @param index index of element to replace.
     * @param element element to be stored at the specified position.
     * @return the element previously at the specified position.
     */
    public Object set (int index, Object element) {
    	Object o = this.array[index];

    	this.array[index] = element;

    	return o;
    }

    /**
     * Copies the components of this vector into the specified array.
     * The array must be big enough to hold
     * all the objects copied from this vector.
     *
     * @param   anArray   the array into which the components get copied.
     * @param fromIndex the index of the first object in the vector
     *                  to copy.
     * @param toIndex the index after the last object in the vector
     *                to copy.
     */
    public native void copyInto (Object* anArray, int fromIndex, int toIndex) {
    	memcpy(anArray, this->array + fromIndex, (toIndex - fromIndex) * sizeof(TObject*));
    }

    /**
     * Searches for the first occurence of the given argument, beginning
     * the search at <code>index</code>.
     *
     * @param   elem    an object.
     * @param   index   the index to start searching from.
     * @return  the index of the first occurrence of the object argument in
     *          this vector at position <code>index</code> or later in the
     *          vector or <code>-1</code> if the object is not
     *          found.
     */
    public int indexOf (Object elem, int index) {
    	int i;

    	for (i = index; i < this.size; i++) {
    		if (this.array[i] == elem) {
    			break;
    		}
    	}

    	if (i < this.size) {
    		return i;
    	}
    	else {
    		return -1;
    	}
    }

    /**
     * Searches backwards for the specified object, starting from the
     * specified index, and returns an index to it.
     *
     * @param  elem    the desired component.
     * @param  index   the index to start searching from.
     * @return the index of the last occurrence of the specified object in this
     *          vector at position less than or equal to <code>index</code> in
     *          the vector or return <code>-1</code> if the object is not found.
     */
    public int lastIndexOf (Object elem, int index) {
    	int i;

    	for (i = index; i >= 0; i--) {
    		if (this.array[i] == elem) {
    			break;
    		}
    	}

    	if (i >= 0) {
    		return i;
    	}
    	else {
    		return -1;
    	}
    }

    /**
     * Sets the capacity of this vector.
     *
     * @param   newCapacity   the new size of this vector.
     */
    public native void setCapacity(int newCapacity) {
    	TObject** array = malloc(sizeof(TObject*) * newCapacity);

    	memcpy(array, this->array, (this->size) * sizeof(TObject*));

    	this->capacity = newCapacity;

    	free(this->array);

    	this->array = array;
    }

    /**
     * Returns a string representation of this Vector, containing
     * the String representation of each element.
     */
    public String toString() {
    	int i;
    	String str = "[";

    	for (i = 0; i < this.size; i++) {
    		str.concat("[");
    		str.concat((String)this.array[i].toString());
    		str.concat("]");
    	}

    	str.concat("]");

    	return str;
    }

    /**
     * Trims the capacity of this vector to be the vector's current
     * size. An application can use this operation to
     * minimize the storage of a vector.
     */
    public native void trimToSize() {
    	TObject** array = malloc(sizeof(TObject*) * this->size);

    	memcpy(array, this->array, (this->size) * sizeof(TObject*));

    	this->capacity = this->size;

    	free(this->array);

    	this->array = array;
    }
}