/*
 * QUSOFT MICROSYSTMES
 * Moka
 * Copyright 2002 Frdric Brown
 */

/*
 *  This file is part of Moka API.
 *
 *  Moka API is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Moka API is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Moka API; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
 package moka.util;

 import moka.lang.Object;
 import moka.event.Interrupt;
 import moka.event.EventListener;

 /**
 * A facility to schedule tasks for future execution in a
 * background loop.  This class proposes also a static method to pause the execution of the
 * program.
 *
 * @author  Frdric Brown
 * @version 1.11, 2002-08-29
 * @since   MDK1.0a
 */
public class Timer extends Interrupt {
	/**
	 * The id number of the timer (1 to 6: BATT_TIMER = 1, APD_TIMER = 2, LIO_TIMER = 3, CURSOR_TIMER = 4, MISC_TIMER = 5, USER_TIMER = 6). Should be modified only when the timer is inactive.
	 * @deprecated As of MDK2.1. Removed from MDK2.21.
	 */
	//public short no;
	/** Is the timer activated ? Should be only read.*/
	public boolean active;
	/** The delay of the timer (in 1/20th of second). Should be modified only when the timer is inactive.*/
	public int delay;
	/** The time stamp value when this timer will expire.*/
	public int time;
	/** The listener binded to this timer.*/
	public EventListener listen;

	/** The current time stamp (in 1/20th of second). Should be only read.*/
	public static volatile int timeStamp;

	static {
		Timer.timeStamp = 0;
	}

	/**
	 * Creates a new Timer using a 20 1/20th of second delay (1 second).
	 */
	public Timer () {
		this(20);
	}

	/**
	 * Creates a new Timer specifying the delay.
	 *
	 * @param delay The delay of the timer (in 1/20th of second).
	 * @since MDK2.1
	 */
	public Timer (int delay) {
		this.active = false;
		this.delay = delay;
		this.listen = null;
		this.time = 0;
		this.registered = false;
	}

	/**
	 * Creates a new Timer specifying the timer ID number and the delay.
	 *
	 * @param no The id number of the timer (1 to 6).
	 * @param delay The delay of the timer (in 1/20th of second).
	 * @deprecated As of MDK2.1, use Timer(short delay) instead. Removed from MDK2.21.
	 */
	/*public Timer (short no, int delay) {
		this.no = no;
		this.active = false;
		this.delay = delay;
		this.listen = null;
		this.time = 0;
		this.registered = false;
	}*/

	/**
	 * Used to verify if this timer should trigger an event. It is the case when
	 * the timer is actived and that its delay is expired. The timer is
	 * reseted if it is the case.
	 *
	 * @return true if this timer should trigger an event
	 */
	public native boolean check () {
		if (this->active) {
			/*if (this->no) {
				return this->active && OSTimerExpired (this->no);
			}
			else */ if (Timer_timeStamp >= this->time) {
				this->time = Timer_timeStamp + this->delay;
				return TRUE;
			}
		}

		return FALSE;
	}

	/**
	 * Handles the code to execute when this object interrupts.
	 * In the case of a timer, the EventListener will be notified.
	 */
	public native void run () {
		if (this->listen) {
			this->listen->eventTriggered_Object(this->listen, (TObject*)this);
			//OSFreeTimer (this->no);
			//OSRegisterTimer (this->no, this->delay);
		}
	}

    /**
     * Frees the ressources used by this Timer.
     *
     * @since MDK2.1
     */
	public void finalize () {
		this.setActive(false);
		native.free(this);
	}

	/**
	 * Sets if this timer is activated.
	 *
	 * @param b a flag indicating whether this timer is activated. If this flag is set to true and
	 * this timer is already activated, the timer is reseted.
	*/
	public native void setActive (boolean b) {
		if (b) {
			/*if (this->no) {
				OSFreeTimer (this->no);
				OSRegisterTimer (this->no, this->delay);
			}
			else {*/
				this->time = Timer_timeStamp + this->delay;
			//}
			if (! this->registered) {
				Interrupt_register_Interrupt(this);
			}
		}
		else if (this->registered) {
			Interrupt_unregister_Interrupt(this);
		}

		this->active = b;
	}

	/**
	 * Pauses the execution of the program until this timer's delay
	 * expires according that this timer is activated.  When this
	 * timer's delay expires, an event is triggered.
	 */
	public native void wait () {
		if (this->active) {
			/*if (this->no) {
				while (!OSTimerExpired (this->no));
			}
			else {*/
				while (Timer_timeStamp < this->time);
				this->time = Timer_timeStamp + this->delay;
			//}

			if (this->listen) {
				this->listen->eventTriggered_Object(this->listen, (TObject*)this);
			}

			//OSFreeTimer (this->no);
			//OSRegisterTimer (this->no, this->delay);
		}
	}

	/**
	 * Pauses the execution of the program until the specified delay
	 * expires.
	 *
	 * @param delay The the timer (in 1/20th of second).
	 */
	public native static void wait (int delay) {
		/*OSFreeTimer (BATT_TIMER);
		OSRegisterTimer (BATT_TIMER, delay);
		while (!OSTimerExpired (BATT_TIMER));*/
		int time = Timer_timeStamp + delay;

		while (Timer_timeStamp < time);
	}
}