/*
 * QUSOFT MICROSYSTMES
 * Moka
 * Copyright 2002 Frdric Brown
 */

/*
 *  This file is part of Moka API.
 *
 *  Moka API is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Moka API is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Moka API; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package moka.util;

import moka.lang.Object;

/**
 * The string tokenizer class allows an application to break a
 * string into tokens.
 * The delimiters (the characters that separate tokens) may
 * be specified either at creation time or on a per-token basis.
 *
 * @author  Frdric Brown
 * @version 1.0, 2002-11-08
 * @since   MDK1.0a
 */
public class StringTokenizer {
	/** An altered copy of the original string.*/
	private String str = null;
	/** A copy of the original string.*/
	private String origin = null;
	/** The index of the current char in the string. Should be only read.*/
	public int currentPosition;


    /**
     * Constructs a string tokenizer for the specified string. The
     * tokenizer uses the default delimiters, which is
     * the space character,
     * the tab character, the newline character, the carriage-return character,
     * and the form-feed character.
     *
     * @param   str   a string to be parsed.
     */
    public native StringTokenizer (String str) {
    	//TAB, NL, FF, CR, SPACE
    	char delim[] = {9, 10, 12, 13, 32, '\0'};
    	TString* dstr = TString_char_p(delim);

    	StringTokenizer__String_String(this, str, dstr);

    	//dstr->finalize_(dstr);
    }

    /**
     * Constructs a string tokenizer for the specified string. The
     * characters in the <code>delim</code> argument is the delimiters
     * for separating tokens.
     *
     * @param   str     a string to be parsed.
     * @param   delim   the delimiter.
     */
    public StringTokenizer (String str, String delim) {
    	int len = native ( strlen(delim->value) );
    	int i;
    	this.str = str;
    	this.origin = str.toString();
    	this.setCurrentPosition(0);

    	for (i = 0; i < len; i++) {
    		str.replace((char)delim.value[i], (char)1);
    	}
    	
    	delim.finalize();
    }

    /**
     * Frees the memory of system ressources used by this StringTokenizer
     * object.
     *
     */
 	public native void finalize() {
 		if (this->str) {
 			this->str->finalize_(this->str);
 			this->origin->finalize_(this->origin);
 		}
 		free(this);
 	}

    /**
     * Tests if there are more tokens available from this tokenizer's string.
     * If this method returns <tt>true</tt>, then a subsequent call to
     * <tt>nextToken</tt> with no argument will successfully return a token.
     *
     * @return  <code>true</code> if and only if there is at least one token
     *          in the string after the current position; <code>false</code>
     *          otherwise.
     */
    public native boolean hasMoreTokens () {
    	return this->currentPosition < strlen(this->str->value);
    }

    /**
     * Returns the next token from this string tokenizer.
     *
     * @return     the next token from this string tokenizer.
     */
    public String nextToken() {
    	int len = native ( strlen(this->str->value) );
    	int old = this.currentPosition;
    	int end;
    	char* begin = native ( &this->str->value[old] );
    	char* pos = native ( strchr(begin, 1) );

    	if (pos) {
    		this.currentPosition = end = old + (int)(pos - begin);
    	}
    	else {
    		this.currentPosition = end = len;
    	}

    	this.setCurrentPosition(this.currentPosition);

    	return this.str.substring(old, end);
    }

    /**
     * Returns the next token in this string tokenizer's string. First,
     * the set of characters considered to be delimiters by this
     * <tt>StringTokenizer</tt> object is changed to be the characters in
     * the string <te recognized tt>delim</tt>. Then the next token in the string
     * after the current position is returned. The current position is
     * advanced beyond thoken.  The new delimiter set
     * remains the default after this call.
     *
     * @param      delim   the new delimiters.
     * @return     the next token, after switching to the new delimiter set.
     */
    public String nextToken(String delim) {
    	//int len = native ( strlen(this->str->value) );
    	StringTokenizer st = new StringTokenizer(this.origin.toString(), delim);

    	this.str.finalize();

    	this.str = st.str.toString();

    	this.setCurrentPosition(this.currentPosition);

    	st.finalize();

    	return this.nextToken();
    }

    /**
     * Calculates the number of times that this tokenizer's
     * <code>nextToken</code> method can be called. The current
     * position is not advanced.
     *
     * @return  the number of tokens remaining in the string using the current
     *          delimiter set.
     */
    public int countTokens() {
    	int len = native ( strlen(this->str->value) );
    	int pos = this.currentPosition;
    	int cpt;


    	for (cpt = 0; pos < len; cpt++) {
    		pos = this.currentPosition + (int)(native ( strchr(&this->str->value[pos], 1) - &this->str->value[pos]));

    		while (pos < len && this.str.value[pos] == 1) pos++;
    	}

    	return cpt;
    }

    /**
     * Sets the current position to the specified index.
     *
     * currentPosition the new current position.
     */
    public void setCurrentPosition(int currentPosition) {
    	int len = native ( strlen(this->str->value) );
    	this.currentPosition = currentPosition;

    	while (this.currentPosition < len && this.str.value[this.currentPosition] == 1) this.currentPosition++;
    }
}