/*
 * QUSOFT MICROSYSTMES
 * Moka
 * Copyright 2002 Frdric Brown
 */

/*
 *  This file is part of Moka API.
 *
 *  Moka API is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Moka API is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Moka API; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package moka.util;

import moka.lang.Object;
import moka.io.Serializable;
import moka.util.SerialEntry;
import moka.util.Enumeration;

 /**
 * The <code>SerialList</code> class implements a serializable
 * list of serializable elements. Like an array, it contains components that can be
 * accessed using an integer index. However, the size of a
 * <code>SerialList</code> can grow or shrink as needed to accommodate
 * adding and removing items after the <code>SerialList</code> has been created.<p>
 *
 * Each serializable linked lists tries to optimize storage management by reallocating memory
 * every time the size of the array grows or shrinks.
 *
 * @author  Frdric Brown
 * @version 1.0, 2002-11-11
 * @since   MDK1.0a
 */
public class SerialList extends Serializable {
	/** The first entry in the LinkedList.*/
 	private SerialEntry first;

 	/** The actual size of the SerialList.  Should be only read.*/
 	public int size;

 	/**
 	 * Creates a new SerialList.
 	 */
 	public SerialList () {
 		this.first = null;
 		this.size = 0;
 	}

 	/**
 	 * Creates a new SerialList adding the elements of
 	 * the specified enumeration.  These elements should be
 	 * serializable  The enumeration is
 	 * finalized after this operation.
 	 *
 	 * @param enum the elements to add to the list.
 	 */
 	public SerialList (Enumeration enu) {
		Serializable obj;
 		SerialEntry n = null;

 		if (enu.hasMoreElements()) {
 			n = this.first = new SerialEntry((Serializable)enu.nextElement(), (SerialEntry)null);
 			this.size++;
 		}

 		while (obj = enu.nextElement()) {
 			n.next = new SerialEntry(obj, (SerialEntry)null);
 			n = n.next;
 			this.size++;
 		}

 		enu.current = enu.first;

 		enu.finalize();
 	}

    /**
     * Returns an enumeration of the components of this SerialList. The
     * returned <tt>Enumeration</tt> object will generate all items in
     * this SerialList. The first item generated is the item at index <tt>0</tt>,
     * then the item at index <tt>1</tt>, and so on.
     *
     * @return  an enumeration of the components of this vector.
     */
    public Enumeration elements() {
    	Enumeration enu = new Enumeration();
    	SerialEntry e;
    	SerialEntry c = this.first;

    	if (c) {
    		e = new SerialEntry(c.element, (SerialEntry)null);
    		enu.first = e;
    		c = c.next;
    	}

    	while (c) {
    		e.next = new SerialEntry(c.element, (SerialEntry)null);
    		e = e.next;
    		c = c.next;
    	}

    	return enu;
    }

    /**
     * Inserts the specified element at the specified position in this serial list.
     * Shifts the element currently at that position
     * (if any) and any subsequent elements to the right (adds one to their
     * indices).
     *
     * @param index index at which the specified element is to be inserted.
     * @param element element to be inserted.
     */
 	public void add (int index, Serializable o) {
 		SerialEntry a;
 		SerialEntry n;
 		SerialEntry p = null;
 		int i;

 		if (!this.first) {
 			n = new SerialEntry(o, (SerialEntry)null);
 			this.first = n;
 		}
 		else if (index < this.size) {
 			if (index == 0) {
 				n = new SerialEntry(o, this.first);
 				this.first = n;
 			}
 			else {
		    	a = this.first;

		    	for (i = 0; i != index; i++) {
		    		p = a;
		    		a = a.next;
		    	}

 				n = new SerialEntry(o, a);
 				p.next = n;
 			}
 		}
 		else {
 			a = this.first;

 			while (a.next) {
 				a = a.next;
 			}

 			n = new SerialEntry(o, (SerialEntry)null);
 			a.next = n;
 		}

 		this.size++;
 	}

    /**
     * Pushes an item onto the top of this stack.
     *
     * @param   item   the item to be pushed onto this stack.
     */
 	public void push (Serializable item) {
 		SerialEntry a;
 		SerialEntry n;

 		if (!this.first) {
 			n = new SerialEntry(item, (SerialEntry)null);
 			this.first = n;
 		}
 		else {
 			a = this.first;

 			while (a.next) {
 				a = a.next;
 			}

 			n = new SerialEntry(item, (SerialEntry)null);
 			a.next = n;
 		}

 		this.size++;
 	}

    /**
     * Enqueues an item at the end of this queue.
     *
     * @param   item   the item to be enqueued.
     */
 	public void enqueue (Serializable item) {
 		SerialEntry a;
 		SerialEntry n;

 		if (!this.first) {
 			n = new SerialEntry(item, (SerialEntry)null);
 			this.first = n;
 		}
 		else {
 			a = this.first;

 			while (a.next) {
 				a = a.next;
 			}

 			n = new SerialEntry(item, (SerialEntry)null);
 			a.next = n;
 		}

 		this.size++;
 	}

    /**
     * Removes all of the elements from this linked list.  This
     * linked list will be empty after this call returns.
     */
    public void clear () {
    	SerialEntry e = this.first;
    	SerialEntry n;

    	while (e) {
    		n = e.next;
    		e.finalize();
    		e = n;
    	}

    	this.size = 0;
    }

    /**
     * Frees the memory used by this object.
     */
    public void finalize () {
    	this.clear();
    	native {
    		free(this);
    	}
    }

    /**
     *
     * Returns <tt>true</tt> if this linked list contains the specified element.
     *
     * @param o element whose presence in this linked list is to be tested.
     * @return <tt>true</tt> if this linked list contains the specified element.
     */
    public boolean contains (Serializable o) {
    	SerialEntry e = this.first;

    	while (e) {
    		if (e.element == o) {
    			return true;
    		}
    		e = e.next;
    	}

    	return false;
    }

    /**
     * Returns the element at the specified position in this linked list.
     *
     * @param index index of element to return.
     * @return the element at the specified position in this linked list.
     */
    public Serializable get (int index) {
    	SerialEntry e;
    	int i;

    	e = this.first;

    	for (i = 0; i != index; i++) {
    		e = e.next;
    	}

    	return e.element;
    }

    /**
     * Returns the index in this linked list of the first occurrence of the specified
     * element, or -1 if this list does not contain this element.
     *
     * @param o element to search for.
     * @return the index in this linked list of the first occurrence of the specified
     * 	       element, or -1 if this linked list does not contain this element.
     */
    public int indexOf (Serializable o) {
    	SerialEntry e = this.first;
    	int i = 0;

    	while (e) {
    		if (e.element == o) {
    			return i;
    		}
    		e = e.next;
    		i++;
    	}

    	return -1;
    }

    /**
     * Returns the index in this linked list of the last occurrence of the specified
     * element, or -1 if this list does not contain this element.
     *
     * @param o element to search for.
     * @return the index in this linked list of the last occurrence of the specified
     * 	       element, or -1 if this linked list does not contain this element.
     */
    public int lastIndexOf (Serializable o) {
    	SerialEntry e = this.first;
    	int ind = -1;
    	int i = 0;

    	while (e) {
    		if (e.element == o) {
    			ind = i;
    		}
    		e = e.next;
    		i++;
    	}

    	return ind;
    }

    /**
     * Removes the element at the specified position in this linked list.
     * Shifts any subsequent elements to the left (subtracts one
     * from their indices).  Returns the element that was removed from the
     * linked list.
     *
     * @param index the index of the element to removed.
     * @return the element previously at the specified position.
     */
    public Serializable remove(int index) {
    	Serializable o;
    	SerialEntry p = null;
    	SerialEntry e = this.first;
    	int i;

    	for (i = 0; i != index; i++) {
    		p = e;
    		e = e.next;
    	}

    	if (e == this.first) {
    		this.first = e.next;
    	}
    	if (p) {
    		p.next = e.next;
    	}
    	o = e.element;
		e.finalize();
		this.size--;

    	return o;
    }

    /**
     * Removes the first occurrence in this linked list of the specified element.
     * If this linked list does not contain the element, it is
     * unchanged.
     *
     * @param o element to be removed from this linked list, if present.
     * @return <tt>true</tt> if this linked list contained the specified element.
     */
    public boolean remove (Serializable o) {
    	SerialEntry p = null;
    	SerialEntry e = this.first;

    	while (e) {
    		if (e.element == o) {
		    	if (e == this.first) {
		    		this.first = e.next;
		    	}
		    	if (p) {
		    		p.next = e.next;
		    	}
    			e.finalize();
    			this.size--;
    			return true;
    		}
    		p = e;
    		e = e.next;
    	}

    	return false;
    }

    /**
     * Looks at the object at the top of this stack without removing it
     * from the stack.
     *
     * @return     the object at the top of this stack (the last item
     *             of the <tt>LinkedList</tt> object).
     */
    public Serializable peek() {
    	SerialEntry e = this.first;

    	while(e.next) {
    		e = e.next;
    	}

    	return e.element;
    }

    /**
     * Removes the object at the top of this stack and returns that
     * object as the value of this function.
     *
     * @return     The object at the top of this stack (the last item
     *             of the <tt>LinkedList</tt> object).
     */
    public Serializable pop () {
    	SerialEntry p = null;
    	SerialEntry e = this.first;
    	Serializable o;

    	while(e.next) {
    		p = e;
    		e = e.next;
    	}

    	o = e.element;

    	if (p) {
    		p.next = null;
    	}

    	e.finalize();

    	this.size--;

    	return o;
    }

    /**
     * Removes the object at the beginning of this queue and returns that
     * object as the value of this function.
     *
     * @return     The object at the beginning of this queue (the first item
     *             of the <tt>LinkedList</tt> object).
     */
    public Serializable dequeue () {
    	SerialEntry e = this.first;
    	Serializable o = e.element;

    	this.first = e.next;

    	e.finalize();

    	this.size--;

    	return o;
    }

    /**
     * Replaces the element at the specified position in this linked list with the
     * specified element.
     *
     * @param index index of element to replace.
     * @param element element to be stored at the specified position.
     * @return the element previously at the specified position.
     */
    public Serializable set (int index, Serializable element) {
    	SerialEntry e = this.get(index);
    	Serializable o = e.element;

    	e.element = element;

    	return o;
    }

    /**
     * Copies the components of this linked list into the specified array.
     * The array must be big enough to hold
     * all the objects copied from this linked list.
     *
     * @param   anArray   the array into which the components get copied.
     * @param fromIndex the index of the first object in the linked list
     *                  to copy.
     * @param toIndex the index after the last object in the linked list
     *                to copy.
     */
    public void copyInto (Serializable* anArray, int fromIndex, int toIndex) {
    	SerialEntry e = this.first;
    	int i;
    	int ind = 0;

    	for (i = 0; i != fromIndex; i++) {
    		e = e.next;
    	}

    	for (i = fromIndex; i < toIndex; i++) {
    		anArray[ind++] = e.element;
    		e = e.next;
    	}
    }

    /**
     * Searches for the first occurence of the given argument, beginning
     * the search at <code>index</code>.
     *
     * @param   elem    an object.
     * @param   index   the index to start searching from.
     * @return  the index of the first occurrence of the object argument in
     *          this linked list at position <code>index</code> or later in the
     *          vector or <code>-1</code> if the object is not
     *          found.
     */
    public int indexOf (Serializable elem, int index) {
    	SerialEntry e = this.first;
    	int i = 0;

    	for (i = 0; i != index; i++) {
    		e = e.next;
    	}

    	while (e) {
    		if (e.element == elem) {
    			return i;
    		}
    		e = e.next;
    		i++;
    	}

    	return -1;
    }

    /**
     * Searches backwards for the specified object, starting from the
     * specified index, and returns an index to it.
     *
     * @param  elem    the desired component.
     * @param  index   the index to start searching from.
     * @return the index of the last occurrence of the specified object in this
     *          vector at position less than or equal to <code>index</code> in
     *          the linked list or return <code>-1</code> if the object is not found.
     */
    public int lastIndexOf (Serializable elem, int index) {
    	SerialEntry e = this.first;
    	int i;
    	int ind = -1;

    	for (i = 0; i != index; i++) {
    		if (e.element == elem) {
    			ind = i;
    		}
    		e = e.next;
    	}

    	return ind;
    }

    /**
     * Returns a string representation of this LinkedList, containing
     * the String representation of each element.
     */
    public String toString() {
    	String str = "[";
    	SerialEntry e = this.first;

    	while (e) {
    		str.concat("[");
    		str.concat(e.element.toString());
    		str.concat("]");
    		e = e.next;
    	}

    	str.concat("]");

    	return str;
    }
}