/*
 * QUSOFT MICROSYSTMES
 * Moka
 * Copyright 2002 Frdric Brown
 */

/*
 *  This file is part of Moka API.
 *
 *  Moka API is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Moka API is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Moka API; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
 package moka.util;

 import moka.lang.Object;

 /**
  * An instance of this class is used to generate a stream of
  * pseudorandom numbers.
  *
  * @author  Frdric Brown
  * @version 1.1, 2003-03-21
  * @since   MDK1.01a
  */
public class Random {

	/** The state associated with this pseudorandom number generator.*/
    public short seed;

    /**
     * Creates a new random number generator. Its seed is initialized to
     * a value based on the current time.
     * Two Random objects created within the same millisecond will have
     * the same sequence of random numbers.
     */
    public native Random() {
		randomize();
		this->seed = rand();
	}

    /**
     * Creates a new random number generator using a single
     * <code>short</code> seed.
     *
     * @param   seed   the initial seed.
     */
    public Random(short seed) {
        this.seed = seed;
    }

    /**
     * Generates the next pseudorandom short number.
     *
     * @return  the next pseudorandom value from this random number generator's sequence.
     */
    public native short nextShort () {
		srand(this->seed++);
		return rand();
	}

    /**
     * Generates the next pseudorandom double number.
     *
     * @return  the next pseudorandom value from this random number generator's sequence.
     */
    public native double nextDouble () {
		srand(this->seed++);
		return rand() / ((double) RAND_MAX);
	}

    /**
     * Generates the next pseudorandom boolean number.
     *
     * @return  the next pseudorandom value from this random number generator's sequence.
     */
    public native boolean nextBoolean () {
		srand(this->seed++);
		return (random(2) > 0);
	}

    /**
     * Generates random bytes and places them into a user-supplied
     * byte array.  The number of random bytes produced is equal to
     * the specified number.
     *
     * @param bytes  the non-null byte array in which to put the
     *               random bytes.
     * @param num the number of random bytes to produce
     */
    public void nextBytes(BYTE* bytes, short num) {
		short i;

		for (i = 0; i < num; i++) {
			bytes[i] = (BYTE)(this.nextShort());
		}
	}
}