/*
 * QUSOFT MICROSYSTMES
 * Moka
 * Copyright 2002 Frdric Brown
 */

/*
 *  This file is part of Moka API.
 *
 *  Moka API is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Moka API is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Moka API; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package moka.util;

import moka.lang.Object;
import moka.util.Entry;
import moka.util.Enumeration;

 /**
 * The <code>LinkedList</code> class implements doubly-linked
 * list. Like an array, it contains components that can be
 * accessed using an integer index. However, the size of a
 * <code>LinkedList</code> can grow or shrink as needed to accommodate
 * adding and removing items after the <code>LinkedList</code> has been created.<p>
 *
 * Each linked lists tries to optimize storage management by reallocating memory
 * every time the size of the array grows or shrinks.
 *
 * @author  Frdric Brown
 * @version 1.0, 2002-08-29
 * @since   MDK1.0a
 */
public class LinkedList {
	/** The first entry in the LinkedList.*/
 	private Entry header;
	/** The last entry in the LinkedList.*/
 	private Entry tail;

 	/** The actual size of the LinkedList.  Should be only read.*/
 	public int size;

 	/**
 	 * Creates a new LinkedList.
 	 */
 	public LinkedList () {
 		this.header = null;
 		this.tail = null;
 		this.size = 0;
 	}

 	/**
 	 * Creates a new LinkedList adding the elements of
 	 * the specified enumeration.  The enumeration is
 	 * finalized after this operation.
 	 *
 	 * @param enum the elements to add to the list.
 	 */
 	public LinkedList (Enumeration enu) {
		Object obj;
 		Entry n = null;

 		if (enu.hasMoreElements()) {
 			n = this.header = new Entry(enu.nextElement(), (Entry)null, (Entry)null);
 			this.size++;
 		}

 		while (obj = enu.nextElement()) {
 			n.next = new Entry(obj, n, (Entry)null);
 			n = n.next;
 			this.size++;
 		}

 		this.tail = n;

 		enu.finalize();
 	}

    /**
     * Returns an enumeration of the components of this LinkedList. The
     * returned <tt>Enumeration</tt> object will generate all items in
     * this LinkedList. The first item generated is the item at index <tt>0</tt>,
     * then the item at index <tt>1</tt>, and so on.
     *
     * @return  an enumeration of the components of this vector.
     */
    public Enumeration elements() {
    	Enumeration enu = new Enumeration();
    	Entry e;
    	Entry c = this.header;

    	if (c) {
    		e = new Entry(c.element, (Entry)null, (Entry)null);
    		enu.first = e;
    		c = c.next;
    	}

    	while (c) {
    		e.next = new Entry(c.element, e, (Entry)null);
    		e = e.next;
    		c = c.next;
    	}

    	enu.current = enu.first;

    	return enu;
    }

    /**
     * Inserts the specified element at the specified position in this linked list.
     * Shifts the element currently at that position
     * (if any) and any subsequent elements to the right (adds one to their
     * indices).
     *
     * @param index index at which the specified element is to be inserted.
     * @param element element to be inserted.
     */
 	public void add (int index, Object o) {
 		Entry a;
 		Entry n;
 		int i;

 		if (!this.header) {
 			n = new Entry(o, (Entry)null, (Entry)null);
 			this.header = n;
 			this.tail = n;
 		}
 		else if (index < this.size) {
 			if (index == 0) {
 				n = new Entry(o, (Entry)null, this.header);
 				this.header.previous = n;
 				this.header = n;
 			}
 			else {
		    	if (index >= this.size / 2) {
		    		a = this.tail;

		    		for (i = this.size - 1; i != index; i--) {
		    			a = a.previous;
		    		}
		    	}
		    	else {
		    		a = this.header;

		    		for (i = 0; i != index; i++) {
		    			a = a.next;
		    		}
		    	}
 				n = new Entry(o, a.previous, a);
 				a.previous.next = n;
 				a.previous = n;
 			}
 		}
 		else {
 			a = this.tail;
 			n = new Entry(o, a, (Entry)null);
 			a.next = n;
 			this.tail = n;
 		}

 		this.size++;
 	}

    /**
     * Pushes an item onto the top of this stack.
     *
     * @param   item   the item to be pushed onto this stack.
     */
 	public void push (Object item) {
 		Entry a;
 		Entry n;

 		if (!this.header) {
 			n = new Entry(item, (Entry)null, (Entry)null);
 			this.header = n;
 			this.tail = n;
 		}
 		else {
 			a = this.tail;
 			n = new Entry(item, a, (Entry)null);
 			a.next = n;
 			this.tail = n;
 		}

 		this.size++;
 	}

    /**
     * Enqueues an item at the end of this queue.
     *
     * @param   item   the item to be enqueued.
     */
 	public void enqueue (Object item) {
 		Entry a;
 		Entry n;

 		if (!this.header) {
 			n = new Entry(item, (Entry)null, (Entry)null);
 			this.header = n;
 			this.tail = n;
 		}
 		else {
 			a = this.tail;
 			n = new Entry(item, a, (Entry)null);
 			a.next = n;
 			this.tail = n;
 		}

 		this.size++;
 	}

    /**
     * Removes all of the elements from this linked list.  This
     * linked list will be empty after this call returns.
     */
    public void clear () {
    	Entry e = this.header;
    	Entry n;

    	while (e) {
    		n = e.next;
    		e.finalize();
    		e = n;
    	}

    	this.size = 0;
    }

    /**
     * Frees the memory used by this object.
     */
    public void finalize () {
    	this.clear();
    	native {
    		free(this);
    	}
    }

    /**
     *
     * Returns <tt>true</tt> if this linked list contains the specified element.
     *
     * @param o element whose presence in this linked list is to be tested.
     * @return <tt>true</tt> if this linked list contains the specified element.
     */
    public boolean contains (Object o) {
    	Entry e = this.header;

    	while (e) {
    		if (e.element == o) {
    			return true;
    		}
    		e = e.next;
    	}

    	return false;
    }

    /**
     * Returns the element at the specified position in this linked list.
     *
     * @param index index of element to return.
     * @return the element at the specified position in this linked list.
     */
    public Object get (int index) {
    	Entry e;
    	int i;

    	if (index >= this.size / 2) {
    		e = this.tail;

    		for (i = this.size - 1; i != index; i--) {
    			e = e.previous;
    		}
    	}
    	else {
    		e = this.header;

    		for (i = 0; i != index; i++) {
    			e = e.next;
    		}
    	}

    	return e.element;
    }

    /**
     * Returns the index in this linked list of the first occurrence of the specified
     * element, or -1 if this list does not contain this element.
     *
     * @param o element to search for.
     * @return the index in this linked list of the first occurrence of the specified
     * 	       element, or -1 if this linked list does not contain this element.
     */
    public int indexOf (Object o) {
    	Entry e = this.header;
    	int i = 0;

    	while (e) {
    		if (e.element == o) {
    			return i;
    		}
    		e = e.next;
    		i++;
    	}

    	return -1;
    }

    /**
     * Returns the index in this linked list of the last occurrence of the specified
     * element, or -1 if this list does not contain this element.
     *
     * @param o element to search for.
     * @return the index in this linked list of the last occurrence of the specified
     * 	       element, or -1 if this linked list does not contain this element.
     */
    public int lastIndexOf (Object o) {
    	Entry e = this.tail;
    	int i = this.size - 1;

    	while (e) {
    		if (e.element == o) {
    			break;
    		}
    		e = e.previous;
    		i--;
    	}

    	return i;
    }

    /**
     * Removes the element at the specified position in this linked list.
     * Shifts any subsequent elements to the left (subtracts one
     * from their indices).  Returns the element that was removed from the
     * linked list.
     *
     * @param index the index of the element to removed.
     * @return the element previously at the specified position.
     */
    public Object remove(int index) {
    	Object o;
    	Entry e;
    	int i;

    	if (index >= this.size / 2) {
    		e = this.tail;

    		for (i = this.size - 1; i != index; i--) {
    			e = e.previous;
    		}
    	}
    	else {
    		e = this.header;

    		for (i = 0; i != index; i++) {
    			e = e.next;
    		}
    	}

    	if (e == this.header) {
    		this.header = e.next;
    	}
    	if (e == this.tail) {
    		this.tail = e.previous;
    	}
    	if (e.previous) {
    		e.previous.next = e.next;
    	}
    	if (e.next) {
    		e.next.previous = e.previous;
    	}
    	o = e.element;
		e.finalize();
		this.size--;

    	return o;
    }

    /**
     * Removes the first occurrence in this linked list of the specified element.
     * If this linked list does not contain the element, it is
     * unchanged.
     *
     * @param o element to be removed from this linked list, if present.
     * @return <tt>true</tt> if this linked list contained the specified element.
     */
    public boolean remove (Object o) {
    	Entry e = this.header;

    	while (e) {
    		if (e.element == o) {
		    	if (e == this.header) {
		    		this.header = e.next;
		    	}
		    	if (e == this.tail) {
		    		this.tail = e.previous;
		    	}
		    	if (e.previous) {
		    		e.previous.next = e.next;
		    	}
		    	if (e.next) {
		    		e.next.previous = e.previous;
		    	}
    			e.finalize();
    			this.size--;
    			return true;
    		}
    		e = e.next;
    	}

    	return false;
    }

    /**
     * Looks at the object at the top of this stack without removing it
     * from the stack.
     *
     * @return     the object at the top of this stack (the last item
     *             of the <tt>LinkedList</tt> object).
     */
    public Object peek() {
    	return this.tail.element;
    }

    /**
     * Removes the object at the top of this stack and returns that
     * object as the value of this function.
     *
     * @return     The object at the top of this stack (the last item
     *             of the <tt>LinkedList</tt> object).
     */
    public Object pop () {
    	Entry e = this.tail;
    	Object o = e.element;

    	this.tail = e.previous;

    	if (e.previous) {
    		e.previous.next = null;
    	}
    	else {
    		this.header = null;
    	}

    	e.finalize();

    	this.size--;

    	return o;
    }

    /**
     * Removes the object at the beginning of this queue and returns that
     * object as the value of this function.
     *
     * @return     The object at the beginning of this queue (the first item
     *             of the <tt>LinkedList</tt> object).
     */
    public Object dequeue () {
    	Entry e = this.header;
    	Object o = e.element;

    	this.header = e.next;

    	if (e.next) {
    		e.next.previous = null;
    	}
    	else {
    		this.tail = null;
    	}

    	e.finalize();

    	this.size--;

    	return o;
    }

    /**
     * Replaces the element at the specified position in this linked list with the
     * specified element.
     *
     * @param index index of element to replace.
     * @param element element to be stored at the specified position.
     * @return the element previously at the specified position.
     */
    public Object set (int index, Object element) {
    	Entry e = this.get(index);
    	Object o = e.element;

    	e.element = element;

    	return o;
    }

    /**
     * Copies the components of this linked list into the specified array.
     * The array must be big enough to hold
     * all the objects copied from this linked list.
     *
     * @param   anArray   the array into which the components get copied.
     * @param fromIndex the index of the first object in the linked list
     *                  to copy.
     * @param toIndex the index after the last object in the linked list
     *                to copy.
     */
    public void copyInto (Object* anArray, int fromIndex, int toIndex) {
    	Entry e = this.header;
    	int i;
    	int ind = 0;

    	for (i = 0; i != fromIndex; i++) {
    		e = e.next;
    	}

    	for (i = fromIndex; i < toIndex; i++) {
    		anArray[ind++] = e.element;
    		e = e.next;
    	}
    }

    /**
     * Searches for the first occurence of the given argument, beginning
     * the search at <code>index</code>.
     *
     * @param   elem    an object.
     * @param   index   the index to start searching from.
     * @return  the index of the first occurrence of the object argument in
     *          this linked list at position <code>index</code> or later in the
     *          vector or <code>-1</code> if the object is not
     *          found.
     */
    public int indexOf (Object elem, int index) {
    	Entry e = this.header;
    	int i = 0;

    	for (i = 0; i != index; i++) {
    		e = e.next;
    	}

    	while (e) {
    		if (e.element == elem) {
    			return i;
    		}
    		e = e.next;
    		i++;
    	}

    	return -1;
    }

    /**
     * Searches backwards for the specified object, starting from the
     * specified index, and returns an index to it.
     *
     * @param  elem    the desired component.
     * @param  index   the index to start searching from.
     * @return the index of the last occurrence of the specified object in this
     *          vector at position less than or equal to <code>index</code> in
     *          the linked list or return <code>-1</code> if the object is not found.
     */
    public int lastIndexOf (Object elem, int index) {
    	Entry e = this.tail;
    	int i = this.size - 1;

    	for (i = this.size - 1; i != index; i--) {
    		e = e.previous;
    	}

    	while (e) {
    		if (e.element == elem) {
    			break;
    		}
    		e = e.previous;
    		i--;
    	}

    	return i;
    }

    /**
     * Returns a string representation of this LinkedList, containing
     * the String representation of each element.
     */
    public String toString() {
    	String str = "[";
    	Entry e = this.header;

    	while (e) {
    		str.concat("[");
    		str.concat(e.element.toString());
    		str.concat("]");
    		e = e.next;
    	}

    	str.concat("]");

    	return str;
    }
}