/*
 * QUSOFT MICROSYSTMES
 * Moka
 * Copyright 2004 Frdric Brown
 */

/*
 *  This file is part of Moka API.
 *
 *  Moka API is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Moka API is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Moka API; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package moka.lang;

import moka.lang.Object;
import moka.lang.String;

/**
 * The UShort class wraps a value of the primitive type <code>ushort</code>
 * in an object. An object of type <code>UShort</code> contains a
 * single field whose type is <code>ushort</code>.
 * <p>
 * In addition, this class provides several methods for converting
 * an <code>ushort</code> to a <code>String</code> and a
 * <code>String</code> to an <code>ushort</code>, as well as other
 * constants and methods useful when dealing with an
 * <code>ushort</code>.
 *
 * @author  Frdric Brown
 * @version 1.0, 2004-03-12
 * @since   MDK2.2
 */
public class UShort {

	/**
	* The smallest value of type <code>short</code>. The constant
	* value of this field is <tt>0</tt>.
	*/
	public static final ushort   MIN_VALUE = 0;

	/**
	* The largest value of type <code>short</code>. The constant
	* value of this field is <tt>32767</tt>.
	*/
	public static final ushort   MAX_VALUE = 65535;

	/**
	* The value of the UShort.
	*/
	private ushort value;

    /**
     * Constructs a newly allocated <code>UShort</code> object that
     * represents the primitive <code>ushort</code> argument.
     *
     * @param   value   the value to be represented by the <code>Short</code>.
     */
    public UShort(ushort value) {
		this.value = value;
    }

    /**
     * Constructs a newly allocated <code>UShort</code> object that
     * represents the value represented by the string. The string is
     * converted to an <tt>ushort</tt> in exactly the manner used by the
     * <tt>parseUshort</tt> method.
     *
     * @param      s   the <code>String</code> to be converted to an
     *                 <code>UShort</code>.
     */
    public UShort(String s) {
		this.value = UShort.parseShort(s);
    }

    /**
     * Returns a <code>String</code> object representing this
     * <code>UShort</code>'s value. The value is converted to
     * decimal representation and returned as a string, exactly as if
     * the ushort value were given as an argument to the valueOf method
     * of String.
     *
     * @return  a string representation of the value of this object.
     */
    public String toString() {
    	return String.valueOf(this.value);
    }

    /**
     * Compares this object to the specified object.  The result is
     * <code>true</code> if and only if the argument is not
     * <code>null</code> and is an <code>UShort</code> object that
     * contains the same <code>ushort</code> value as this object.
     *
     * @param   obj   the object to compare with.
     * @return  <code>true</code> if the objects are the same;
     *          <code>false</code> otherwise.
     */
    public boolean equals(Object obj) {
		if (obj instanceof Short) {
			return this.value == ((Short)obj).value;
		}
		return false;
    }

    /**
     * Returns the value of this <code>UShort</code> as an
     * <code>ushort</code>.
     */
    public ushort ushortValue() {
		return this.value;
    }

    /**
     * Returns the value of this <code>UShort</code> as an
     * <code>short</code>.
     */
    public short shortValue() {
		return (short)this.value;
    }

    /**
     * Parses the string argument as a decimal integer. The
     * characters in the string must all be decimal digits. The resulting
     * ushort value is returned.  The string argument is finalized before the method
     * returns.
     *
     * @param s	   a <code>String</code> containing the <code>ushort</code>
     *             representation to be parsed
     * @return     the ushort value represented by the argument in decimal.
     */
    public native static ushort parseShort(String s) {
    	unsigned short int value = atoi(s->value);

    	s->finalize_(s);

    	return value;
    }

    /**
     * Returns the value of this UShort as an <code>int</code>.
     * This may involve rounding or truncation.
     *
     * @return  the numeric value represented by this object after conversion
     *          to type <code>int</code>.
     */
    public int intValue () {
    	return (int)this.value;
    }

    /**
     * Returns the value of this UShort as a <code>long</code>.
     * This may involve rounding or truncation.
     *
     * @return  the numeric value represented by this object after conversion
     *          to type <code>long</code>.
     */
    public long longValue () {
    	return (long)this.value;
    }

    /**
     * Returns the value of this UShort as a <code>double</code>.
     * This may involve rounding.
     *
     * @return  the numeric value represented by this object after conversion
     *          to type <code>double</code>.
     */
    public double doubleValue () {
    	return (double)this.value;
    }

    /**
     * Returns the value of this UShort as a <code>byte</code>.
     * This may involve rounding or truncation.
     *
     * @return  the numeric value represented by this object after conversion
     *          to type <code>byte</code>.
     */
    public byte byteValue () {
    	return (byte)this.value;
    }

    /**
     * Checks if the specified string is a decimal representation
     * of a ushort integer.
     *
     * @param str The string to verify
     * @return true if the specified string is a decimal representation of a ushort integer, false otherwize.
     */
    public static boolean isUshort (String str) {
    	short i;
    	short len = native ( strlen(str->value) );

    	if (len > 5) {
    		return false;
    	}

    	for (i = 0; i < len; i++) {
    		if (Character.isDigit((char)str.value[i])) {
    			continue;
    		}
    		return false;
    	}

    	return true;
    }
}