/*
 * QUSOFT MICROSYSTMES
 * Moka
 * Copyright 2004 Frdric Brown
 */

/*
 *  This file is part of Moka API.
 *
 *  Moka API is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Moka API is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Moka API; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package moka.lang;

import moka.lang.Object;
import moka.lang.String;
import moka.lang.Character;

/**
 * The ULong class wraps a value of the primitive type <code>ulong</code>
 * in an object. An object of type <code>ulong</code> contains a
 * single field whose type is <code>ulong</code>.
 * <p>
 * In addition, this class provides several methods for converting
 * an <code>ulong</code> to a <code>String</code> and a
 * <code>String</code> to an <code>ulong</code>, as well as other
 * constants and methods useful when dealing with an
 * <code>ulong</code>.
 * <p>
 *
 * @author  Frdric Brown
 * @version 1.0, 2004-03-12
 * @since   MDK2.2
 */
public class ULong {

    /**
     * A constant holding the minimum value a <code>ulong</code> can
     * have, 0.
     */
	public static final ulong   MIN_VALUE = 0x0000000000000000;

    /**
     * A constant holding the maximum value a <code>ulong</code> can
     * have, 2<sup>64</sup>.
     */
	public static final ulong   MAX_VALUE = 0xffffffffffffffff;

	/**
	* The value of the ULong.
	*/
	private ulong value;

    /**
     * Constructs a newly allocated <code>ULong</code> object that
     * represents the primitive <code>ulong</code> argument.
     *
     * @param   value   the value to be represented by the <code>long</code>.
     */
    public ULong(ulong value) {
		this.value = value;
    }

    /**
     * Constructs a newly allocated <code>ULong</code> object that
     * represents the value represented by the string. The string is
     * converted to an <tt>ulong</tt> in exactly the manner used by the
     * <tt>parseUlong</tt> method.
     *
     * @param      s   the <code>String</code> to be converted to an
     *                 <code>ULong</code>.
     */
    public ULong(String s) {
		this.value = ULong.parseUlong(s);
    }

    /**
     * Returns a <code>String</code> object representing this
     * <code>ULong</code>'s value. The value is converted to signed
     * decimal representation and returned as a string, exactly as if
     * the ulong value were given as an argument to the valueOf method
     * of String.
     *
     * @return  a string representation of the value of this object.
     */
    public String toString() {
    	return String.valueOf(this.value);
    }

    /**
     * Compares this object to the specified object.  The result is
     * <code>true</code> if and only if the argument is not
     * <code>null</code> and is an <code>ULong</code> object that
     * contains the same <code>ulong</code> value as this object.
     *
     * @param   obj   the object to compare with.
     * @return  <code>true</code> if the objects are the same;
     *          <code>false</code> otherwise.
     */
    public boolean equals(Object obj) {
		if (obj instanceof ULong) {
				return this.value == ((ULong)obj).value;
		}
		return false;
    }

    /**
     * Returns the value of this <code>ULong</code> as an
     * <code>ulong</code>.
     */
    public ulong ulongValue() {
		return this.value;
    }

    /**
     * Returns the value of this <code>ULong</code> as an
     * <code>long</code>.
     */
    public long longValue() {
		return (long)this.value;
    }

    /**
     * Parses the string argument as a unsigned decimal ulong. The
     * characters in the string must all be decimal digits. The resulting
     * integer value is returned.  The string argument is finalized before the method
     * returns.
     *
     * @param s	   a <code>String</code> containing the <code>ulong</code>
     *             representation to be parsed
     * @return     the ulong value represented by the argument in decimal.
     */
    public native static long parseUlong(String s) {
    	unsigned long long int value = 0;
		unsigned long long int m = 1;
		long int pos = 0;
		long int l;
		long int i;

		l = strlen(s->value) - 1;

		for (i = 0; i < l; i++) {
			m *= 10;
		}

		while (m >= 10) {
			value += (s->value[pos++] - '0') * m;
			 m /= 10;
		}

		value += s->value[pos] - '0';

		s->finalize_(s);

		return value;
    }

    /**
     * Returns the value of this ULong as an <code>int</code>.
     * This may involve rounding or truncation.
     *
     * @return  the numeric value represented by this object after conversion
     *          to type <code>int</code>.
     */
    public int intValue () {
    	return (int)this.value;
    }

    /**
     * Returns the value of this ULong as a <code>double</code>.
     * This may involve rounding.
     *
     * @return  the numeric value represented by this object after conversion
     *          to type <code>double</code>.
     */
    public double doubleValue () {
    	return (double)((int)this.value);
    }

    /**
     * Returns the value of this ULong as a <code>byte</code>.
     * This may involve rounding or truncation.
     *
     * @return  the numeric value represented by this object after conversion
     *          to type <code>byte</code>.
     */
    public byte byteValue () {
    	return (byte)this.value;
    }

    /**
     * Returns the value of this ULong as a <code>short</code>.
     * This may involve rounding or truncation.
     *
     * @return  the numeric value represented by this object after conversion
     *          to type <code>short</code>.
     */
    public short shortValue() {
    	return (short)this.value;
    }

    /**
     * Checks if the specified string is a decimal representation
     * of a ulong integer.
     *
     * @param str The string to verify
     * @return true if the specified string is a decimal representation of a ulong integer, false otherwize.
     */
    public static boolean isUlong (String str) {
    	short i;
    	short len = native ( strlen(str->value) );

    	if (len > 20) {
    		return false;
    	}

    	for (i = 0; i < len; i++) {
    		if (Character.isDigit((char)str.value[i])) {
    			continue;
    		}
    		return false;
    	}

    	return true;
    }
}