/*
 * QUSOFT MICROSYSTMES
 * Moka
 * Copyright 2004 Frdric Brown
 */

/*
 *  This file is part of Moka API.
 *
 *  Moka API is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Moka API is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Moka API; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package moka.lang;

import moka.lang.Object;
import moka.lang.String;

/**
 * The UByte class wraps a value of the primitive type <code>ubyte</code>
 * in an object. An object of type <code>UByte</code> contains a
 * single field whose type is <code>ubyte</code>.
 * <p>
 * In addition, this class provides several methods for converting
 * an <code>ubyte</code> to a <code>String</code> and a
 * <code>String</code> to an <code>ubyte</code>, as well as other
 * constants and methods useful when dealing with an
 * <code>ubyte</code>.
 *
 * @author  Frdric Brown
 * @version 1.0, 2004-03-13
 * @since   MDK2.2
 */
public class UByte {

	/**
	* The smallest value of type <code>byte</code>. The constant
	* value of this field is <tt>0</tt>.
	*/
	public static final ubyte   MIN_VALUE = 0;

	/**
	* The largest value of type <code>byte</code>. The constant
	* value of this field is <tt>127</tt>.
	*/
	public static final ubyte   MAX_VALUE = 256;

	/**
	* The value of the UByte.
	*/
	private ubyte value;

    /**
     * Constructs a newly allocated <code>UByte</code> object that
     * represents the primitive <code>ubyte</code> argument.
     *
     * @param   value   the value to be represented by the <code>Byte</code>.
     */
    public UByte(ubyte value) {
		this.value = value;
    }

    /**
     * Constructs a newly allocated <code>UByte</code> object that
     * represents the value represented by the string. The string is
     * converted to an <tt>ubyte</tt> in exactly the manner used by the
     * <tt>parseUbyte</tt> method.
     *
     * @param      s   the <code>String</code> to be converted to an
     *                 <code>UByte</code>.
     */
    public UByte(String s) {
		this.value = UByte.parseUbyte(s);
    }

    /**
     * Returns a <code>String</code> object representing this
     * <code>UByte</code>'s value. The value is converted to signed
     * decimal representation and returned as a string.
     *
     * @return  a string representation of the value of this object.
     */
    public native String toString() {
		char vec[4];

		sprintf(vec, "%i", (short int)this->value);

		return TString_char_p_BOOL(strcpy(malloc(strlen(vec) + 1), vec), FALSE);
    }

    /**
     * Compares this object to the specified object.  The result is
     * <code>true</code> if and only if the argument is not
     * <code>null</code> and is an <code>UByte</code> object that
     * contains the same <code>ubyte</code> value as this object.
     *
     * @param   obj   the object to compare with.
     * @return  <code>true</code> if the objects are the same;
     *          <code>false</code> otherwise.
     */
    public boolean equals(Object obj) {
		if (obj instanceof UByte) {
			return this.value == ((UByte)obj).value;
		}
		return false;
    }

    /**
     * Returns the value of this <code>UByte</code> as an
     * <code>byte</code>.
     */
    public ubyte ubyteValue() {
		return this.value;
    }

    /**
     * Returns the value of this <code>UByte</code> as an
     * <code>byte</code>.
     */
    public byte byteValue() {
		return (byte)this.value;
    }

    /**
     * Parses the string argument as a decimal integer. The
     * characters in the string must all be decimal digits. The resulting
     * ubyte value is returned.  The string argument is finalized before the method
     * returns.
     *
     * @param s	   a <code>String</code> containing the <code>ubyte</code>
     *             representation to be parsed
     * @return     the byte value represented by the argument in decimal.
     */
    public native static ubyte parseUbyte(String s) {
    	unsigned byte value = (byte)atoi(s->value);

    	s->finalize_(s);

    	return value;
    }

    /**
     * Returns the value of this UByte as an <code>int</code>.
     * This may involve rounding or truncation.
     *
     * @return  the numeric value represented by this object after conversion
     *          to type <code>int</code>.
     */
    public int intValue () {
    	return (int)this.value;
    }

    /**
     * Returns the value of this UByte as a <code>long</code>.
     * This may involve rounding or truncation.
     *
     * @return  the numeric value represented by this object after conversion
     *          to type <code>long</code>.
     */
    public long longValue () {
    	return (long)this.value;
    }

    /**
     * Returns the value of this UByte as a <code>double</code>.
     * This may involve rounding.
     *
     * @return  the numeric value represented by this object after conversion
     *          to type <code>double</code>.
     */
    public double doubleValue () {
    	return (double)this.value;
    }

    /**
     * Returns the value of this UByte as a <code>short</code>.
     * This may involve rounding or truncation.
     *
     * @return  the numeric value represented by this object after conversion
     *          to type <code>short</code>.
     */
    public short shortValue() {
    	return (short)this.value;
    }

    /**
     * Checks if the specified string is a decimal representation
     * of a ubyte.
     *
     * @param str The string to verify
     * @return true if the specified string is a decimal representation of a ubyte, false otherwize.
     */
    public static boolean isUbyte (String str) {
    	short i;
    	short len = native ( strlen(str->value) );

    	if (len > 3) {
    		return false;
    	}

    	for (i = 0; i < len; i++) {
    		if (Character.isDigit((char)str.value[i])) {
    			continue;
    		}
    		return false;
    	}

    	return true;
    }
}