/*
 * QUSOFT MICROSYSTMES
 * Moka
 * Copyright 2002 Frdric Brown
 */

/*
 *  This file is part of Moka API.
 *
 *  Moka API is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Moka API is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Moka API; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package moka.lang;

import moka.lang.Object;
import moka.lang.String;

/**
 * The <code>System</code> class contains several useful class fields
 * and methods. It cannot be instantiated.
 * <p>
 * Among the facilities provided by the <code>System</code> class
 * are input and output functions and most of the calc-dependent and AMS-dependent constants.
 *
 * @author  Frdric Brown
 * @version 1.22, 2004-11-22
 * @since   MDK1.0a
 */
 public final class System {

 	/** Indicate that the AMS version is AMS_1xx.*/
 	public static final short AMS_1 = 1;
 	/** Indicate that the AMS version is AMS_2xx.*/
 	public static final short AMS_2 = 2;

 	/** Indicate that the calculator model is TI89.*/
 	public static final short CALCULATOR_TI89 = 0;
 	/** Indicate that the calculator model is TI92PLUS.*/
 	public static final short CALCULATOR_TI92PLUS = 1;
 	/** Indicate that the calculator model is V200.*/
 	public static final short CALCULATOR_V200 = 3;

 	/** Number of entries in TIOS jump table: may be used for determining actual AMS version (for example, it is 972 on AMS 1.05 and 1463 on AMS 2.03).*/
 	public static short ENTRIES = native.TIOS_entries ;
 	/**The AMS version (AMS_1xx or AMS_2xx)*/
 	public static short AMS = ( ( System.ENTRIES == 972 ) ? System.AMS_1 : System.AMS_2 ) ;
 	/**The calculator model (TI-89, TI-92 or V200)*/
 	public static short CALCULATOR ;
 	/** Base address of the ROM (0x200000 on TI-89 or V200, 0x400000 on TI-92 Plus).*/
 	public static void* ROM_BASE = native.ROM_base ;


	/** The number added to the key code if the Diamond key is pressed with it.*/
	public static short KEY_DIAMOND = native.KEY_DIAMOND ;
	/** The number added to the key code if the Shift key is pressed with it.*/
	public static short KEY_SHIFT = native.KEY_SHIFT ;

	/** Key code for the down arrow key.*/
	public static short  KEY_DOWN = native.KEY_DOWN ;
	/** Key code for simultaneous pressing of the down and left arrow keys.*/
	public static short KEY_DOWNLEFT = native.KEY_DOWNLEFT ;
	/** Key code for the left arrow key.*/
	public static short  KEY_LEFT = native.KEY_LEFT ;
	/** Key code for the right arrow key.*/
	public static short KEY_RIGHT = native.KEY_RIGHT ;
	/** Key code for the up arrow key.*/
	public static short KEY_UP = native.KEY_UP ;
	/** Key code for simultaneous pressing of the up and left arrow keys.*/
	public static short KEY_UPLEFT = native( ( unsigned short ) System_KEY_UP ) | ( ( unsigned short )System_KEY_LEFT) ;
	/** Key code for simultaneous pressing on up and right arrow keys.*/
	public static short KEY_UPRIGHT = native.KEY_UPRIGHT ;
	/** Key code for simultaneous pressing on up and right arrow keys.*/
	public static short KEY_DOWNRIGHT = native( ( unsigned short ) System_KEY_DOWN ) | ( ( unsigned short )System_KEY_RIGHT) ;

	/** The number of arguments passed to the program. Should be only read. */
	public static short argCount = native.ArgCount();

	private System() {}

 	/** The argument pointer.*/
 	private static ESI argPtr = null;

	/**
	 * Return an newly allocated array of <code>System.argCount</code> <code>String</code> instances built uppon the arguments passed to the program.
	 *
	 * @return an newly allocated array of <code>System.argCount</code> <code>String</code> instances built uppon the arguments passed to the program
	 * @since MDK2.21
	 */
	private static String[] getMainArgs() {
		short i;
        String[] args;

        if (System_argCount) {
        	args = native.malloc(sizeof(TString *) * System_argCount) ;
        }
        else {
        	args = NULL;
        }

        for (i = 0; i < System.argCount; i++) {
                args[i] = System.getString() ;
        }

        return args;
	}

	/**
	 * If the status flags indicate that a help message is being displayed, this function redraws the status line, effectively removing the message.
	 *
	 * @return true if the drawing is performed, false otherwize
	 * @since MDK1.01a
	 */
	 /*Alias implemented for stEraseHelp ().*/
	 public abstract native static boolean stEraseHelp () ;
	/*public native static boolean stEraseHelp () ; {
		return ST_eraseHelp();
	}*/

	/**
	 * Displays a message in the status line then finalizes the String argument.
	 *
	 * @param msg The message to display in the status line
	 * @since MDK1.01a
	 */
	public native static void setStHelpMsg (String msg) {
		ST_helpMsg (msg->value);
		msg->finalize_(msg);
	}

	/**
	 * Returns the availlable RAM memory in bytes.
	 * To be accurate, this method call the garbage collection
	 * of the RAM memory then determines the total amount
	 * of free bytes.
	 *
	 * @return The availlable RAM memory in bytes
	 * @since MDK1.01a
	 */
	public native static int getAvailRAM () {
		HeapCompress();
		return HeapAvail();
	}

	/**
	 * Calls the garbage collection of the RAM memory.
	 *
	 * @since MDK1.01a
	 */
	/*Alias implemented for gcRAM ().*/
	public abstract native static void gcRAM () ;
	/*public native static void gcRAM () {
		HeapCompress();
	}*/

	/**
	 * Resets the calculator.
	 *
	 * @since MDK1.01a
	 */
	/*Alias implemented for reset ().*/
	public abstract native static void reset () ;
	/*public native static void reset () {
		OSReset();
	}*/

	/**
	 * Increases the display contrast.
	 *
	 * @since MDK1.01a
	 */
	/*Alias implemented for contrastUp ().*/
	public abstract native static void contrastUp () ;
	/*public native static void contrastUp () {
		OSContrastUp();
	}*/

	/**
	 * Decreases the display contrast.
	 *
	 * @since MDK1.01a
	 */
	/*Alias implemented for contrastDn ().*/
	public abstract native static void contrastDn () ;
	/*public native static void contrastDn () {
		OSContrastDn();
	}*/

	/**
	 * Turns the calculator in "low-power" state until the next interrupt occurs then returns.
	 *
	 * @since MDK1.01a
	 */
	/*Alias implemented for idle ().*/
	public abstract native static void idle () ;
	/*public native static void idle () {
		idle();
	}*/

	/**
	 * Turns off the calculator.
	 *
	 * @since MDK1.01a
	 */
	/*Alias implemented for off ().*/
	public abstract native static void off () ;
	/*public native static void off () {
		off();
	}*/

	/**
	 * Terminates the current program returning 0 as return value.
	 * Usualy the argument serves as a status code; by convention,
	 * a nonzero status code indicates abnormal termination.
	 */
	public static void exit () {
		System.exit(0);
	}

	/**
	 * Clears the expression stack.
	 */
	public static native void clearEstack() {
		TRY
			while (GetArgType (top_estack) != END_TAG)
  				top_estack = next_expression_index (top_estack);
			top_estack--;
		ONERR
		ENDTRY
	}

	/**
	 * Terminates the current program returning a short as return value.
	 *
	 * @param ret The value to return
	 */
	public native static void exit (short ret) {
		System_clearEstack_();

		push_shortint(ret);
		exit(0);
	}

	/**
	 * Terminates the current program returning an int as return value.
	 *
	 * @param ret The value to return
	 */
	public native static void exit (int ret) {
		System_clearEstack_();

		push_longint(ret);
		exit(0);
	}

	/**
	 * Terminates the current program returning a long as return value.
	 *
	 * @param ret The value to return
	 */
	public native static void exit (long ret) {
		System_clearEstack_();

		push_longlongint(ret);
		exit(0);
	}

	/**
	 * Terminates the current program returning a double as return value.
	 *
	 * @param ret The value to return
	 */
	public native static void exit (double ret) {
		System_clearEstack_();

		push_Float(ret);
		exit(0);
	}

	/**
	 * Terminates the current program returning a String as return value.
	 *
	 * @param ret The value to return
	 */
	public native static void exit (String ret) {
		System_clearEstack_();

		char* output_string = malloc(strlen(ret->value) + 2);
		//put the zero at the beginning of the string
		output_string[0] = 0;
		//copy the string, including the terminating zero, to the output string
		strcpy(output_string + 1, ret->value);

		//put the string, with all spaces intact, on the expression stack
		push_expr_quantum(output_string + strlen(output_string + 1) + 1, STR_TAG);

		//always make sure to free your memory
		free(output_string);

		ret->finalize_(ret);

		exit(0);
	}

	/**
	 * Initializes the argument pointer or do nothing if it is already done.
	 */
	private native static void initArgPtr () {
		if (! System_argPtr) {
			InitArgPtr(System_argPtr);
		}
	}

	/**
	 * Returns the next argument as a short.
	 *
	 * @return the next argument as a short
	 */
	public static short getShort() {
		return (short)System.getInt();
	}

	/**
	 * Returns the next argument as a int.
	 *
	 * @return the next argument as a int
	 */
	public static int getInt() {
		int arg;
		System.initArgPtr();

		native {
			switch (GetArgType(System_argPtr)) {
				case POSINT_TAG: {
					arg = GetIntArg(System_argPtr);
					break;
				}
				case NEGINT_TAG: {
					arg = -1 * GetIntArg(System_argPtr);
					break;
				}
			}
		}

		return arg;
	}

	/**
	 * Returns the next argument as a long.
	 *
	 * @return the next argument as a long
	 */
	public static long getLong() {
		long arg;
		System.initArgPtr();

		native {
			switch (GetArgType(System_argPtr)) {
				case POSINT_TAG: {
					arg = GetLongLongArg(System_argPtr);
					break;
				}
				case NEGINT_TAG: {
					arg = -1 * GetLongLongArg(System_argPtr);
					break;
				}
			}
		}

		return arg;
	}

	/**
	 * Returns the next argument as a double.
	 *
	 * @return the next argument as a double
	 */
	public static double getDouble() {
		double arg;
		System.initArgPtr();

		native {
			switch (GetArgType(System_argPtr)) {
				case POSINT_TAG: {
					arg = GetIntArg(System_argPtr);
					break;
				}
				case NEGINT_TAG: {
					arg = -1 * GetIntArg(System_argPtr);
					break;
				}
				case FLOAT_TAG: {
					arg = GetFloatArg(System_argPtr);
					break;
				}
				case POSFRAC_TAG: {
					arg = GetIntArg(System_argPtr);
					System_argPtr++;
					arg /= GetIntArg(System_argPtr);
					break;
				}
				case NEGFRAC_TAG: {
					arg = -1 * GetIntArg(System_argPtr);
					System_argPtr++;
					arg /= GetIntArg(System_argPtr);
					break;
				}
			}
		}

		return arg;
	}

	/**
	 * Returns the next argument as a String.
	 *
	 * @return the next argument as a String
	 */
	public static String getString() {
		double db;
		String arg;
		System.initArgPtr();

		native {
			switch (GetArgType(System_argPtr)) {
				case POSINT_TAG: {
					arg = String_valueOf_long_int((long int)GetIntArg(System_argPtr));
					break;
				}
				case NEGINT_TAG: {
					arg = String_valueOf_long_int((long int)(-1 * GetIntArg(System_argPtr)));
					break;
				}
				case FLOAT_TAG: {
					arg = String_valueOf_double(GetFloatArg(System_argPtr));
					break;
				}
				case POSFRAC_TAG: {
					db = GetIntArg(System_argPtr);
					System_argPtr++;
					db /= GetIntArg(System_argPtr);
					arg = String_valueOf_double(db);
					break;
				}
				case NEGFRAC_TAG: {
					db = -1 * GetIntArg(System_argPtr);
					System_argPtr++;
					db /= GetIntArg(System_argPtr);
					arg = String_valueOf_double(db);
					break;
				}
				case STR_TAG: {
					arg = TString_char_p(GetStrnArg(System_argPtr));
					break;
				}
			}
		}

		return arg;
	}

	/**
	 * Clears the screen and moves the print position to the first
	 * character of the first line.
	 */
	/*Alias implemented for clrscr ().*/
	public abstract native static void clrscr () ;
	/*public native static void clrscr () {
		clrscr();
	}*/

    /**
     * Terminates the line.
     */
    /*Alias implemented for println ().*/
    public abstract static void println () ;
    /*public static void println () {
    	native {
    		printf("\n");
    	}
    }*/

    /**
     * Prints an object.  In fact, prints the string produced by the
     * <code>toString</code> method.
     *
     * @param      obj   The <code>Object</code> to be printed
     */
    public static void print (Object obj) {
    	String str = obj.toString();

    	native {
    		printf(str->value);
    	}

    	str.finalize();
    }

    /**
     * Prints an object and then terminates the line.  In fact, prints the
     * string produced by the <code>toString</code> method.
     *
     * @param      obj   The <code>Object</code> to be printed
     */
    public static void println (Object obj) {
    	System.print(obj);
    	native {
    		printf("\n");
    	}
    }

    /**
     * Prints a String then finalize the String.
     *
     * @param      s   The <code>String</code> to be printed
     */
    public static void print (String s) {
    	native {
    		printf(s->value);
    	}
    	s.finalize();
    }

    /**
     * Prints a String and then terminates the line and finalize the String.
     *
     * @param      s   The <code>String</code> to be printed
     */
    public static void println (String s) {
    	System.print(s);
    	native {
    		printf("\n");
    	}
    }

    /**
     * Prints an ANSI string.
     *
     * @param      s   The <code>ANSI string</code> to be printed
     */
    public abstract static void print (char[] s) ;

    /**
     * Prints an ANSI string and then terminates the line.
     *
     * @param      s   The <code>ANSI string</code> to be printed
     */
    public abstract static void println (char[] s) ;

    /**
     * Prints a char.
     *
     * @param      c   The <code>char</code> to be printed
     */
    /*Alias implemented for print (char c).*/
    public abstract static void print (char c) ;
    /*public static void print (char c) {

    	native {
    		printf("%c", c);
    	}
    }*/

    /**
     * Prints a char and then terminates the line.
     *
     * @param      c   The <code>char</code> to be printed
     */
    public abstract static void println (char c) ;
    /*public static void println (char c) {
    	System.print(c);
    	native {
    		printf("\n");
    	}
    }*/

    /**
     * Prints an unsigned byte.
     *
     * @param      b   The <code>ubyte</code> to be printed
     */
    public abstract static void print (ubyte b) ;

    /**
     * Prints an unsigned byte and then terminates the line.
     *
     * @param      s   The <code>ubyte</code> to be printed
     */
    public abstract static void println (ubyte b) ;

    /**
     * Prints a byte.
     *
     * @param      b   The <code>byte</code> to be printed
     */
    /*Alias implemented for print (byte b).*/
    public abstract static void print (byte b) ;
    /*public static void print (byte b) {

    	native {
    		printf("%i", (short int)b);
    	}
    }*/

    /**
     * Prints a byte and then terminates the line.
     *
     * @param      s   The <code>byte</code> to be printed
     */
    /*Alias implemented for println (byte b).*/
    public abstract static void println (byte b) ;
    /*public static void println (byte b) {
    	System.print(b);
    	native {
    		printf("\n");
    	}
    }*/

    /**
     * Prints an unsigned short.
     *
     * @param      s   The <code>ushort</code> to be printed
     */
    /*Alias implemented for print (short s).*/
    public abstract static void print (ushort s) ;

    /**
     * Prints an unsigned short and then terminates the line.
     *
     * @param      s   The <code>ushort</code> to be printed
     */
    public abstract static void println (ushort s) ;

    /**
     * Prints a short.
     *
     * @param      s   The <code>short</code> to be printed
     */
    /*Alias implemented for print (short s).*/
    public abstract static void print (short s) ;
    /*public static void print (short s) {

    	native {
    		printf("%i", s);
    	}
    }*/

    /**
     * Prints a short and then terminates the line.
     *
     * @param      s   The <code>short</code> to be printed
     */
    public abstract static void println (short s) ;
    /*public static void println (short s) {
    	System.print(s);
    	native {
    		printf("\n");
    	}
    }*/

    /**
     * Prints an unsigned int.
     *
     * @param      i   The <code>uint</code> to be printed
     */
    /*Alias implemented for print (int i).*/
    public abstract static void print (uint i) ;

    /**
     * Prints an unsigned int and then terminates the line.
     *
     * @param      i   The <code>uint</code> to be printed
     */
    public abstract static void println (uint i) ;

    /**
     * Prints an int.
     *
     * @param      i   The <code>int</code> to be printed
     */
    /*Alias implemented for print (int i).*/
    public abstract static void print (int i) ;
    /*public static void print (int i) {

    	native {
    		printf("%ld", i);
    	}
    }*/

    /**
     * Prints an int and then terminates the line.
     *
     * @param      i   The <code>int</code> to be printed
     */
    public abstract static void println (int i) ;
    /*public static void println (int i) {
    	System.print(i);
    	native {
    		printf("\n");
    	}
    }*/

    /**
     * Prints a long.
     *
     * @param      l   The <code>long</code> to be printed
     */
    public static void print (long l) {
    	String str = String.valueOf(l);

    	native {
    		printf(str->value);
    	}

    	str.finalize();
    }

    /**
     * Prints a long and then terminates the line.
     *
     * @param      l   The <code>long</code> to be printed
     */
    public static void println (long l) {
    	System.print(l);
    	native {
    		printf("\n");
    	}
    }

    /**
     * Prints an unsigned long.
     *
     * @param      l   The <code>ulong</code> to be printed
     */
    public static void print (ulong l) {
    	String str = String.valueOf(l);

    	native {
    		printf(str->value);
    	}

    	str.finalize();
    }

    /**
     * Prints an unsigned long and then terminates the line.
     *
     * @param      l   The <code>ulong</code> to be printed
     */
    public static void println (ulong l) {
    	System.print(l);
    	native {
    		printf("\n");
    	}
    }

    /**
     * Prints a double.
     *
     * @param      d   The <code>double</code> to be printed
     */
    public static void print (double d) {
    	String str = String.valueOf(d);

    	native {
    		printf(str->value);
    	}

    	str.finalize();
    }

    /**
     * Prints a double and then terminates the line.
     *
     * @param      d   The <code>double</code> to be printed
     */
    public static void println (double d) {
    	System.print(d);
    	native {
    		printf("\n");
    	}
    }

    /**
     * Read a character from the keyboard.
     *
     * @return     The character read from the keyboard.
     */
    /*Alias implemented for print (int i).*/
    public abstract native static char read () ;
    /*public native static char read () {
    	return (char)ngetchx();
    }*/

    /**
     * Read a line of text.  A line is considered to be terminated by
     * a carriage return or when the 255 character buffer is full.
     *
     * @return     A String containing the contents of the line, not including
     *             any line-termination characters.
     */
    public static String readLine () {
    	return System.readLine(true);
    }

    /**
     * Read a line of text.  A line is considered to be terminated by
     * a carriage return or when the 255 character buffer is full.
     *
     * @param	   echo if true, echos the keys pressed on the screen.
     * @return     A String containing the contents of the line, not including
     *             any line-termination characters.
     */
    public static String readLine (boolean echo) {
    	char buffer[256];
    	BYTE ind = 0;
    	short key = 0;
    	SCR_STATE ss;

		native {
			SaveScrState (&ss);
		}

    	while (ind < 255) {
    		//key = ngetchx();
    		key = native ( GKeyIn(NULL, 0) );
    		if (key != KEY_ENTER) {
    			switch(key) {
    				case KEY_BACKSPACE: {
    					//if i != 0, decrement ind
    					if (ind) {
    						buffer[ind] = '\0';
    						buffer[--ind] = ' ';
    					}
    					break;
    				}
    				case KEY_CLEAR: {
    					native {
    						memset(buffer, ' ', ind);
    					}
    					buffer[ind] = '\0';
    					ind = 0;
    					break;
    				}
    				default: {
	    				buffer[ind++] = key;
	    				buffer[ind] = '\0';
	    			}
	    		}
				if (echo) {
					native {
						MoveTo (ss.CurX, ss.CurY);
						printf(buffer);
					}
				}
				buffer[ind] = '\0';
    		}
    		else {
    			break;
    		}
    	}

    	return new String(Character.copyValueOf((char*)buffer), false);
    }

    /**
     * Executes the specified string command.  The string command object
     * is finalized before this method returns.
     *
     * @param      command   a specified system command.
     */
    public native static void exec (String command) {
		HANDLE handle;

		push_parse_text (command->value);
		handle = HS_popEStack ();

		TRY
    		NG_execute (handle, FALSE);
  		FINALLY
    		HeapFree (handle);
    		command->finalize_(command);
		ENDFINAL
    }

    /**
     * Sets the device's "APD" time. This is the time in 1/20th seconds where,
     * if no user interaction occurs with the device, it turns off.
     *
     * @param time the time in 1/20th seconds where, if no user interaction occurs with the device, it turns off.
     * @return the previous device's "APD" time.
     */
    public native static int setAPD (int time) {
    	long int prev = OSTimerCurVal (APD_TIMER);

    	OSFreeTimer (APD_TIMER);
		OSRegisterTimer (APD_TIMER, time);

    	return prev;
    }

    /**
     * Allocate a block of memory from the memory heap.
     * If not enough space exists for the new block or the argument size is zero, it returns null.
     * The memory allocated with this method can be freed by System.free method.
     * This method supports the garbage collection feature (if the main class of the project implements
     * Use_GarbageCollection) instead of the native C equivalent.
     *
     * @param      size   the size (in bytes) of the block.
     * @return     A pointer to a block of memory, or null if 0 is specified for size or if there is not enough memory.
     */
    /* Alias implemented */
    public abstract static void* malloc (int size);
    /*public native static void* malloc (int size) {
		return malloc(size);
    }*/

    /**
     * Deallocates a memory block previously allocated by a call of System.malloc.
     *
     * @param      ptr   pointer to the memory block.
     */
    /* Alias implemented */
    public abstract static void free (void* ptr);
    /*public native static void free (void* ptr) {
		free(ptr);
    }*/
 }