/*
 * QUSOFT MICROSYSTMES
 * Moka
 * Copyright 2002 Frdric Brown
 */

/*
 *  This file is part of Moka API.
 *
 *  Moka API is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Moka API is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Moka API; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package moka.lang;

import moka.lang.Object;
import moka.lang.String;

/**
 * The Short class wraps a value of the primitive type <code>short</code>
 * in an object. An object of type <code>Short</code> contains a
 * single field whose type is <code>short</code>.
 * <p>
 * In addition, this class provides several methods for converting
 * an <code>short</code> to a <code>String</code> and a
 * <code>String</code> to an <code>short</code>, as well as other
 * constants and methods useful when dealing with an
 * <code>short</code>.
 *
 * @author  Frdric Brown
 * @version 1.0, 2002-10-30
 * @since   MDK1.0a
 */
public class Short {

	/**
	* The smallest value of type <code>short</code>. The constant
	* value of this field is <tt>-32768</tt>.
	*/
	public static final short   MIN_VALUE = -32768;

	/**
	* The largest value of type <code>short</code>. The constant
	* value of this field is <tt>32767</tt>.
	*/
	public static final short   MAX_VALUE = 32767;

	/**
	* The value of the Short.
	*/
	private short value;

    /**
     * Constructs a newly allocated <code>Short</code> object that
     * represents the primitive <code>short</code> argument.
     *
     * @param   value   the value to be represented by the <code>Short</code>.
     */
    public Short(short value) {
		this.value = value;
    }

    /**
     * Constructs a newly allocated <code>Short</code> object that
     * represents the value represented by the string. The string is
     * converted to an <tt>short</tt> in exactly the manner used by the
     * <tt>parseShort</tt> method.
     *
     * @param      s   the <code>String</code> to be converted to an
     *                 <code>Short</code>.
     */
    public Short(String s) {
		this.value = Short.parseShort(s);
    }

    /**
     * Returns a <code>String</code> object representing this
     * <code>Short</code>'s value. The value is converted to signed
     * decimal representation and returned as a string, exactly as if
     * the short value were given as an argument to the valueOf method
     * of String.
     *
     * @return  a string representation of the value of this object.
     */
    public String toString() {
    	return String.valueOf(this.value);
    }

    /**
     * Compares this object to the specified object.  The result is
     * <code>true</code> if and only if the argument is not
     * <code>null</code> and is an <code>Short</code> object that
     * contains the same <code>short</code> value as this object.
     *
     * @param   obj   the object to compare with.
     * @return  <code>true</code> if the objects are the same;
     *          <code>false</code> otherwise.
     */
    public boolean equals(Object obj) {
		if (obj instanceof Short) {
			return this.value == ((Short)obj).value;
		}
		return false;
    }

    /**
     * Returns the value of this <code>Short</code> as an
     * <code>short</code>.
     */
    public short shortValue() {
		return this.value;
    }

    /**
     * Parses the string argument as a signed decimal integer. The
     * characters in the string must all be decimal digits, except that
     * the first character may be an ASCII minus sign <code>'-'</code>
     * (<code>'&#92;u002D'</code>) to indicate a negative value. The resulting
     * short value is returned.  The string argument is finalized before the method
     * returns.
     *
     * @param s	   a <code>String</code> containing the <code>short</code>
     *             representation to be parsed
     * @return     the short value represented by the argument in decimal.
     */
    public native static short parseShort(String s) {
    	short int value = atoi(s->value);

    	s->finalize_(s);

    	return value;
    }

    /**
     * Returns the value of this Short as an <code>int</code>.
     * This may involve rounding or truncation.
     *
     * @return  the numeric value represented by this object after conversion
     *          to type <code>int</code>.
     */
    public int intValue () {
    	return (int)this.value;
    }

    /**
     * Returns the value of this Short as a <code>long</code>.
     * This may involve rounding or truncation.
     *
     * @return  the numeric value represented by this object after conversion
     *          to type <code>long</code>.
     */
    public long longValue () {
    	return (long)this.value;
    }

    /**
     * Returns the value of this Short as a <code>double</code>.
     * This may involve rounding.
     *
     * @return  the numeric value represented by this object after conversion
     *          to type <code>double</code>.
     */
    public double doubleValue () {
    	return (double)this.value;
    }

    /**
     * Returns the value of this Short as a <code>byte</code>.
     * This may involve rounding or truncation.
     *
     * @return  the numeric value represented by this object after conversion
     *          to type <code>byte</code>.
     */
    public byte byteValue () {
    	return (byte)this.value;
    }

    /**
     * Checks if the specified string is a decimal representation
     * of a short integer.
     *
     * @param str The string to verify
     * @return true if the specified string is a decimal representation of a short integer, false otherwize.
     */
    public static boolean isShort (String str) {
    	short i;
    	short len = native ( strlen(str->value) );

    	if (len > 6 || len > 5 && str.value[0] != '-') {
    		return false;
    	}

    	for (i = 0; i < len; i++) {
    		if (Character.isDigit((char)str.value[i]) || i == 0 && str.value[i] == '-') {
    			continue;
    		}
    		return false;
    	}

    	return true;
    }
}