/*
 * QUSOFT MICROSYSTMES
 * Moka
 * Copyright 2002 Frdric Brown
 */

/*
 *  This file is part of Moka API.
 *
 *  Moka API is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Moka API is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Moka API; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package moka.lang;

import moka.lang.Object;

/**
 * The class <code>Math</code> contains methods for performing basic
 * numeric operations such as the elementary exponential, logarithm,
 * square root, and trigonometric functions.
 *
 * @author  Frdric Brown
 * @version 1.0, 2003-03-19
 * @since   MDK1.01a
 */
public final abstract class Math {

    /**
     * The <code>double</code> value that is closer than any other to
     * <i>e</i>, the base of the natural logarithms.
     */
    public static final double E =  2.718281828459045;

    /**
     * The <code>double</code> value that is closer than any other to
     * <i>pi</i>, the ratio of the circumference of a circle to its
     * diameter.
     */
    public static final double PI = native.PI;

    /**
     * Returns the absolute value of a <code>byte</code> value.
     * If the argument is not negative, the argument is returned.
     * If the argument is negative, the negation of the argument is returned.
     *
     * @param   a   the argument whose absolute value is to be determined
     * @return  the absolute value of the argument.
     */
    public static byte abs (byte a) {
		return (a < 0) ? -a : a;
    }

    /**
     * Returns the absolute value of a <code>short</code> value.
     * If the argument is not negative, the argument is returned.
     * If the argument is negative, the negation of the argument is returned.
     *
     * @param   a   the argument whose absolute value is to be determined
     * @return  the absolute value of the argument.
     */
    public static short abs (short a) {
		return (a < 0) ? -a : a;
    }

    /**
     * Returns the absolute value of an <code>int</code> value.
     * If the argument is not negative, the argument is returned.
     * If the argument is negative, the negation of the argument is returned.
     *
     * @param   a   the argument whose absolute value is to be determined
     * @return  the absolute value of the argument.
     */
    public static int abs (int a) {
		return (a < 0) ? -a : a;
    }

    /**
     * Returns the absolute value of a <code>long</code> value.
     * If the argument is not negative, the argument is returned.
     * If the argument is negative, the negation of the argument is returned.
     *
     * @param   a   the argument whose absolute value is to be determined
     * @return  the absolute value of the argument.
     */
    public static long abs (long a) {
		return (a < 0) ? -a : a;
    }

    /**
     * Returns the absolute value of an <code>int</code> value.
     * If the argument is not negative, the argument is returned.
     * If the argument is negative, the negation of the argument is returned.
     *
     * @param   a   the argument whose absolute value is to be determined
     * @return  the absolute value of the argument.
     */
    public static double abs (double a) {
		return (a < 0) ? -a : a;
    }

    /**
     * Returns the arc cosine of an angle.
     *
     * @param   a   the value whose arc cosine is to be returned.
     * @return  the arc cosine of the argument.
     */
    /*Alias implemented for acos (double a).*/
    public abstract native static double acos (double a) ;
    /*public native static double acos (double a) {
		return acos(a);
    }*/

    /**
     * Returns the arc sine of an angle.
     *
     * @param   a   the value whose arc sine is to be returned.
     * @return  the arc sine of the argument.
     */
    /*Alias implemented for asin (double a).*/
    public abstract native static double asin (double a) ;
    /*public native static double asin (double a) {
		return asin(a);
    }*/

    /**
     * Returns the arc tangent of an angle.
     *
     * @param   a   the value whose arc tangent is to be returned.
     * @return  the arc tangent of the argument.
     */
    /*Alias implemented for atan (double a).*/
    public abstract native static double atan (double a) ;
    /*public native static double atan (double a) {
		return atan(a);
    }*/

    /**
     * Converts rectangular coordinates (<code>x</code>,&nbsp;<code>y</code>)
     * to polar (r,&nbsp;<i>theta</i>).
     * This method computes the phase <i>theta</i> by computing an arc tangent
     * of <code>y/x</code> in the range of -<i>pi</i> to <i>pi</i>.
     *
     * @param   y   the ordinate coordinate
     * @param   x   the abscissa coordinate
     * @return  the <i>theta</i> component of the point
     *          (<i>r</i>,&nbsp;<i>theta</i>)
     *          in polar coordinates that corresponds to the point
     *          (<i>x</i>,&nbsp;<i>y</i>) in Cartesian coordinates.
     */
    /*Alias implemented for atan2 (double y, double x).*/
    public abstract native static double atan2 (double y, double x) ;
    /*public native static double atan2 (double y, double x) {
		return atan2(x, y);
    }*/

    /**
     * Returns the smallest (closest to negative infinity)
     * <code>double</code> value that is not less than the argument and is
     * equal to a mathematical integer.
     *
     * @param   a   a value.
     * @return  the smallest (closest to negative infinity)
     *          floating-point value that is not less than the argument
     *          and is equal to a mathematical integer.
     */
    /*Alias implemented for ceil (double a).*/
    public abstract native static double ceil (double a) ;
    /*public native static double ceil (double a) {
		return ceil(a);
    }*/

    /**
     * Returns the trigonometric cosine of an angle.
     *
     * @param   a   an angle, in radians.
     * @return  the cosine of the argument.
     */
    /*Alias implemented for cos (double a).*/
    public abstract native static double cos (double a) ;
    /*public native static double cos (double a) {
		return cos(a);
    }*/

    /**
     * Returns Euler's number <i>e</i> raised to the power of a
     * <code>double</code> value.
     *
     * @param   a   the exponent to raise <i>e</i> to.
     * @return  the value <i>e</i><sup><code>a</code></sup>,
     *          where <i>e</i> is the base of the natural logarithms.
     */
    /*Alias implemented for exp (double a).*/
    public abstract native static double exp (double a) ;
    /*public native static double exp (double a) {
		return exp(a);
    }*/

    /**
     * Returns the largest (closest to positive infinity)
     * <code>double</code> value that is not greater than the argument and
     * is equal to a mathematical integer.
     *
     * @param   a   a value
     * @return  the largest (closest to positive infinity)
     *          floating-point value that is not greater than the argument
     *          and is equal to a mathematical integer.
     */
    /*Alias implemented for floor (double a).*/
    public abstract native static double floor (double a) ; 
    /*public native static double floor (double a) {
		return floor(a);
    }*/

    /**
     * Returns the natural logarithm (base <i>e</i>) of a <code>double</code>
     * value.
     *
     * @param   a   a number greater than <code>0.0</code>.
     * @return  the value ln&nbsp;<code>a</code>, the natural logarithm of
     *          <code>a</code>.
     */
    /*Alias implemented for log (double a).*/
    public abstract native static double log (double a) ;
    /*public native static double log (double a) {
		return log(a);
    }*/

    /**
     * Returns the greater of two <code>byte</code> values. That is, the
     * result is the argument closer to the value of
     * <code>Byte.MAX_VALUE</code>. If the arguments have the same value,
     * the result is that same value.
     *
     * @param   a   an argument.
     * @param   b   another argument.
     * @return  the larger of <code>a</code> and <code>b</code>.
     */
    public static byte max (byte a, byte b) {
		return (a >= b) ? a : b;
    }

    /**
     * Returns the greater of two <code>short</code> values. That is, the
     * result is the argument closer to the value of
     * <code>Short.MAX_VALUE</code>. If the arguments have the same value,
     * the result is that same value.
     *
     * @param   a   an argument.
     * @param   b   another argument.
     * @return  the larger of <code>a</code> and <code>b</code>.
     */
    public static short max (short a, short b) {
		return (a >= b) ? a : b;
    }

    /**
     * Returns the greater of two <code>int</code> values. That is, the
     * result is the argument closer to the value of
     * <code>Integer.MAX_VALUE</code>. If the arguments have the same value,
     * the result is that same value.
     *
     * @param   a   an argument.
     * @param   b   another argument.
     * @return  the larger of <code>a</code> and <code>b</code>.
     */
    public static int max (int a, int b) {
		return (a >= b) ? a : b;
    }

    /**
     * Returns the greater of two <code>long</code> values. That is, the
     * result is the argument closer to the value of
     * <code>Long.MAX_VALUE</code>. If the arguments have the same value,
     * the result is that same value.
     *
     * @param   a   an argument.
     * @param   b   another argument.
     * @return  the larger of <code>a</code> and <code>b</code>.
     */
    public static long max (long a, long b) {
		return (a >= b) ? a : b;
    }

    /**
     * Returns the greater of two <code>double</code> values. That is, the
     * result is the argument closer to the value of
     * <code>Double.MAX_VALUE</code>. If the arguments have the same value,
     * the result is that same value.
     *
     * @param   a   an argument.
     * @param   b   another argument.
     * @return  the larger of <code>a</code> and <code>b</code>.
     */
    public static double max (double a, double b) {
		return (a >= b) ? a : b;
    }

    /**
     * Returns the smaller of two <code>byte</code> values. That is,
     * the result the argument closer to the value of
     * <code>Byte.MIN_VALUE</code>.  If the arguments have the same
     * value, the result is that same value.
     *
     * @param   a   an argument.
     * @param   b   another argument.
     * @return  the smaller of <code>a</code> and <code>b</code>.
     */
    public static byte min (byte a, byte b) {
		return (a <= b) ? a : b;
    }

    /**
     * Returns the smaller of two <code>short</code> values. That is,
     * the result the argument closer to the value of
     * <code>Short.MIN_VALUE</code>.  If the arguments have the same
     * value, the result is that same value.
     *
     * @param   a   an argument.
     * @param   b   another argument.
     * @return  the smaller of <code>a</code> and <code>b</code>.
     */
    public static short min (short a, short b) {
		return (a <= b) ? a : b;
    }

    /**
     * Returns the smaller of two <code>int</code> values. That is,
     * the result the argument closer to the value of
     * <code>Integer.MIN_VALUE</code>.  If the arguments have the same
     * value, the result is that same value.
     *
     * @param   a   an argument.
     * @param   b   another argument.
     * @return  the smaller of <code>a</code> and <code>b</code>.
     */
    public static int min (int a, int b) {
		return (a <= b) ? a : b;
    }

    /**
     * Returns the smaller of two <code>long</code> values. That is,
     * the result the argument closer to the value of
     * <code>Long.MIN_VALUE</code>.  If the arguments have the same
     * value, the result is that same value.
     *
     * @param   a   an argument.
     * @param   b   another argument.
     * @return  the smaller of <code>a</code> and <code>b</code>.
     */
    public static long min (long a, long b) {
		return (a <= b) ? a : b;
    }

    /**
     * Returns the smaller of two <code>double</code> values. That is,
     * the result the argument closer to the value of
     * <code>Double.MIN_VALUE</code>.  If the arguments have the same
     * value, the result is that same value.
     *
     * @param   a   an argument.
     * @param   b   another argument.
     * @return  the smaller of <code>a</code> and <code>b</code>.
     */
    public static double min (double a, double b) {
		return (a <= b) ? a : b;
    }

    /**
     * Returns of value of the first argument raised to the power of the
     * second argument.
     *
     * @param   a   the base.
     * @param   b   the exponent.
     * @return  the value <code>a<sup>b</sup></code>.
     */
    /*Alias implemented for pow(double a, double b).*/
    public abstract native static double pow(double a, double b) ;    
    /*public native static double pow(double a, double b) {
		return pow(a, b);
    }*/

	/**
	 * Initializes the random number generator with a random value (picked from the timer).
	 */
	/*Alias implemented for pow(double a, double b).*/
	public abstract native static void randomize () ;
	/*public native static void randomize () {
		randomize();
	}*/

    /**
     * Returns a <code>double</code> value with a positive sign, greater
     * than or equal to <code>0.0</code> and less than <code>1.0</code>.
     * Returned values are chosen pseudorandomly with (approximately)
     * uniform distribution from that range.
     *
     * @return  a pseudorandom <code>double</code> greater than or equal
     * to <code>0.0</code> and less than <code>1.0</code>.
     */
    /*Alias implemented for random ().*/
    public abstract native static double random () ;
    /*public native static double random () {
        return rand() / ((double) RAND_MAX);
    }*/

    /**
     * Returns a <code>short</code> value with a positive sign, greater
     * than or equal to <code>0</code> and less than <code>32767</code>.
     * Returned values are chosen pseudorandomly with (approximately)
     * uniform distribution from that range.
     *
     * @return  a pseudorandom <code>short</code> greater than or equal
     * to <code>0</code> and less than <code>32767</code>.
     */
    /*Alias implemented for rand ().*/
    public abstract native static short rand () ;
    /*public native static short rand () {
        return rand();
    }*/

    /**
     * Returns a <code>short</code> value with a positive sign, greater
     * than or equal to <code>0</code> and less than the specified number.
     * Returned values are chosen pseudorandomly with (approximately)
     * uniform distribution from that range.
     *
     * @param num the number to be specified
     * @return  a pseudorandom <code>short</code> greater than or equal
     * to <code>0</code> and less than the specified number.
     */
    /*Alias implemented for rand (num).*/
    public abstract native static short rand (short num) ;
    /*public native static short rand (short num) {
        return random(num);
    }*/

    /**
     * Returns the closest <code>int</code> to the argument. The result
     * is rounded to an integer by adding 1/2, taking the floor of the
     * result, and casting the result to type <code>int</code>.
     *
     * @param   a   a floating-point value to be rounded to a
     *		<code>int</code>.
     * @return  the value of the argument rounded to the nearest
     *          <code>int</code> value.
     */
    /*Alias implemented for round(double a).*/
    public abstract native static int round(double a) ;
    /*public native static int round(double a) {
		return (long int)floor(a + 0.5);
    }*/

    /**
     * Returns the trigonometric sine of an angle.
     *
     * @param   a   an angle, in radians.
     * @return  the sine of the argument.
     */
    /*Alias implemented for sin (double a).*/
    public abstract native static double sin (double a) ;
    /*public native static double sin (double a) {
		return sin(a);
    }*/

    /**
     * Returns the correctly rounded positive square root of a
     * <code>double</code> value.
     *
     * @param   a   a value.
     * @return  the positive square root of <code>a</code>.
     */
    /*Alias implemented for sqrt (double a).*/
    public abstract native static double sqrt (double a) ;
    /*public native static double sqrt (double a) {
		return sqrt(a);
    }*/

    /**
     * Returns the trigonometric tangent of an angle.
     *
     * @param   a   an angle, in radians.
     * @return  the tangent of the argument.
     */
    /*Alias implemented for tan (double a).*/
    public abstract native static double tan (double a) ;
    /*public native static double tan (double a) {
		return tan(a);
    }*/

    /**
     * Converts an angle measured in degrees to an approximately
     * equivalent angle measured in radians.  The conversion from
     * degrees to radians is generally inexact.
     *
     * @param   angdeg   an angle, in degrees
     * @return  the measurement of the angle <code>angdeg</code>
     *          in radians.
     */
    /*Alias implemented for toRadians(double angdeg).*/
    public abstract static double toRadians(double angdeg) ;
    /*public static double toRadians(double angdeg) {
		return angdeg / 180.0 * Math.PI;
    }*/

    /**
     * Converts an angle measured in radians to an approximately
     * equivalent angle measured in degrees.  The conversion from
     * radians to degrees is generally inexact; users should
     * <i>not</i> expect <code>cos(toRadians(90.0))</code> to exactly
     * equal <code>0.0</code>.
     *
     * @param   angrad   an angle, in radians
     * @return  the measurement of the angle <code>angrad</code>
     *          in degrees.
     */
    /*Alias implemented for toDegrees(double angrad).*/
    public abstract static double toDegrees(double angrad) ;
    /*public static double toDegrees(double angrad) {
		return angrad * 180.0 / Math.PI;
    }*/
}