/*
 * QUSOFT MICROSYSTMES
 * Moka
 * Copyright 2002 Frdric Brown
 */
 
/*
 *  This file is part of Moka API.
 *
 *  Moka API is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Moka API is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Moka API; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package moka.lang;

import moka.lang.Object;
import moka.lang.String;
import moka.lang.Character;

/**
 * The Long class wraps a value of the primitive type <code>long</code>
 * in an object. An object of type <code>long</code> contains a
 * single field whose type is <code>long</code>.
 * <p>
 * In addition, this class provides several methods for converting
 * an <code>long</code> to a <code>String</code> and a
 * <code>String</code> to an <code>long</code>, as well as other
 * constants and methods useful when dealing with an
 * <code>long</code>.
 * <p>
 *
 * @author  Frdric Brown
 * @version 1.0, 2002-08-29
 * @since   MDK1.0a
 */
public class Long {

    /**
     * A constant holding the minimum value a <code>long</code> can
     * have, -2<sup>63</sup>.
     */
	public static final long   MIN_VALUE = 0x8000000000000000;

    /**
     * A constant holding the maximum value a <code>long</code> can
     * have, 2<sup>63</sup>-1.
     */
	public static final long   MAX_VALUE = 0x7fffffffffffffff;

	/**
	* The value of the Long.
	*/
	private long value;

    /**
     * Constructs a newly allocated <code>Long</code> object that
     * represents the primitive <code>long</code> argument.
     *
     * @param   value   the value to be represented by the <code>long</code>.
     */
    public Long(long value) {
		this.value = value;
    }

    /**
     * Constructs a newly allocated <code>Long</code> object that 
     * represents the value represented by the string. The string is 
     * converted to an <tt>long</tt> in exactly the manner used by the 
     * <tt>parseLong</tt> method.
     *
     * @param      s   the <code>String</code> to be converted to an
     *                 <code>Long</code>.
     */
    public Long(String s) {
		this.value = Long.parseLong(s);
    }
    
    /**
     * Returns a <code>String</code> object representing this
     * <code>Long</code>'s value. The value is converted to signed
     * decimal representation and returned as a string, exactly as if
     * the long value were given as an argument to the valueOf method
     * of String.
     *
     * @return  a string representation of the value of this object.
     */
    public String toString() {
    	return String.valueOf(this.value);
    }
    
    /**
     * Compares this object to the specified object.  The result is
     * <code>true</code> if and only if the argument is not
     * <code>null</code> and is an <code>Long</code> object that
     * contains the same <code>long</code> value as this object.
     *
     * @param   obj   the object to compare with.
     * @return  <code>true</code> if the objects are the same;
     *          <code>false</code> otherwise.
     */
    public boolean equals(Object obj) {
		if (obj instanceof Long) {
				return this.value == ((Long)obj).value;
		}
		return false;
    }
    
    /**
     * Returns the value of this <code>Long</code> as an
     * <code>long</code>.
     */
    public long longValue() {
		return this.value;
    }

    /**
     * Returns the quotient of the first operand divided by the second operand.
     *
     * @param   l0   the first operand.
     * @param   l1   the second operand.
     * @return  the quotient of the first operand divided by the second operand.
     */
    /*public native static long div(long l0, long l1) {
    	long long int i;
    	
    	for (i = 0; l0 > l1; i++) {
    		l0 = l0 - l1;
    	}
    	
    	return i;
    }*/
    
    /**
     * Returns the remainder of the first operand divided by the second operand.
     *
     * @param   l0   the first operand.
     * @param   l1   the second operand.
     * @return  the remainder of the first operand divided by the second operand.
     */
    /*public native static long mod(long l0, long l1) {
    	long long int i;
    	
    	for (i = 0; l0 > l1; i++) {
    		l0 = l0 - l1;
    	}
    	
    	return l0;
    }*/
    
    /**
     * Parses the string argument as a signed decimal long. The 
     * characters in the string must all be decimal digits, except that 
     * the first character may be an ASCII minus sign <code>'-'</code> 
     * (<code>'&#92;u002D'</code>) to indicate a negative value. The resulting 
     * integer value is returned.  The string argument is finalized before the method
     * returns.
     *
     * @param s	   a <code>String</code> containing the <code>long</code>
     *             representation to be parsed
     * @return     the long value represented by the argument in decimal.
     */
    public native static long parseLong(String s) {
    	long long int value = 0;
		long long int m = 1;
		long long pos = 0;
		long long l;
		long long i;
		
		if (s->value[pos] == '-') {
			pos++;
		}
		
		l = strlen(s->value) - pos - 1;
		
		for (i = 0; i < l; i++) {
			m *= 10;
		}
		
		while (m >= 10) {
			value += (s->value[pos++] - '0') * m;
			 m /= 10;
		}
		
		value += s->value[pos] - '0';
		
		if (s->value[0] == '-') {
			value *= -1;
		}
		
		s->finalize_(s);
		
		return value;
    }
    
    /**
     * Returns the value of this Long as an <code>int</code>.
     * This may involve rounding or truncation.
     *
     * @return  the numeric value represented by this object after conversion
     *          to type <code>int</code>.
     */
    public int intValue () {
    	return (int)this.value;
    }

    /**
     * Returns the value of this Long as a <code>double</code>.
     * This may involve rounding.
     *
     * @return  the numeric value represented by this object after conversion
     *          to type <code>double</code>.
     */
    public double doubleValue () {
    	return (double)((int)this.value);
    }

    /**
     * Returns the value of this Long as a <code>byte</code>.
     * This may involve rounding or truncation.
     *
     * @return  the numeric value represented by this object after conversion
     *          to type <code>byte</code>.
     */
    public byte byteValue () {
    	return (byte)this.value;
    }

    /**
     * Returns the value of this Long as a <code>short</code>.
     * This may involve rounding or truncation.
     *
     * @return  the numeric value represented by this object after conversion
     *          to type <code>short</code>.
     */
    public short shortValue() {
    	return (short)this.value;
    }
    
    /**
     * Checks if the specified string is a decimal representation
     * of a long integer.
     *
     * @param str The string to verify
     * @return true if the specified string is a decimal representation of a long integer, false otherwize.
     */
    public static boolean isLong (String str) {
    	short i;
    	short len = native ( strlen(str->value) );
    	
    	if (len > 20 || len > 19 && str.value[0] != '-') {
    		return false;
    	}
    	
    	for (i = 0; i < len; i++) {
    		if (Character.isDigit((char)str.value[i]) || i == 0 && str.value[i] == '-') {
    			continue;
    		}
    		return false;
    	}
    	
    	return true;
    }
}