/*
 * QUSOFT MICROSYSTMES
 * Moka
 * Copyright 2002 Frdric Brown
 */
 
/*
 *  This file is part of Moka API.
 *
 *  Moka API is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Moka API is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Moka API; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package moka.lang;

import moka.lang.Object;
import moka.lang.String;

/**
 * The Integer class wraps a value of the primitive type <code>int</code>
 * in an object. An object of type <code>Integer</code> contains a
 * single field whose type is <code>int</code>.
 * <p>
 * In addition, this class provides several methods for converting
 * an <code>int</code> to a <code>String</code> and a
 * <code>String</code> to an <code>int</code>, as well as other
 * constants and methods useful when dealing with an
 * <code>int</code>.
 * <p>
 *
 * @author  Frdric Brown
 * @version 1.0, 2002-08-29
 * @since   MDK1.0a
 */
public class Integer {

	/**
	* The smallest value of type <code>int</code>. The constant 
	* value of this field is <tt>-2147483648</tt>.
	*/
	public static final int   MIN_VALUE = 0x80000000;

	/**
	* The largest value of type <code>int</code>. The constant 
	* value of this field is <tt>2147483647</tt>.
	*/
	public static final int   MAX_VALUE = 0x7fffffff;

	/**
	* The value of the Integer.
	*/
	private int value;

    /**
     * Constructs a newly allocated <code>Integer</code> object that
     * represents the primitive <code>int</code> argument.
     *
     * @param   value   the value to be represented by the <code>Integer</code>.
     */
    public Integer (int value) {
		this.value = value;
    }

    /**
     * Constructs a newly allocated <code>Integer</code> object that 
     * represents the value represented by the string. The string is 
     * converted to an <tt>int</tt> in exactly the manner used by the 
     * <tt>parseInt</tt> method.
     *
     * @param      s   the <code>String</code> to be converted to an
     *                 <code>Integer</code>.
     */
    public Integer (String s) {
		this.value = Integer.parseInt(s);
    }
    
    /**
     * Returns a <code>String</code> object representing this
     * <code>Integer</code>'s value. The value is converted to signed
     * decimal representation and returned as a string, exactly as if
     * the integer value were given as an argument to the valueOf method
     * of String.
     *
     * @return  a string representation of the value of this object.
     */
    public String toString () {
    	return String.valueOf(this.value);
    }
    
    /**
     * Compares this object to the specified object.  The result is
     * <code>true</code> if and only if the argument is not
     * <code>null</code> and is an <code>Integer</code> object that
     * contains the same <code>int</code> value as this object.
     *
     * @param   obj   the object to compare with.
     * @return  <code>true</code> if the objects are the same;
     *          <code>false</code> otherwise.
     */
    public boolean equals (Object obj) {
		if (obj instanceof Integer) {
			return this.value == ((Integer)obj).value;
		}
		return false;
    }
    
    /**
     * Returns the value of this <code>Integer</code> as an
     * <code>int</code>.
     */
    public int intValue() {
		return this.value;
    }
    
    /**
     * Parses the string argument as a signed decimal integer. The 
     * characters in the string must all be decimal digits, except that 
     * the first character may be an ASCII minus sign <code>'-'</code> 
     * (<code>'&#92;u002D'</code>) to indicate a negative value. The resulting 
     * integer value is returned.  The string argument is finalized before the method
     * returns.
     *
     * @param s	   a <code>String</code> containing the <code>int</code>
     *             representation to be parsed
     * @return     the integer value represented by the argument in decimal.
     */
    public native static int parseInt (String s) {
    	long int value = atol(s->value);
    	
    	s->finalize_(s);
    	
    	return value;
    }

    /**
     * Returns the value of this Integer as a <code>long</code>.
     * This may involve rounding or truncation.
     *
     * @return  the numeric value represented by this object after conversion
     *          to type <code>long</code>.
     */
    public long longValue () {
    	return (long)this.value;
    }

    /**
     * Returns the value of this Integer as a <code>double</code>.
     * This may involve rounding.
     *
     * @return  the numeric value represented by this object after conversion
     *          to type <code>double</code>.
     */
    public double doubleValue () {
    	return (double)this.value;
    }

    /**
     * Returns the value of this Integer as a <code>byte</code>.
     * This may involve rounding or truncation.
     *
     * @return  the numeric value represented by this object after conversion
     *          to type <code>byte</code>.
     */
    public byte byteValue () {
    	return (byte)this.value;
    }

    /**
     * Returns the value of this Integer as a <code>short</code>.
     * This may involve rounding or truncation.
     *
     * @return  the numeric value represented by this object after conversion
     *          to type <code>short</code>.
     */
    public short shortValue() {
    	return (short)this.value;
    }
    
    /**
     * Checks if the specified string is a decimal representation
     * of an integer.
     *
     * @param str The string to verify
     * @return true if the specified string is a decimal representation of an integer, false otherwize.
     */
    public static boolean isInt (String str) {
    	short i;
    	short len = native ( strlen(str->value) );
    	
    	if (len > 11 || len > 10 && str.value[0] != '-') {
    		return false;
    	}
    	
    	for (i = 0; i < len; i++) {
    		if (Character.isDigit((char)str.value[i]) || i == 0 && str.value[i] == '-') {
    			continue;
    		}
    		return false;
    	}
    	
    	return true;
    }
}