/*
 * QUSOFT MICROSYSTMES
 * Moka
 * Copyright 2002 Frdric Brown
 */

/*
 *  This file is part of Moka API.
 *
 *  Moka API is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Moka API is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Moka API; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package moka.lang;

/**
 * The <code>Double</code> class wraps a value of the primitive type
 * <code>double</code> in an object. An object of type
 * <code>Double</code> contains a single field whose type is
 * <code>double</code>.
 * <p>
 * In addition, this class provides several methods for converting a
 * <code>double</code> to a <code>String</code> and a
 * <code>String</code> to a <code>double</code>, as well as other
 * constants and methods useful when dealing with a
 * <code>double</code>.
 * <p>
 * This class can also be used to manipulate float.
 *
 * @author  Frdric Brown
 * @version 1.0, 2002-09-05
 * @since   MDK1.0a
 */
public class Double {

    /**
     * A constant holding the largest positive finite value of type
     * <code>double</code>.
     */
    public static final double MAX_VALUE = 9.999999999999999e999;

    /**
     * A constant holding the smallest positive nonzero value of type
     * <code>double</code>.
     */
    public static final double MIN_VALUE = 1e-999;

	/**
	* The value of the Double.
	*/
	private double value;

    /**
     * Constructs a newly allocated <code>Double</code> object that
     * represents the primitive <code>double</code> argument.
     *
     * @param   value   the value to be represented by the <code>Integer</code>.
     */
    public Double(double value) {
		this.value = value;
    }

    /**
     * Constructs a newly allocated <code>Double</code> object that
     * represents the value represented by the string. The string is
     * converted to an <tt>double</tt> in exactly the manner used by the
     * <tt>parseDouble</tt> method.
     *
     * @param      s   the <code>String</code> to be converted to an
     *                 <code>Integer</code>.
     */
    public Double(String s) {
		this.value = Double.parseDouble(s);
    }

    /**
     * Returns a <code>String</code> object representing this
     * <code>Double</code>'s value. The value is converted to floating point,
     * format [-]d.dddde[sign]ddd (exponential format) representation and returned as a string, exactly as if
     * the double value were given as an argument to the valueOf method
     * of String.
     *
     * @return  a string representation of the value of this object.
     */
    public String toString() {
    	return String.valueOf(this.value);
    }

    /**
     * Compares this object to the specified object.  The result is
     * <code>true</code> if and only if the argument is not
     * <code>null</code> and is an <code>Double</code> object that
     * contains the same <code>double</code> value as this object.
     *
     * @param   obj   the object to compare with.
     * @return  <code>true</code> if the objects are the same;
     *          <code>false</code> otherwise.
     */
    public boolean equals(Object obj) {
		if (obj instanceof Double) {
			return this.value == ((Double)obj).value;
		}
		return false;
    }

    /**
     * Returns the value of this <code>Double</code> as an
     * <code>double</code>.
     */
    public double doubleValue() {
		return this.value;
    }

    /**
     * Parses the string argument as a signed decimal integer.  It recognizes
     * the character representation of a floating point number. The resulting
     * double value is returned.  The string argument is finalized before the method
     * returns.
     *
     * @param s	   a <code>String</code> containing the <code>double</code>
     *             representation to be parsed
     * @return     the double value represented by the argument in decimal.
     */
    public native static double parseDouble(String s) {
		char* buffer = Character_copyValueOf_char_p(s->value);
		short i;
		double value;
		char c;

		for (i = 0; (c = buffer[i]); i++)
		{
			if (c == '-') buffer[i] = 0xAD;
			if ((c|32) == 'e') buffer[i] = 0x95;
		}

		value = atof(buffer);

		free(buffer);

		s->finalize_(s);

    	return value;
    }

    /**
     * Returns the value of this Double as an <code>int</code>.
     * This may involve rounding or truncation.
     *
     * @return  the numeric value represented by this object after conversion
     *          to type <code>int</code>.
     */
    public int intValue () {
    	return (int)this.value;
    }

    /**
     * Returns the value of this Double as a <code>long</code>.
     * This may involve rounding or truncation.
     *
     * @return  the numeric value represented by this object after conversion
     *          to type <code>long</code>.
     */
    public long longValue () {
    	//return (long)this.value;
    	return (long)((int)this.value);
    }

    /**
     * Returns the value of this Double as a <code>byte</code>.
     * This may involve rounding or truncation.
     *
     * @return  the numeric value represented by this object after conversion
     *          to type <code>byte</code>.
     */
    public byte byteValue () {
    	return (byte)this.value;
    }

    /**
     * Returns the value of this Double as a <code>short</code>.
     * This may involve rounding or truncation.
     *
     * @return  the numeric value represented by this object after conversion
     *          to type <code>short</code>.
     */
    public short shortValue() {
    	return (short)this.value;
    }

    /**
     * Checks if the specified string is a decimal representation
     * of a double.
     *
     * @param str The string to verify
     * @return true if the specified string is a decimal representation of a double, false otherwize.
     */
    public static boolean isDouble (String str) {
    	return Double.parseDouble(str.toString()) != NAN;
    }
}