/*
 * QUSOFT MICROSYSTMES
 * Moka
 * Copyright 2004 Frdric Brown
 */

/*
 *  This file is part of Moka API.
 *
 *  Moka API is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Moka API is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Moka API; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package moka.lang;

import moka.lang.Object;
import moka.lang.String;

/**
 * Class DLL provides methods to deal with DLLs. Look moka.lang.DLL_Interface interface to learn more about using DLLs in Moka.
 *
 * @author  Frdric Brown
 * @version 1.0, 2004-03-18
 * @since   MDK2.2
 */
 public final class DLL {
 	/** The DLL was loaded and initialized successfully. */
 	public static final short DLL_OK = native.DLL_OK;

 	/** The DLL could not be loaded because the program counter is not in the "ghost space" (the virtual address space where HW2 protections do not take effect). Note that exe-packed programs are always executed in the ghost space. By default Moka programs executes in ghost space. */
 	public static final short DLL_NOTINGHOSTSPACE = native.DLL_NOTINGHOSTSPACE; 

 	/** The DLL is not found. This means that either a file with the name DLLName is not found in any folder, or such files exist, but none of them has the correct extension ('DLL'), the correct embedded signature, and the correct identification number (determined by the ID parameter). */
 	public static final short DLL_NOTFOUND = native.DLL_NOTFOUND;  
 
 	/** The attempt to lock the DLL in memory has failed due to some strange reason. This error code is very unlikely. */
 	public static final short DLL_LOCKFAILED = native.DLL_LOCKFAILED;   
 	
 	/** There is not memory to load the DLL into RAM. */
 	public static final short DLL_OUTOFMEM = native.DLL_OUTOFMEM;   

 	/** There is already another DLL loaded in the RAM. Due to efficiency reasons, only one DLL is allowed to be loaded at the same time. You need to unload the current DLL using UnloadDLL before loading another one. Anyway, using more than one DLL is strongly deprecated if you don't know exactly what you are doing and why you are doing so. */
 	public static final short DLL_ALREADYLOADED = native.DLL_ALREADYLOADED;    
  
 	/** There is at least one valid DLL file with the name DLLName and with the correct extension, signature, and identification number, but none of them has a major version number which is equal to the expected major number (determined by the Major parameter) and a minor version number which is greater or equal than the expected minor number (determined by the Minor parameter). */
 	public static final short DLL_WRONGVERSION = native.DLL_WRONGVERSION;  
 	
 	private DLL () {}
 	
    /**
     * Loads a DLL into memory and prepares it for use. Unloads the previously loaded DLL, if any, using System.unloadDLL(), before
     * attempting to load a new DLL.
     *
     * @param varName an ANSI string representing the standard ANSI C name of the DLL file which is to be loaded.
     * @param id a DLL identification number introduced for safety reasons: any file which does not have the extension 'DLL', a valid embedded signature and an embedded identification number which is equal to ID is ignored.
     * @param major the expected major version number of the DLL, loadDLL will refuse to load a DLL if the expected and actual version numbers are not the same.
     * @param minor the expected minor version number of the DLL, loadDLL will refuse to load a DLL if the expected minor version number is greater than the actual minor version number.
     * @return System.DLL_OK if the DLL was loaded successfully, an error code otherwize.
     */
    public abstract static short load(char[] varName, uint id, ushort major, ushort minor) ;

    /**
     * Loads a DLL into memory and prepares it for use.Unloads the previously loaded DLL, if any, using System.unloadDLL(), before
     * attempting to load a new DLL.
     *
     * @param varName an String representing the standard ANSI C name of the DLL file which is to be loaded.
     * @param id a DLL identification number introduced for safety reasons: any file which does not have the extension 'DLL', a valid embedded signature and an embedded identification number which is equal to ID is ignored.
     * @param major the expected major version number of the DLL, loadDLL will refuse to load a DLL if the expected and actual version numbers are not the same.
     * @param minor the expected minor version number of the DLL, loadDLL will refuse to load a DLL if the expected minor version number is greater than the actual minor version number.
     * @return System.DLL_OK if the DLL was loaded successfully, an error code otherwize.
     */
    public static short load(String varName, uint id, ushort major, ushort minor) {
		short v = DLL.load((char[])varName.value, id, major, minor);

		varName.finalize();
		
		return v;
	}

	/**
	 * Unloads a previously loaded DLL. It does nothing if the DLL was not previously loaded.
	 */
	public abstract static void unload () ;
}