/*
 * QUSOFT MICROSYSTMES
 * Moka
 * Copyright 2002 Frdric Brown
 */

/*
 *  This file is part of Moka API.
 *
 *  Moka API is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Moka API is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Moka API; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package moka.lang;

import moka.lang.Object;
import moka.lang.String;

/**
 * The Byte class wraps a value of the primitive type <code>byte</code>
 * in an object. An object of type <code>Byte</code> contains a
 * single field whose type is <code>byte</code>.
 * <p>
 * In addition, this class provides several methods for converting
 * an <code>byte</code> to a <code>String</code> and a
 * <code>String</code> to an <code>byte</code>, as well as other
 * constants and methods useful when dealing with an
 * <code>byte</code>.
 *
 * @author  Frdric Brown
 * @version 1.0, 2002-10-30
 * @since   MDK1.0a
 */
public class Byte {

	/**
	* The smallest value of type <code>byte</code>. The constant
	* value of this field is <tt>-128</tt>.
	*/
	public static final byte   MIN_VALUE = -128;

	/**
	* The largest value of type <code>byte</code>. The constant
	* value of this field is <tt>127</tt>.
	*/
	public static final byte   MAX_VALUE = 127;

	/**
	* The value of the Byte.
	*/
	private byte value;

    /**
     * Constructs a newly allocated <code>Byte</code> object that
     * represents the primitive <code>byte</code> argument.
     *
     * @param   value   the value to be represented by the <code>Byte</code>.
     */
    public Byte(byte value) {
		this.value = value;
    }

    /**
     * Constructs a newly allocated <code>Byte</code> object that
     * represents the value represented by the string. The string is
     * converted to an <tt>byte</tt> in exactly the manner used by the
     * <tt>parseByte</tt> method.
     *
     * @param      s   the <code>String</code> to be converted to an
     *                 <code>Byte</code>.
     */
    public Byte(String s) {
		this.value = Byte.parseByte(s);
    }

    /**
     * Returns a <code>String</code> object representing this
     * <code>Byte</code>'s value. The value is converted to signed
     * decimal representation and returned as a string.
     *
     * @return  a string representation of the value of this object.
     */
    public native String toString() {
		char vec[4];

		sprintf(vec, "%i", (short int)this->value);

		return TString_char_p_BOOL(strcpy(malloc(strlen(vec) + 1), vec), FALSE);
    }

    /**
     * Compares this object to the specified object.  The result is
     * <code>true</code> if and only if the argument is not
     * <code>null</code> and is an <code>Byte</code> object that
     * contains the same <code>byte</code> value as this object.
     *
     * @param   obj   the object to compare with.
     * @return  <code>true</code> if the objects are the same;
     *          <code>false</code> otherwise.
     */
    public boolean equals(Object obj) {
		if (obj instanceof Byte) {
			return this.value == ((Byte)obj).value;
		}
		return false;
    }

    /**
     * Returns the value of this <code>Byte</code> as an
     * <code>byte</code>.
     */
    public byte byteValue() {
		return this.value;
    }

    /**
     * Parses the string argument as a signed decimal integer. The
     * characters in the string must all be decimal digits, except that
     * the first character may be an ASCII minus sign <code>'-'</code>
     * (<code>'&#92;u002D'</code>) to indicate a negative value. The resulting
     * byte value is returned.  The string argument is finalized before the method
     * returns.
     *
     * @param s	   a <code>String</code> containing the <code>byte</code>
     *             representation to be parsed
     * @return     the byte value represented by the argument in decimal.
     */
    public native static byte parseByte(String s) {
    	byte value = (byte)atoi(s->value);

    	s->finalize_(s);

    	return value;
    }

    /**
     * Returns the value of this Byte as an <code>int</code>.
     * This may involve rounding or truncation.
     *
     * @return  the numeric value represented by this object after conversion
     *          to type <code>int</code>.
     */
    public int intValue () {
    	return (int)this.value;
    }

    /**
     * Returns the value of this Byte as a <code>long</code>.
     * This may involve rounding or truncation.
     *
     * @return  the numeric value represented by this object after conversion
     *          to type <code>long</code>.
     */
    public long longValue () {
    	return (long)this.value;
    }

    /**
     * Returns the value of this Byte as a <code>double</code>.
     * This may involve rounding.
     *
     * @return  the numeric value represented by this object after conversion
     *          to type <code>double</code>.
     */
    public double doubleValue () {
    	return (double)this.value;
    }

    /**
     * Returns the value of this Byte as a <code>short</code>.
     * This may involve rounding or truncation.
     *
     * @return  the numeric value represented by this object after conversion
     *          to type <code>short</code>.
     */
    public short shortValue() {
    	return (short)this.value;
    }

    /**
     * Checks if the specified string is a decimal representation
     * of a byte.
     *
     * @param str The string to verify
     * @return true if the specified string is a decimal representation of a byte, false otherwize.
     */
    public static boolean isByte (String str) {
    	short i;
    	short len = native ( strlen(str->value) );

    	if (len > 4 || len > 3 && str.value[0] != '-') {
    		return false;
    	}

    	for (i = 0; i < len; i++) {
    		if (Character.isDigit((char)str.value[i]) || i == 0 && str.value[i] == '-') {
    			continue;
    		}
    		return false;
    	}

    	return true;
    }
}