/*
 * QUSOFT MICROSYSTMES
 * Moka
 * Copyright 2002 Frdric Brown
 */
 
/*
 *  This file is part of Moka API.
 *
 *  Moka API is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Moka API is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Moka API; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package moka.lang;

import moka.lang.Object;
import moka.lang.String;

/**
 * The Boolean class wraps a value of the primitive type <code>boolean</code>
 * in an object. An object of type <code>Boolean</code> contains a
 * single field whose type is <code>boolean</code>.
 * <p>
 * In addition, this class provides several methods for converting
 * an <code>boolean</code> to a <code>String</code> and a
 * <code>String</code> to an <code>boolean</code>, as well as other
 * constants and methods useful when dealing with an
 * <code>boolean</code>.
 *
 * @author  Frdric Brown
 * @version 1.0, 2002-10-30
 * @since   MDK1.0a
 */
public class Boolean {

	/**
	* The value of the Boolean.
	*/
	private boolean value;

    /**
     * Constructs a newly allocated <code>Boolean</code> object that
     * represents the primitive <code>boolean</code> argument.
     *
     * @param   value   the value to be represented by the <code>Boolean</code>.
     */
    public Boolean(boolean value) {
		this.value = value;
    }

    /**
     * Constructs a newly allocated <code>Boolean</code> object that 
     * represents the value represented by the string. The string is 
     * converted to an <tt>boolean</tt> in exactly the manner used by the 
     * <tt>parseBoolean</tt> method.
     *
     * @param      s   the <code>String</code> to be converted to an
     *                 <code>Boolean</code>.
     */
    public Boolean(String s) {
		this.value = Boolean.parseBoolean(s);
    }
    
    /**
     * Returns a <tt>String</tt> object representing this Boolean's
     * value.  If this object represents the value <code>true</code>,
     * a string equal to <code>"true"</code> is returned. Otherwise, a
     * string equal to <code>"false"</code> is returned.
     *
     * @return  a string representation of the value of this object.
     */
    public String toString() {
    	return String.valueOf(this.value);
    }
    
    /**
     * Compares this object to the specified object.  The result is
     * <code>true</code> if and only if the argument is not
     * <code>null</code> and is an <code>Boolean</code> object that
     * contains the same <code>boolean</code> value as this object.
     *
     * @param   obj   the object to compare with.
     * @return  <code>true</code> if the objects are the same;
     *          <code>false</code> otherwise.
     */
    public boolean equals(Object obj) {
		if (obj instanceof Boolean) {
			return this.value == ((Boolean)obj).value;
		}
		return false;
    }
    
    /**
     * Returns the value of this <code>Boolean</code> as a
     * <code>boolean</code>.
     */
    public boolean booleanValue() {
		return this.value;
    }
    
    /**
    * Returns a <code>Boolean</code> with a value represented by the
     * specified String.  The <code>Boolean</code> returned represents the
     * value <code>true</code> if the string argument is not <code>null</code>
     * and is equal to the string <code>"true"</code>.  The string argument is finalized before the method
     * returns.
     *
     * @param s	   a <code>String</code>
     * @return  the <code>Boolean</code> value represented by the string.
     */
    public native static int parseBoolean(String s) {
    	BOOL value = !strcmp(s->value, "true");
    	
    	s->finalize_(s);
    	
    	return value;
    }
}