/*
 * QUSOFT MICROSYSTMES
 * Moka
 * Copyright 2004 Frdric Brown
 */

/*
 *  This file is part of Moka API.
 *
 *  Moka API is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Moka API is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Moka API; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package moka.io;

import moka.lang.String;
import moka.io.File;

/**
 * Provides methods to deal with the TIOS' variable Allocation table (VAT) and offers a full random access to its data.
 * This class aims to provide a faster and more efficient
 * way to access storage than <code>File</code> class. However,
 * you must allocate the space needed for data storage manually.
 *
 * Also, you must invoke <code>allocHandle()<code> to create a new file (otherwise
 * the instance of <code>VatEntry</code> will try to refer an existing VAT entry).
 *
 * @author  Frdric Brown
 * @version 1.0, 2004-11-24
 * @since   MDK2.21
 */
 /*
 * There is not important gain when using the inherited methods from class <code>IOStream</code> because they
 * aren't a lot faster than those provided by the class <code>File</code> (there is however a certain difference because there is no error checking).
 */
public class VatEntry {
	/** The VAT string representation of the <code>VatEntry</code>'s name. */
	private char[] symName;

	/** The length of the name. */
	private short len;

	/** A flag used to indicate if the entry should be unlocked when the <code>VatEntry</code> is finalized or released. */
	private boolean unlock = false;

	/** The handle associated with this <code>VatEntry</code>. Should be only read. */
	public HANDLE entryHandle = H_NULL;

	/* Used by the IOStream class' inherited methods. */
	//private char *ptr;

	/** Opens the stream. Mandatory before using the IOStream class' inherited methods. */
	/*public void open() {
		opened = true;
		error = false;
		ptr = derefHandle();
	}*/

	/**
	 * Closes the stream. Basically releases this <code>VatEntry</code> instance.
	 */
	/*public void close() {
		opened = false;
		release();
	}*/

    /**
     * Writes an 8 bit byte.
     *
     * @param	val the byte value to be written
     */
	/*public void writeByte(char val) {
		ptr[0] = val;
		ptr++;
	}*/

    /**
     * Writes a buffer of bytes up to the specified length.
     *
     * @param	buffer the buffer of bytes to be written
     * @param	len the specified length
     */
	/*public void writeBytes(char* buffer, short len) {
		memcpy(ptr, buffer, len);
		ptr += len;
	}*/

    /**
     * Reads an 8 bit byte.
	 * Since this method is an abstract one, subclasses of IOStream
	 * must overwrite this method.
     *
     * @return	the byte value read
     */
	/*public char readByte() {
		ptr ++;
		return *(ptr - 1);
	}*/

    /**
     * Copies to a buffer of bytes up to the specified length bytes read from the stream.
	 * Since this method is an abstract one, subclasses of IOStream
	 * must overwrite this method.
     *
     * @param	buffer the buffer of bytes where the bytes will be copied
     * @param	len the specified length
     */
	/*public void readBytes(char* buffer, short len) {
		memcpy(buffer, ptr, len);
		ptr += len;
	}*/

    /**
     * Creates a new <code>VatEntry</code> instance by using the given name string.
     * If the name string object risk to be used later, use a copy of the string object
     * instead of the original.
     *
     @param name The name of the VAT entry
     */
	public VatEntry(String name) {
		len = strlen(name.value);
		symName = name.toVat();
		name.finalize();
	}

    /**
     * Creates a new <code>VatEntry</code> instance by using the given <code>File</code>.
     * The <code>File</code> will be finalized.
     *
     @param f The symbolic path name of the VAT entry
     */
	public VatEntry(File f) {
		len = strlen(f.name.value);
		symName = f.name.toVat();
		f.finalize();
	}

	/**
	 * Tests whether the variable denoted by this <code>VatEntry</code> instance exists.
	 *
     @return  <code>true</code> if and only if the variable denoted by this <code>VatEntry</code> instance exists; <code>false</code> otherwise
	*/
	public native boolean exists () {

		return !(SymFind(this->symName).folder == HS_NULL.folder) ;
	}

    /**
     * Deletes the variable denoted by this <code>VatEntry</code> instance.
     *
     * @return  <code>true</code> if and only if the variable is
     *          successfully deleted; <code>false</code> otherwise
     */
    public boolean delete() {
    	return SymDel (symName);
    }

    /**
     * Creates the directory named by this <code>VatEntry</code> instance.
     *
     * @return  <code>true</code> if and only if the directory was
     *          created; <code>false</code> otherwise
     */
    public boolean mkdir() {
    	return (FolderAdd (symName) != H_NULL);
    }

    /**
     * Removes the directory named by this <code>VatEntry</code> instance including
     * all files in it.
     *
     * @return  <code>true</code> if and only if the directory was
     *          removed; <code>false</code> otherwise
     */
    public boolean rmdir() {
    	return FolderDel (symName);
    }

    /**
     * Renames the variable denoted by this <code>VatEntry</code> instance.
     * If the dest string object risk to be used later, use a copy of the string object
     * instead of the original. Even if the operation is successful, this <code>VatEntry</code> instance
     * pathname will still refer to the previous VAT entry.
     *
     * @param  dest  The new abstract pathname for the named variable
     *
     * @return  <code>true</code> if and only if the renaming succeeded;
     *          <code>false</code> otherwise
     */
    public boolean renameTo(String dest) {
    	char* newVat = dest.toVat();
    	boolean ok = SymMove (symName, newVat);

    	free(newVat - strlen(dest.value) - 1);
    	dest.finalize();

		return ok;
    }

    /**
     * Associates a new VAT entry with this <code>VatEntry</code> instance, after releasing the
     * <code>VatEntry</code>.
     *
     @param name The name of the entry to assign to this <code>VatEntry</code> instance.
     */

    public void assign(String name) {
		release();
		free(symName - len - 1);
		len = strlen(name.value);
		symName = name.toVat();
		name.finalize();
    }

	/**
     * Returns a pointer to the dereferenced handle associated with this
     * <code>VatEntry</code>. If the handle is not already locked, The associated handle
     * will be locked until the <code>VatEntry</code> is released or finalized.
     *
     @return The dereferenced handle associated with the VAT entry. <code>null</code> in case of error.
	 */
	public byte[] derefHandle() {
		if (!entryHandle) {
			SYM_ENTRY *entry = DerefSym(SymFind(symName));

			if (!entry) {
				return null;
			}

			entryHandle = native(entry->handle);

			if (!HeapGetLock(entryHandle)) {
				unlock = true;
				HeapLock(entryHandle);
			}
		}

		return (((char *)HeapDeref(entryHandle)) + 2);
	}

	/**
     * Creates a new entry in the VAT, and associates it with this <code>VatEntry</code>. Will allocate
     * <code>size</code> bytes of space for user data. The associated handle
     * will be locked until the <code>VatEntry</code> is released or finalized.
     *
     @param size The size needed for user data.
     @return The dereferenced handle associated with the new VAT entry. <code>null</code> in case of error.
	 */
	public byte[] allocHandle(ulong size) {
		SYM_ENTRY *entry = DerefSym(SymAdd(symName));
		ushort *fsize;

		if (!entry) {
			return null;
		}

		native(entry->handle) = entryHandle = HeapAlloc(size + 2);

		if (!entryHandle) {
			return null;
		}

		HeapLock(entryHandle);
		unlock = true;
		
		fsize = HeapDeref(entryHandle);
		
		*fsize = size;

		return (((char *)fsize) + 2);
	}

	/**
     * Reallocates the user data space associated with the <code>VatEntry</code>. Will reallocate the
     * space for user data to <code>size</code> bytes. The associated handle
     * will be locked until the <code>VatEntry</code> is released or finalized. One of the two
     * initial acces method (<code>derefHandle()<code> or <code>allocHandle()<code>) must have been called
     * previously for this method to work properly. Also, keep in mind that the behavior
     * is undefined if the handle have been locked by another entity than this <code>VatEntry</code>, because
     * this method will unlock the handle, reallocate it and relock it.
     *
     @param size The size needed for user data.
     @return The dereferenced handle associated with the VAT entry. <code>null</code> in case of error.
	 */
	public byte[] reallocHandle(ulong size) {
		HANDLE h;
		ushort *fsize;

		if (!entryHandle) {
			return null;
		}

		HeapUnlock(entryHandle);

		h = HeapRealloc(entryHandle, (size + 2));

		if (!h) {
			HeapLock(entryHandle);
			return null;
		}

		HeapLock(h);
		unlock = true;
		
		fsize = HeapDeref(h);
		
		*fsize = size;

		return (((char *)fsize) + 2);
	}

	/**
     * Creates a new entry in the VAT, and associates it with this <code>VatEntry</code>. Will sets the
     * specified buffer as user data. The associated handle
     * will be locked until the <code>VatEntry</code> is released or finalized.
     *
     @param ptr The pointer to the userdata.
     @return The dereferenced handle associated with the VAT entry. <code>null</code> in case of error.
	 */
	/*public byte[] setHandle(byte[] ptr) {
	    //BUGGY !
		SYM_ENTRY *entry;

		if (entryHandle) {
			entry = DerefSym(SymFind(symName));
			HeapFree(entryHandle);
		}
		else {
			entry = DerefSym(SymAdd(symName));
		}

		if (!entry) {
			return null;
		}
		
		native(entry->handle) = entryHandle = (HANDLE)(*(ptr - 2));

		HeapLock(entryHandle);
		unlock = true;

		return ptr;
	}*/

	/**
     * Releases the <code>VatEntry</code>. Will unlock the handle only if this instance
     * locked it during its operation (will be the case if the handle was not locked
     * before <code>derefHandle()<code> or <code>allocHandle()<code> was called).
	 */
	public void release() {
		if (unlock) {
			HeapUnlock(entryHandle);
		}
		entryHandle = H_NULL;
	}

    /**
     * Frees the memory of system ressources used by this <code>VatEntry</code>
     * object.  The handle is released before the method returns (see <code>release()</code>).
     */
	public void finalize() {
		if (unlock) {
			HeapUnlock(entryHandle);
		}

		free(symName - len - 1);

		free(this);
	}
}