/*
 * QUSOFT MICROSYSTMES
 * Moka
 * Copyright 2002 Frdric Brown
 */
 
/*
 *  This file is part of Moka API.
 *
 *  Moka API is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Moka API is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Moka API; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package moka.io;

import moka.io.IOStream;

/**
 * This class implements sockets. A socket is an endpoint
 * for communication between two machines.
 * <p>
 * The actual work of the socket is performed by encapsulating
 * the TIOS link functions.
 *
 * @author  Frdric Brown
 * @version 1.0, 2002-10-01
 * @since   MDK1.0a
 */
public class Socket extends IOStream {
	
	/** A record of the INT_HANDLER replaced while opening the stream.*/
	private INT_HANDLER intHandler;
	
	/** The timeout, in 1/20th of second.*/
	public int timeout;
	
    /**
     * Creates an unconnected socket.
     */
	public Socket () {
		super();
		this.timeout = 20;
	}
	
	/**
	 * Opens the socket for reading and writting.
	 */
	public native void open() {
		this->intHandler=GetIntVec(AUTO_INT_1);
		SetIntVec(AUTO_INT_1,DUMMY_HANDLER);
		OSLinkOpen();
		this->opened = TRUE;
		this->error = FALSE;
	}
	
	/**
	 * Closes the socket.
	 */
	public native void close() {
		OSLinkClose();
		SetIntVec(AUTO_INT_1,this->intHandler);
		this->opened = FALSE;
		this->error = FALSE;
	}
    
    /**
     * Writes an 8 bit byte.
     *
     * @param	val the byte value to be written
     */
	public native void writeByte(char val) {
		this->error = LIO_SendData(&val,1);
	}
	
    /**
     * Writes a buffer of bytes up to the specified length.
     *
     * @param	buffer the buffer of bytes to be written
     * @param	len the specified length
     */
	public native void writeBytes(char* buffer, short len) {
		this->error = LIO_SendData(buffer,len);
	}
	
    /**
     * Reads an 8 bit byte.
     *
     * @return	the byte value read
     */
	public native char readByte() {
		char val;
		
		this->error = LIO_RecvData(&val,1,this->timeout);
		
		return val;
	}
	
    /**
     * Copies to a buffer of bytes up to the specified length bytes read from the stream.
     *
     * @param	buffer the buffer of bytes where the bytes will be copied
     * @param	len the specified length
     */
	public native void readBytes(char* buffer, short len) {
		this->error = LIO_RecvData(buffer,len,this->timeout);
	}
}