/*
 * QUSOFT MICROSYSTMES
 * Moka
 * Copyright 2004 Frdric Brown
 */

/*
 *  This file is part of Moka API.
 *
 *  Moka API is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Moka API is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Moka API; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package moka.io;

import moka.lang.Object;
import moka.lang.System;

/**
 * Class <code>RowRead</code> is used to get low-level input from the
 * keyboard. Its features are less powerfull than the ones of the Keyboard class,
 * but it is more efficient. The key static methods provided in this class are
 * realy fast ways to determine if a particular key is pressed.
 *
 * @author  Frdric Brown
 * @version 1.0, 2004-03-11
 * @since   MDK2.2
 */
public final class RowRead {

	private static short RR_R_DIAMOND = (System.CALCULATOR ? ~((short)(1<<0)) : ~((short)(1<<0)));
	private static short RR_C_DIAMOND = (System.CALCULATOR ? (1<<1) : (1<<6));
	private static short RR_R_SHIFT = (System.CALCULATOR ? ~((short)(1<<0)) : ~((short)(1<<0)));
	private static short RR_C_SHIFT = (System.CALCULATOR ? (1<<2) : (1<<5));
	private static short RR_R_2ND   = (System.CALCULATOR ? ~((short)(1<<0)) : ~((short)(1<<0)));
	private static short RR_C_2ND   = (System.CALCULATOR ? (1<<0) : (1<<4));

	private static short RR_R_APPS  = (System.CALCULATOR ? ~((short)(1<<7)) : ~((short)(1<<5)));
	private static short RR_C_APPS  = (System.CALCULATOR ? (1<<6) : (1<<0));

	private static short RR_R_ESC = (System.CALCULATOR ? ~((short)(1<<8)) : ~((short)(1<<6)));
	private static short RR_C_ESC = (System.CALCULATOR ? (1<<6) : (1<<0));

	private static short RR_R_RIGHT = (System.CALCULATOR ? ~((short)(1<<0)) : ~((short)(1<<0)));
	private static short RR_C_RIGHT = (System.CALCULATOR ? (1<<6) : (1<<3));
	private static short RR_R_DOWN  = (System.CALCULATOR ? ~((short)(1<<0)) : ~((short)(1<<0)));
	private static short RR_C_DOWN  = (System.CALCULATOR ? (1<<7) : (1<<2));
	private static short RR_R_LEFT  = (System.CALCULATOR ? ~((short)(1<<0)) : ~((short)(1<<0)));
	private static short RR_C_LEFT  = (System.CALCULATOR ? (1<<4) : (1<<1));
	private static short RR_R_UP    = (System.CALCULATOR ? ~((short)(1<<0)) : ~((short)(1<<0)));
	private static short RR_C_UP    = (System.CALCULATOR ? (1<<5) : (1<<0));
	private static short RR_R_ENTER1 = (System.CALCULATOR ? ~((short)(1<<9)) : ~((short)(1<<1)));
	private static short RR_C_ENTER1 = (System.CALCULATOR ? (1<<1) : (1<<0));
	private static short RR_R_ENTER2 = (System.CALCULATOR ? ~((short)(1<<6)) : ~((short)0));
	private static short RR_C_ENTER2 = (System.CALCULATOR ? (1<<6) : 0);

	private static short RR_R_F1 = (System.CALCULATOR ? ~((short)(1<<6)) : ~((short)(1<<5)));
	private static short RR_C_F1 = (System.CALCULATOR ? (1<<4) : (1<<5));
	private static short RR_R_F2 = (System.CALCULATOR ? ~((short)(1<<4)) : ~((short)(1<<4)));
	private static short RR_C_F2 = (System.CALCULATOR ? (1<<4) : (1<<7));
	private static short RR_R_F3 = (System.CALCULATOR ? ~((short)(1<<2)) : ~((short)(1<<3)));
	private static short RR_C_F3 = (System.CALCULATOR ? (1<<4) : (1<<7));
	private static short RR_R_F4 = (System.CALCULATOR ? ~((short)(1<<9)) : ~((short)(1<<2)));
	private static short RR_C_F4 = (System.CALCULATOR ? (1<<4) : (1<<7));
	private static short RR_R_F5 = (System.CALCULATOR ? ~((short)(1<<7)) : ~((short)(1<<1)));
	private static short RR_C_F5 = (System.CALCULATOR ? (1<<4) : (1<<7));

	private static short RR_R_A = (System.CALCULATOR ? ~((short)(1<<9)) : ~((short)(1<<5)));
	private static short RR_C_A = (System.CALCULATOR ? (1<<2) : (1<<4));
	private static short RR_R_B = (System.CALCULATOR ? ~((short)(1<<5)) : ~((short)(1<<4)));
	private static short RR_C_B = (System.CALCULATOR ? (1<<1) : (1<<4));
	private static short RR_R_C = (System.CALCULATOR ? ~((short)(1<<3)) : ~((short)(1<<3)));
	private static short RR_C_C = (System.CALCULATOR ? (1<<1) : (1<<4));
	private static short RR_R_D = (System.CALCULATOR ? ~((short)(1<<2)) : ~((short)(1<<2)));
	private static short RR_C_D = (System.CALCULATOR ? (1<<2) : (1<<4));
	private static short RR_R_E = (System.CALCULATOR ? ~((short)(1<<2)) : ~((short)(1<<1)));
	private static short RR_C_E = (System.CALCULATOR ? (1<<3) : (1<<4));
	private static short RR_R_F = (System.CALCULATOR ? ~((short)(1<<3)) : ~((short)(1<<5)));
	private static short RR_C_F = (System.CALCULATOR ? (1<<2) : (1<<3));
	private static short RR_R_G = (System.CALCULATOR ? ~((short)(1<<4)) : ~((short)(1<<4)));
	private static short RR_C_G = (System.CALCULATOR ? (1<<2) : (1<<3));
	private static short RR_R_H = (System.CALCULATOR ? ~((short)(1<<5)) : ~((short)(1<<3)));
	private static short RR_C_H = (System.CALCULATOR ? (1<<2) : (1<<3));
	private static short RR_R_I = (System.CALCULATOR ? ~((short)(1<<7)) : ~((short)(1<<2)));
	private static short RR_C_I = (System.CALCULATOR ? (1<<3) : (1<<3));
	private static short RR_R_J = (System.CALCULATOR ? ~((short)(1<<6)) : ~((short)(1<<1)));
	private static short RR_C_J = (System.CALCULATOR ? (1<<2) : (1<<3));
	private static short RR_R_K = (System.CALCULATOR ? ~((short)(1<<7)) : ~((short)(1<<5)));
	private static short RR_C_K = (System.CALCULATOR ? (1<<2) : (1<<2));
	private static short RR_R_L = (System.CALCULATOR ? ~((short)(1<<8)) : ~((short)(1<<4)));
	private static short RR_C_L = (System.CALCULATOR ? (1<<2) : (1<<2));
	private static short RR_R_M = (System.CALCULATOR ? ~((short)(1<<7)) : ~((short)(1<<3)));
	private static short RR_C_M = (System.CALCULATOR ? (1<<1) : (1<<2));
	private static short RR_R_N = (System.CALCULATOR ? ~((short)(1<<6)) : ~((short)(1<<2)));
	private static short RR_C_N = (System.CALCULATOR ? (1<<1) : (1<<2));
	private static short RR_R_O = (System.CALCULATOR ? ~((short)(1<<8)) : ~((short)(1<<1)));
	private static short RR_C_O = (System.CALCULATOR ? (1<<3) : (1<<2));
	private static short RR_R_P = (System.CALCULATOR ? ~((short)(1<<6)) : ~((short)(1<<5)));
	private static short RR_C_P = (System.CALCULATOR ? (1<<7) : (1<<1));
	private static short RR_R_Q = (System.CALCULATOR ? ~((short)(1<<9)) : ~((short)(1<<4)));
	private static short RR_C_Q = (System.CALCULATOR ? (1<<3) : (1<<1));
	private static short RR_R_R = (System.CALCULATOR ? ~((short)(1<<3)) : ~((short)(1<<3)));
	private static short RR_C_R = (System.CALCULATOR ? (1<<3) : (1<<1));
	private static short RR_R_S = (System.CALCULATOR ? ~((short)(1<<1)) : ~((short)(1<<2)));
	private static short RR_C_S = (System.CALCULATOR ? (1<<2) : (1<<1));
	private static short RR_R_T = (System.CALCULATOR ? ~((short)(1<<4)) : ~((short)(1<<2)));
	private static short RR_C_T = (System.CALCULATOR ? (1<<3) : (1<<5));
	private static short RR_R_U = (System.CALCULATOR ? ~((short)(1<<6)) : ~((short)(1<<1)));
	private static short RR_C_U = (System.CALCULATOR ? (1<<3) : (1<<1));
	private static short RR_R_V = (System.CALCULATOR ? ~((short)(1<<4)) : ~((short)(1<<4)));
	private static short RR_C_V = (System.CALCULATOR ? (1<<1) : (1<<0));
	private static short RR_R_W = (System.CALCULATOR ? ~((short)(1<<1)) : ~((short)(1<<3)));
	private static short RR_C_W = (System.CALCULATOR ? (1<<3) : (1<<0));
	private static short RR_R_X = (System.CALCULATOR ? ~((short)(1<<2)) : ~((short)(1<<5)));
	private static short RR_C_X = (System.CALCULATOR ? (1<<1) : (1<<5));
	private static short RR_R_Y = (System.CALCULATOR ? ~((short)(1<<5)) : ~((short)(1<<4)));
	private static short RR_C_Y = (System.CALCULATOR ? (1<<3) : (1<<5));
	private static short RR_R_Z = (System.CALCULATOR ? ~((short)(1<<1)) : ~((short)(1<<3)));
	private static short RR_C_Z = (System.CALCULATOR ? (1<<1) : (1<<5));

	private static INT_HANDLER save_int_1;
	private static INT_HANDLER save_int_5;

	/**
	 * Enable RowRead operation. Saves then disable interupt handlers 1 and 5.
	 */
	public static native void enable() {
		RowRead_save_int_1 = GetIntVec (AUTO_INT_1);
		RowRead_save_int_5 = GetIntVec (AUTO_INT_5);
		SetIntVec (AUTO_INT_1, DUMMY_HANDLER);
		SetIntVec (AUTO_INT_5, DUMMY_HANDLER);
	}

	/**
	 * Disable RowRead operation. Restores interupt handlers 1 and 5.
	 */
	public static native void disable() {
		SetIntVec (AUTO_INT_1, RowRead_save_int_1);
		SetIntVec (AUTO_INT_5, RowRead_save_int_5);
	}

	/**
	 * Handle low level keyboard reading for simultaneous reading of more than one key. This method is hardware dependant, use the
	 * key methods to overcome dependance.
	 *
	 * @return a ushort which all bits (cols) corresponding to keys which are being held down are set.
	 * @param row Setting a bit in row masks the corresponding row of the keyboard from being read, so if row is zero, all rows are read at the same time.
	 */
	public abstract static ushort rowRead ( ushort row ) ;

	/**
	 * Handle low level keyboard reading for single keys. This method is hardware dependant, use the
	 * key methods to overcome dependance.
	 *
	 * @return true if the key is pressed, false otherwize.
	 * @param row The row to read.
	 * @param col The column to read.
	 */
	public static boolean keyTest (short row, short col) {
		return native._rowread(~((short)(1<<row))) & (1<<col);
	}

	/**
	 * Checks if the DIAMOND key is pressed. Value is true if the key is pressed, false otherwize.
	 */
	/* Alias implemented */
	public static final boolean key_Diamnd =  RowRead.RR_R_DIAMOND RowRead.RR_C_DIAMOND;

	/**
	 * Checks if the SHIFT key is pressed. Value is true if the key is pressed, false otherwize.
	 */
	/* Alias implemented */
	public static final boolean key_Shift = RowRead.RR_R_SHIFT RowRead.RR_C_SHIFT;

	/**
	 * Checks if the 2ND key is pressed. Value is true if the key is pressed, false otherwize.
	 */
	/* Alias implemented */
	public static final boolean key_2nd =  RowRead.RR_R_2ND RowRead.RR_C_2ND;

	/**
	 * Checks if the APPS key is pressed. Value is true if the key is pressed, false otherwize.
	 */
	/* Alias implemented */
	public static final boolean key_APPS =  RowRead.RR_R_APPS RowRead.RR_C_APPS;

	/**
	 * Checks if the ESC key is pressed. Value is true if the key is pressed, false otherwize.
	 */
	/* Alias implemented */
	public static final boolean key_ESC = RowRead.RR_R_ESC RowRead.RR_C_ESC;

	/**
	 * Checks if the RIGHT key is pressed. Value is true if the key is pressed, false otherwize.
	 */
	/* Alias implemented */
	public static final boolean key_Right = RowRead.RR_R_RIGHT RowRead.RR_C_RIGHT;

	/**
	 * Checks if the DOWN key is pressed. Value is true if the key is pressed, false otherwize.
	 */
	/* Alias implemented */
	public static final boolean key_Down = RowRead.RR_R_DOWN RowRead.RR_C_DOWN;

	/**
	 * Checks if the LEFT key is pressed. Value is true if the key is pressed, false otherwize.
	 */
	/* Alias implemented */
	public static final boolean key_Left = RowRead.RR_R_LEFT RowRead.RR_C_LEFT;

	/**
	 * Checks if the UP key is pressed. Value is true if the key is pressed, false otherwize.
	 */
	/* Alias implemented */
	public static final boolean key_Up = RowRead.RR_R_UP RowRead.RR_C_UP;

	/**
	 * Checks if the ENTER key is pressed. Value is true if the key is pressed, false otherwize.
	 */
	/* Alias implemented */
	public static final boolean key_ENTER = RowRead.key_ENTER1 RowRead.key_ENTER2;

	/**
	 * Checks if the ENTER1 key is pressed. Value is true if the key is pressed, false otherwize.
	 */
	/* Alias implemented */
	public static final boolean key_ENTER1 = RowRead.RR_R_ENTER1 RowRead.RR_C_ENTER1;

	/**
	 * Checks if the ENTER2 key (doesn't exist on TI-89) is pressed. Value is true if the key is pressed, false otherwize.
	 */
	/* Alias implemented */
	public static final boolean key_ENTER2 = RowRead.RR_R_ENTER2 RowRead.RR_C_ENTER2;

	/**
	 * Checks if the F1 key is pressed. Value is true if the key is pressed, false otherwize.
	 */
	/* Alias implemented */
	public static final boolean key_F1 = RowRead.RR_R_F1 RowRead.RR_C_F1;

	/**
	 * Checks if the F2 key is pressed. Value is true if the key is pressed, false otherwize.
	 */
	/* Alias implemented */
	public static final boolean key_F2= RowRead.RR_R_F2 RowRead.RR_C_F2;

	/**
	 * Checks if the F3 key is pressed. Value is true if the key is pressed, false otherwize.
	 */
	/* Alias implemented */
	public static final boolean key_F3 = RowRead.RR_R_F3 RowRead.RR_C_F3;

	/**
	 * Checks if the F4 key is pressed. Value is true if the key is pressed, false otherwize.
	 */
	/* Alias implemented */
	public static final boolean key_F4 = RowRead.RR_R_F4 RowRead.RR_C_F4;

	/**
	 * Checks if the F5 key is pressed. Value is true if the key is pressed, false otherwize.
	 */
	/* Alias implemented */
	public static final boolean key_F5 = RowRead.RR_R_F5 RowRead.RR_C_F5;

	/**
	 * Checks if the A key is pressed. Value is true if the key is pressed, false otherwize.
	 */
	/* Alias implemented */
	public static final boolean key_A = RowRead.RR_R_A RowRead.RR_C_A;
	/**
	 * Checks if the B key is pressed. Value is true if the key is pressed, false otherwize.
	 */
	/* Alias implemented */
	public static final boolean key_B = RowRead.RR_R_B RowRead.RR_C_B;
	/**
	 * Checks if the C key is pressed. Value is true if the key is pressed, false otherwize.
	 */
	/* Alias implemented */
	public static final boolean key_C = RowRead.RR_R_C RowRead.RR_C_C;
	/**
		 * Checks if the D key is pressed. Value is true if the key is pressed, false otherwize.
		 */
		/* Alias implemented */
	public static final boolean key_D = RowRead.RR_R_D RowRead.RR_C_D;
	/**
		 * Checks if the E key is pressed. Value is true if the key is pressed, false otherwize.
		 */
		/* Alias implemented */
	public static final boolean key_E = RowRead.RR_R_E RowRead.RR_C_E;
	/**
		 * Checks if the F key is pressed. Value is true if the key is pressed, false otherwize.
		 */
		/* Alias implemented */
	public static final boolean key_F = RowRead.RR_R_F RowRead.RR_C_F;
	/**
		 * Checks if the G key is pressed. Value is true if the key is pressed, false otherwize.
		 */
		/* Alias implemented */
	public static final boolean key_G = RowRead.RR_R_G RowRead.RR_C_G;
	/**
		 * Checks if the H key is pressed. Value is true if the key is pressed, false otherwize.
		 */
		/* Alias implemented */
	public static final boolean key_H = RowRead.RR_R_H RowRead.RR_C_H;
	/**
		 * Checks if the I key is pressed. Value is true if the key is pressed, false otherwize.
		 */
		/* Alias implemented */
	public static final boolean key_I = RowRead.RR_R_I RowRead.RR_C_I;
	/**
		 * Checks if the J key is pressed. Value is true if the key is pressed, false otherwize.
		 */
		/* Alias implemented */
	public static final boolean key_J = RowRead.RR_R_J RowRead.RR_C_J;
	/**
		 * Checks if the K key is pressed. Value is true if the key is pressed, false otherwize.
		 */
		/* Alias implemented */
	public static final boolean key_K = RowRead.RR_R_K RowRead.RR_C_K;
	/**
		 * Checks if the L key is pressed. Value is true if the key is pressed, false otherwize.
		 */
		/* Alias implemented */
	public static final boolean key_L = RowRead.RR_R_L RowRead.RR_C_L;
	/**
		 * Checks if the M key is pressed. Value is true if the key is pressed, false otherwize.
		 */
		/* Alias implemented */
	public static final boolean key_M = RowRead.RR_R_M RowRead.RR_C_M;
	/**
		 * Checks if the N key is pressed. Value is true if the key is pressed, false otherwize.
		 */
		/* Alias implemented */
	public static final boolean key_N = RowRead.RR_R_N RowRead.RR_C_N;
	/**
		 * Checks if the O key is pressed. Value is true if the key is pressed, false otherwize.
		 */
		/* Alias implemented */
	public static final boolean key_O = RowRead.RR_R_O RowRead.RR_C_O;
	/**
		 * Checks if the P key is pressed. Value is true if the key is pressed, false otherwize.
		 */
		/* Alias implemented */
	public static final boolean key_P = RowRead.RR_R_P RowRead.RR_C_P;
	/**
		 * Checks if the Q key is pressed. Value is true if the key is pressed, false otherwize.
		 */
		/* Alias implemented */
	public static final boolean key_Q = RowRead.RR_R_Q RowRead.RR_C_Q;
	/**
		 * Checks if the R key is pressed. Value is true if the key is pressed, false otherwize.
		 */
		/* Alias implemented */
	public static final boolean key_R = RowRead.RR_R_R RowRead.RR_C_R;
	/**
		 * Checks if the S key is pressed. Value is true if the key is pressed, false otherwize.
		 */
		/* Alias implemented */
	public static final boolean key_S = RowRead.RR_R_S RowRead.RR_C_S;
	/**
		 * Checks if the T key is pressed. Value is true if the key is pressed, false otherwize.
		 */
		/* Alias implemented */
	public static final boolean key_T = RowRead.RR_R_T RowRead.RR_C_T;
	/**
		 * Checks if the U key is pressed. Value is true if the key is pressed, false otherwize.
		 */
		/* Alias implemented */
	public static final boolean key_U = RowRead.RR_R_U RowRead.RR_C_U;
	/**
		 * Checks if the V key is pressed. Value is true if the key is pressed, false otherwize.
		 */
		/* Alias implemented */
	public static final boolean key_V = RowRead.RR_R_V RowRead.RR_C_V;
	/**
		 * Checks if the W key is pressed. Value is true if the key is pressed, false otherwize.
		 */
		/* Alias implemented */
	public static final boolean key_W = RowRead.RR_R_W RowRead.RR_C_W;
	/**
		 * Checks if the X key is pressed. Value is true if the key is pressed, false otherwize.
		 */
		/* Alias implemented */
	public static final boolean key_X = RowRead.RR_R_X RowRead.RR_C_X;
	/**
		 * Checks if the Y key is pressed. Value is true if the key is pressed, false otherwize.
		 */
		/* Alias implemented */
	public static final boolean key_Y = RowRead.RR_R_Y RowRead.RR_C_Y;
	/**
		 * Checks if the Z key is pressed. Value is true if the key is pressed, false otherwize.
		 */
		/* Alias implemented */
	public static final boolean key_Z = RowRead.RR_R_Z RowRead.RR_C_Z;
	/**
	 * Checks if ANY key is pressed. Value is true if any key is pressed, false otherwize.
	 */
	/* Alias implemented */
	public static final boolean key_ANY = 0;
}