/*
 * QUSOFT MICROSYSTMES
 * Moka
 * Copyright 2002 Frdric Brown
 */

/*
 *  This file is part of Moka API.
 *
 *  Moka API is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Moka API is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Moka API; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package moka.io;

import moka.lang.Object;
import moka.lang.System;

/**
 * Class <code>Keyboard</code> is used to get low-level input from the
 * keyboard. Note that some methods has a static version. These methods are written with
 * an uppercase letter in this document to differenciate them from their instance counterparts, but they must be write with
 * a lowercase in a program. Ex: Keyboard.ngetchx() (Althrought it is written Ngetchx in this doccument)
 *
 * @author  Frdric Brown
 * @version 1.1, 2002-10-25
 * @since   MDK1.0a
 */
public class Keyboard {
	/** The number of key the keyboard class handle. */
	public static final short KEY_COUNT = 43;
	/** The number of system key the keyboard class handle. */
	public static final short SYSTEM_KEY_COUNT = 17;
	/** The number of alphabetic key the keyboard class handle. */
	public static final short ALPHA_KEY_COUNT = 26;
	
	/** The F1 key. */
	public static final short KEY_F1 = 268;
	/** The F2 key. */
	public static final short KEY_F2 = 269;
	/** The F3 key. */
	public static final short KEY_F3 = 270;
	/** The F4 key. */
	public static final short KEY_F4 = 271;
	/** The F5 key. */
	public static final short KEY_F5 = 272;
	/** The F6 key. */
	public static final short KEY_F6 = 273;
	/** The F7 key. */
	public static final short KEY_F7 = 274;
	/** The F8 key. */
	public static final short KEY_F8 = 275;
	/** The ESC key. */
	public static final short KEY_ESC = 264;
	/** The QUIT key. */
	public static final short KEY_QUIT = 4360;
	/** The APPS key. */
	public static final short KEY_APPS = 265;
	/** The SWITCH key. */
	public static final short KEY_SWITCH = 4361;
	/** The MODE key. */
	public static final short KEY_MODE = 266;
	/** The BACKSPACE key. */
	public static final short KEY_BACKSPACE = native.KEY_BACKSPACE;
	/** The INS key. */
	public static final short KEY_INS = native.KEY_INS;
	/** The CLEAR key. */
	public static final short KEY_CLEAR = native.KEY_CLEAR;
	/** The VARLNK key. */
	public static final short KEY_VARLNK = native.KEY_VARLNK;
	/** The CHAR key. */
	public static final short KEY_CHAR = native.KEY_CHAR;
	/** The ENTER key. */
	public static final short KEY_ENTER = native.KEY_ENTER;
	/** The ENTRY key. */
	public static final short KEY_ENTRY = native.KEY_ENTRY;	
	/** The STO key. */
	public static final short KEY_STO = native.KEY_STO;
	/** The RCL key. */
	public static final short KEY_RCL = native.KEY_RCL;
	/** The SIGN key. */
	public static final short KEY_SIGN = native.KEY_SIGN;
	/** The MATH key. */
	public static final short KEY_MATH = native.KEY_MATH;
	/** The MEM key. */
	public static final short KEY_MEM = native.KEY_MEM;
	/** The ON key. */
	public static final short KEY_ON = native.KEY_ON;
	/** The OFF key. */
	public static final short KEY_OFF = native.KEY_OFF;
	
	/* Various flags to indicate which keys are pressed.*/
	public byte key_Diamnd;
	public byte key_Shift;
	public byte key_2nd;
	public byte key_APPS;
	public byte key_ESC;
	public byte key_Right;
	public byte key_Down;
	public byte key_Left;
	public byte key_Up;
	public byte key_ENTER;
	public byte key_ENTER1;
	public byte key_ENTER2;
	public byte key_F1;
	public byte key_F2;
	public byte key_F3;
	public byte key_F4;
	public byte key_F5;
	public byte key_A;
	public byte key_B;
	public byte key_C;
	public byte key_D;
	public byte key_E;
	public byte key_F;
	public byte key_G;
	public byte key_H;
	public byte key_I;
	public byte key_J;
	public byte key_K;
	public byte key_L;
	public byte key_M;
	public byte key_N;
	public byte key_O;
	public byte key_P;
	public byte key_Q;
	public byte key_R;
	public byte key_S;
	public byte key_T;
	public byte key_U;
	public byte key_V;
	public byte key_W;
	public byte key_X;
	public byte key_Y;
	public byte key_Z;
	/*public byte key_0;
	public byte key_1;
	public byte key_2;
	public byte key_3;
	public byte key_4;
	public byte key_5;
	public byte key_6;
	public byte key_7;
	public byte key_8;
	public byte key_9;*/

	/**
	 * Initializes a newly allocated Keyboard.
	 */
	public Keyboard () {
		this.checkKeys();
	}

	/**
	 * Checks if any "system" key is pressed.
	 */
	public native void checkSysKeys () {
		INT_HANDLER save_int_1 = GetIntVec (AUTO_INT_1);
		INT_HANDLER save_int_5 = GetIntVec (AUTO_INT_5);

		memset(&this->key_Diamnd, 0, Keyboard_SYSTEM_KEY_COUNT);

		SetIntVec (AUTO_INT_1, DUMMY_HANDLER);
		SetIntVec (AUTO_INT_5, DUMMY_HANDLER);

		if (! System_CALCULATOR) {//If calculator == TI-89
			if (_rowread(~((short)(1<<0))) & (1<<6)) {
				this->key_Diamnd = 1;
			}
			if (_rowread(~((short)(1<<0))) & (1<<5)) {
				this->key_Shift = 1;
			}
			if (_rowread(~((short)(1<<0))) & (1<<4)) {
				this->key_2nd = 1;
			}
			
			if (_rowread(~((short)(1<<5))) & (1<<0)) {
				this->key_APPS = 1;
			}
			if (_rowread(~((short)(1<<6))) & (1<<0)) {
				this->key_ESC = 1;
			}
			if (_rowread(~((short)(1<<0))) & (1<<3)) {
				this->key_Right = 1;
			}
			if (_rowread(~((short)(1<<0))) & (1<<2)) {
				this->key_Down = 1;
			}
			if (_rowread(~((short)(1<<0))) & (1<<1)) {
				this->key_Left = 1;
			}
			if (_rowread(~((short)(1<<0))) & (1<<0)) {
				this->key_Up = 1;
			}
			if (_rowread(~((short)(1<<1))) & (1<<0)) {
				this->key_ENTER = 1;
			}
			
			if (_rowread(~((short)(1<<5))) & (1<<7)) {
				this->key_F1 = 1;
			}
			if (_rowread(~((short)(1<<4))) & (1<<7)) {
				this->key_F2 = 1;
			}
			if (_rowread(~((short)(1<<3))) & (1<<7)) {
				this->key_F3 = 1;
			}
			if (_rowread(~((short)(1<<2))) & (1<<7)) {
				this->key_F4 = 1;
			}
			if (_rowread(~((short)(1<<1))) & (1<<7)) {
				this->key_F5 = 1;
			}
		}
		else {
			if (_rowread(~((short)(1<<0))) & (1<<1)) {
				this->key_Diamnd = 1;
			}
			if (_rowread(~((short)(1<<0))) & (1<<2)) {
				this->key_Shift = 1;
			}
			if (_rowread(~((short)(1<<0))) & (1<<0)) {
				this->key_2nd = 1;
			}
			
			if (_rowread(~((short)(1<<7))) & (1<<6)) {
				this->key_APPS = 1;
			}
			if (_rowread(~((short)(1<<8))) & (1<<6)) {
				this->key_ESC = 1;
			}
			if (_rowread(~((short)(1<<0))) & (1<<6)) {
				this->key_Right = 1;
			}
			if (_rowread(~((short)(1<<0))) & (1<<7)) {
				this->key_Down = 1;
			}
			if (_rowread(~((short)(1<<0))) & (1<<4)) {
				this->key_Left = 1;
			}
			if (_rowread(~((short)(1<<0))) & (1<<5)) {
				this->key_Up = 1;
			}
			if (_rowread(~((short)(1<<9))) & (1<<1)) {
				this->key_ENTER = 1;
				this->key_ENTER1 = 1;
			}
			if (_rowread(~((short)(1<<6))) & (1<<6)) {
				this->key_ENTER = 1;
				this->key_ENTER2 = 1;
			}
			
			if (_rowread(~((short)(1<<6))) & (1<<4)) {
				this->key_F1 = 1;
			}
			if (_rowread(~((short)(1<<4))) & (1<<4)) {
				this->key_F2 = 1;
			}
			if (_rowread(~((short)(1<<2))) & (1<<4)) {
				this->key_F3 = 1;
			}
			if (_rowread(~((short)(1<<9))) & (1<<4)) {
				this->key_F4 = 1;
			}
			if (_rowread(~((short)(1<<7))) & (1<<4)) {
				this->key_F5 = 1;
			}
		}

		SetIntVec (AUTO_INT_1, save_int_1);
		SetIntVec (AUTO_INT_5, save_int_5);
	}
	
	/**
	 * Wait for the user to press a key, then return the key.
	 *
	 * @return the key pressed
	 */
	public short ngetchx() {
		this.checkKeys();

		while (!this.hit()) {
			this.checkKeys();
		}

		return this.getKey();
	}

	/**
	 * Wait for the user to press a key, then return the key.
	 *
	 * @return the key pressed
	 */
	public static short ngetchx() {
		short key;
		while (native.OSdequeue (&key, kbd_queue ())) ;

		return key;
	}

	/**
	 * Wait a specified delay for the user to press a key, then return the key.
	 *
	 * @return the key actually pressed or null if no key is pressed.
	 * @param delay The delay
	 */
	public static short ngetchx (int delay) {
		short keyTmp = 0;

		OSFreeTimer (BATT_TIMER);
		OSRegisterTimer (BATT_TIMER, delay);
		while (!OSTimerExpired (BATT_TIMER)) {
			if (!OSdequeue (&keyTmp, kbd_queue ())) {
				return keyTmp;
			}
		}

		return 0;
	}

	/**
	 * Checks to see if a keystroke is currently available. If so
	 * returns a nonzero integer (in fact, it returns the exactly same value as ngetchx); if not, it returns 0.
	 *
	 * @return a nonzero integer or 0 if no keystroke is currently available.
	 */
	public static short getKey() {
		short key;
		
		return (native.OSdequeue (&key, kbd_queue()) ? key : 0);
	}

	/**
	 * Checks if any key of the keyboard is pressed.
	 */
	public native void checkKeys () {
		INT_HANDLER save_int_1 = GetIntVec (AUTO_INT_1);
		INT_HANDLER save_int_5 = GetIntVec (AUTO_INT_5);

		memset(&this->key_Diamnd, 0, Keyboard_KEY_COUNT);

		SetIntVec (AUTO_INT_1, DUMMY_HANDLER);
		SetIntVec (AUTO_INT_5, DUMMY_HANDLER);

		if (! System_CALCULATOR) {//If calculator == TI-89
			if (_rowread(~((short)(1<<0))) & (1<<6)) {
				this->key_Diamnd = 1;
			}
			if (_rowread(~((short)(1<<0))) & (1<<5)) {
				this->key_Shift = 1;
			}
			if (_rowread(~((short)(1<<0))) & (1<<4)) {
				this->key_2nd = 1;
			}
			
			if (_rowread(~((short)(1<<5))) & (1<<0)) {
				this->key_APPS = 1;
			}
			if (_rowread(~((short)(1<<6))) & (1<<0)) {
				this->key_ESC = 1;
			}
			if (_rowread(~((short)(1<<0))) & (1<<3)) {
				this->key_Right = 1;
			}
			if (_rowread(~((short)(1<<0))) & (1<<2)) {
				this->key_Down = 1;
			}
			if (_rowread(~((short)(1<<0))) & (1<<1)) {
				this->key_Left = 1;
			}
			if (_rowread(~((short)(1<<0))) & (1<<0)) {
				this->key_Up = 1;
			}
			if (_rowread(~((short)(1<<1))) & (1<<0)) {
				this->key_ENTER = 1;
			}

			if (_rowread(~((short)(1<<5))) & (1<<7)) {
				this->key_F1 = 1;
			}
			if (_rowread(~((short)(1<<4))) & (1<<7)) {
				this->key_F2 = 1;
			}
			if (_rowread(~((short)(1<<3))) & (1<<7)) {
				this->key_F3 = 1;
			}
			if (_rowread(~((short)(1<<2))) & (1<<7)) {
				this->key_F4 = 1;
			}
			if (_rowread(~((short)(1<<1))) & (1<<7)) {
				this->key_F5 = 1;
			}

			if (_rowread(~((short)(1<<5))) & (1<<4)) {
				this->key_A = 1;
			}
			if (_rowread(~((short)(1<<4))) & (1<<4)) {
				this->key_B = 1;
			}
			if (_rowread(~((short)(1<<3))) & (1<<4)) {
				this->key_C = 1;
			}
			if (_rowread(~((short)(1<<2))) & (1<<4)) {
				this->key_D = 1;
			}
			if (_rowread(~((short)(1<<1))) & (1<<4)) {
				this->key_E = 1;
			}
			if (_rowread(~((short)(1<<5))) & (1<<3)) {
				this->key_F = 1;
			}
			if (_rowread(~((short)(1<<4))) & (1<<3)) {
				this->key_G = 1;
			}
			if (_rowread(~((short)(1<<3))) & (1<<3)) {
				this->key_H = 1;
			}
			if (_rowread(~((short)(1<<2))) & (1<<3)) {
				this->key_I = 1;
			}
			if (_rowread(~((short)(1<<1))) & (1<<3)) {
				this->key_J = 1;
			}
			if (_rowread(~((short)(1<<5))) & (1<<2)) {
				this->key_K = 1;
			}
			if (_rowread(~((short)(1<<4))) & (1<<2)) {
				this->key_L = 1;
			}
			if (_rowread(~((short)(1<<3))) & (1<<2)) {
				this->key_M = 1;
			}
			if (_rowread(~((short)(1<<2))) & (1<<2)) {
				this->key_N = 1;
			}
			if (_rowread(~((short)(1<<1))) & (1<<2)) {
				this->key_O = 1;
			}
			if (_rowread(~((short)(1<<5))) & (1<<1)) {
				this->key_P = 1;
			}
			if (_rowread(~((short)(1<<4))) & (1<<1)) {
				this->key_Q = 1;
			}
			if (_rowread(~((short)(1<<3))) & (1<<1)) {
				this->key_R = 1;
			}
			if (_rowread(~((short)(1<<2))) & (1<<1)) {
				this->key_S = 1;
			}
			if (_rowread(~((short)(1<<2))) & (1<<5)) {
				this->key_T = 1;
			}
			if (_rowread(~((short)(1<<1))) & (1<<1)) {
				this->key_U = 1;
			}
			if (_rowread(~((short)(1<<4))) & (1<<0)) {
				this->key_V = 1;
			}
			if (_rowread(~((short)(1<<3))) & (1<<0)) {
				this->key_W = 1;
			}
			if (_rowread(~((short)(1<<5))) & (1<<5)) {
				this->key_X = 1;
			}
			if (_rowread(~((short)(1<<4))) & (1<<5)) {
				this->key_Y = 1;
			}
			if (_rowread(~((short)(1<<3))) & (1<<5)) {
				this->key_Z = 1;
			}
		}
		else {
			if (_rowread(~((short)(1<<0))) & (1<<1)) {
				this->key_Diamnd = 1;
			}
			if (_rowread(~((short)(1<<0))) & (1<<2)) {
				this->key_Shift = 1;
			}
			if (_rowread(~((short)(1<<0))) & (1<<0)) {
				this->key_2nd = 1;
			}
			
			if (_rowread(~((short)(1<<7))) & (1<<6)) {
				this->key_APPS = 1;
			}
			if (_rowread(~((short)(1<<8))) & (1<<6)) {
				this->key_ESC = 1;
			}
			if (_rowread(~((short)(1<<0))) & (1<<6)) {
				this->key_Right = 1;
			}
			if (_rowread(~((short)(1<<0))) & (1<<7)) {
				this->key_Down = 1;
			}
			if (_rowread(~((short)(1<<0))) & (1<<4)) {
				this->key_Left = 1;
			}
			if (_rowread(~((short)(1<<0))) & (1<<5)) {
				this->key_Up = 1;
			}
			if (_rowread(~((short)(1<<9))) & (1<<1)) {
				this->key_ENTER = 1;
				this->key_ENTER1 = 1;
			}
			if (_rowread(~((short)(1<<6))) & (1<<6)) {
				this->key_ENTER = 1;
				this->key_ENTER2 = 1;
			}
			
			if (_rowread(~((short)(1<<6))) & (1<<4)) {
				this->key_F1 = 1;
			}
			if (_rowread(~((short)(1<<4))) & (1<<4)) {
				this->key_F2 = 1;
			}
			if (_rowread(~((short)(1<<2))) & (1<<4)) {
				this->key_F3 = 1;
			}
			if (_rowread(~((short)(1<<9))) & (1<<4)) {
				this->key_F4 = 1;
			}
			if (_rowread(~((short)(1<<7))) & (1<<4)) {
				this->key_F5 = 1;
			}

			if (_rowread(~((short)(1<<9))) & (1<<2)) {
				this->key_A = 1;
			}
			if (_rowread(~((short)(1<<5))) & (1<<1)) {
				this->key_B = 1;
			}
			if (_rowread(~((short)(1<<3))) & (1<<1)) {
				this->key_C = 1;
			}
			if (_rowread(~((short)(1<<2))) & (1<<2)) {
				this->key_D = 1;
			}
			if (_rowread(~((short)(1<<2))) & (1<<3)) {
				this->key_E = 1;
			}
			if (_rowread(~((short)(1<<3))) & (1<<2)) {
				this->key_F = 1;
			}
			if (_rowread(~((short)(1<<4))) & (1<<2)) {
				this->key_G = 1;
			}
			if (_rowread(~((short)(1<<5))) & (1<<2)) {
				this->key_H = 1;
			}
			if (_rowread(~((short)(1<<7))) & (1<<3)) {
				this->key_I = 1;
			}
			if (_rowread(~((short)(1<<6))) & (1<<2)) {
				this->key_J = 1;
			}
			if (_rowread(~((short)(1<<7))) & (1<<2)) {
				this->key_K = 1;
			}
			if (_rowread(~((short)(1<<8))) & (1<<2)) {
				this->key_L = 1;
			}
			if (_rowread(~((short)(1<<7))) & (1<<1)) {
				this->key_M = 1;
			}
			if (_rowread(~((short)(1<<6))) & (1<<1)) {
				this->key_N = 1;
			}
			if (_rowread(~((short)(1<<8))) & (1<<3)) {
				this->key_O = 1;
			}
			if (_rowread(~((short)(1<<6))) & (1<<7)) {
				this->key_P = 1;
			}
			if (_rowread(~((short)(1<<9))) & (1<<3)) {
				this->key_Q = 1;
			}
			if (_rowread(~((short)(1<<3))) & (1<<3)) {
				this->key_R = 1;
			}
			if (_rowread(~((short)(1<<1))) & (1<<2)) {
				this->key_S = 1;
			}
			if (_rowread(~((short)(1<<4))) & (1<<3)) {
				this->key_T = 1;
			}
			if (_rowread(~((short)(1<<6))) & (1<<3)) {
				this->key_U = 1;
			}
			if (_rowread(~((short)(1<<4))) & (1<<1)) {
				this->key_V = 1;
			}
			if (_rowread(~((short)(1<<1))) & (1<<3)) {
				this->key_W = 1;
			}
			if (_rowread(~((short)(1<<2))) & (1<<1)) {
				this->key_X = 1;
			}
			if (_rowread(~((short)(1<<5))) & (1<<3)) {
				this->key_Y = 1;
			}
			if (_rowread(~((short)(1<<1))) & (1<<1)) {
				this->key_Z = 1;
			}
		}

		SetIntVec (AUTO_INT_1, save_int_1);
		SetIntVec (AUTO_INT_5, save_int_5);
	}

	/**
	 * Returns the first (in alphabetic order) of the alphabetic keys pressed.
	 * If no such key is pressed, the null character is returned.
	 *
	 * @return the first alphabetic key pressed or the null character if no such key is pressed.
	 */
	public native char readAlpha () {
		char c = '\0';
		void * begin = &this->key_A;
		void * result;

		if ((result = memchr(begin, 1, Keyboard_ALPHA_KEY_COUNT))) {
			c = 'A' + (char)((char*)result - (char*)begin);
		}

		return c;
	}

	/**
	 * Checks if a key is pressed.
	 *
	 * @return true if any key is pressed, false otherwize.
	 */
	public native boolean hit () {
		if (memchr(&this->key_APPS, 1, Keyboard_KEY_COUNT)) {
			return TRUE;
		}

		return FALSE;
	}

	/**
	 * Checks if a key stroke is available.
	 *
	 * @return true if a key stroke is available, false otherwize.
	 */
	/*Alias implemented for kbhit ().*/
	public abstract static native boolean kbhit () ;
	/*public static native boolean kbhit () {
		return kbhit();
	}*/

	/**
	 * Flushes the keyboard queue.
	 */
	/*Alias implemented for kbhit ().*/
	public abstract native static void flush () ;
	/*public native static void flush () {
		GKeyFlush();
	}*/

	/**
	 * Returns the key pressed in a specified delay. If two keys are pressed, the last key pressed is returned.
	 * If no key is pressed during the specified delay, the null character ('\0') is returned.
	 *
	 * @return the key actually pressed or null if no key is pressed.
	 * @param delay The delay
	 */
	public native static short getKey (int delay) {
		short int key = 0;
		short int keyTmp = 0;

		OSFreeTimer (BATT_TIMER);
		OSRegisterTimer (BATT_TIMER, delay);
		while (!OSTimerExpired (BATT_TIMER)) {
			if (!OSdequeue (&keyTmp, kbd_queue ())) {
				key = keyTmp;
			}
		}

		return key;
	}

	/**
	 * Returns the key actually pressed. If two keys are pressed, the one with the lesser value is returned.
	 * If no key is pressed, the null character is returned.
	 *
	 * @return the key actually pressed or null if no key is pressed.
	 */
	public native short getKey () {
		char c = '\0';

		if ((c = this->readAlpha_(this))) {
			return c;
		}

		//13
		if (this->key_ENTER) {
			return KEY_ENTER;
		}

		//264
		if (this->key_ESC) {
			return KEY_ESC;
		}
		//265
		if (this->key_APPS) {
			return KEY_APPS;
		}
		
		if (this->key_Diamnd) {
			return System_KEY_DIAMOND;
		}
		
		if (this->key_Shift) {
			return KEY_SHIFT;
		}
		
		/*if (this->key_2nd) {
			return ???;
		}*/

		if (this->key_Right) {
			return System_KEY_RIGHT;
		}
		if (this->key_Down) {
			return System_KEY_DOWN;
		}
		if (this->key_Left) {
			return System_KEY_LEFT;
		}
		if (this->key_Up) {
			return System_KEY_UP;
		}
		
		if (this->key_F1) {
			return KEY_F1;
		}
		if (this->key_F2) {
			return KEY_F2;
		}
		if (this->key_F3) {
			return KEY_F3;
		}
		if (this->key_F4) {
			return KEY_F4;
		}
		if (this->key_F5) {
			return KEY_F5;
		}

		return c;
	}
}