/*
 * QUSOFT MICROSYSTMES
 * Moka
 * Copyright 2002 Frdric Brown
 */
 
/*
 *  This file is part of Moka API.
 *
 *  Moka API is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Moka API is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Moka API; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package moka.io;

import moka.lang.Object;
import moka.io.Serializable;

/**
 * This class represents an Input/Output data stream.
 *
 * @author  Frdric Brown
 * @version 1.0, 2002-10-01
 * @since   MDK1.0a
 */
public abstract class IOStream {
	
	/** Is the stream opened ? Should be only read.*/
	public boolean opened;
	/** Is the stream in error ? Should be only read.*/
	public boolean error;
	
    /**
     * Creates an IOStream.
     */
	public IOStream () {
		this.opened = false;
		this.error = false;
	}
	
	/**
	 * Opens the stream.
	 * Since this method is an abstract one, subclasses of IOStream
	 * must overwrite this method.
     *
	 */
	public abstract void open();
	
	/**
	 * Closes the stream.
	 * Since this method is an abstract one, subclasses of IOStream
	 * must overwrite this method.
     *
	 */
	public abstract void close();
    
    /**
     * Writes an 8 bit byte.
	 * Since this method is an abstract one, subclasses of IOStream
	 * must overwrite this method.
     *
     * @param	val the byte value to be written
     */
	public abstract void writeByte(char val);
	
    /**
     * Writes a buffer of bytes up to the specified length.
	 * Since this method is an abstract one, subclasses of IOStream
	 * must overwrite this method.
     *
     * @param	buffer the buffer of bytes to be written
     * @param	len the specified length
     */
	public abstract void writeBytes(char* buffer, short len);
	
    /**
     * Reads an 8 bit byte.
	 * Since this method is an abstract one, subclasses of IOStream
	 * must overwrite this method.
     *
     * @return	the byte value read
     */
	public abstract char readByte();
	
    /**
     * Copies to a buffer of bytes up to the specified length bytes read from the stream.
	 * Since this method is an abstract one, subclasses of IOStream
	 * must overwrite this method.
     *
     * @param	buffer the buffer of bytes where the bytes will be copied
     * @param	len the specified length
     */
	public abstract void readBytes(char* buffer, short len);

    /**
     * Frees the memory of system ressources used by this IOStream
     * object.  If the stream is opened, it will be closed first.
     */
	public void finalize() {
		if (this.opened) {
			this.close();
		}
		
		native {
			free(this);
		}
	}
	
	/**
	 * Writes a boolean.
	 @param val the boolean value to be written
	*/
	public void writeBoolean (boolean val) {
		
		this.writeByte((char)val);
	}
	
	/**
	 * Writes a char.
	 @param val the char value to be written
	*/
	public void writeChar (char val) {
		
		this.writeByte(val);
	}
	
	/**
	 * Writes a double.
	 @param val the double value to be written
	*/
	public native void writeDouble (double val) {
		
		this->writeBytes_char_p_short_int(this, (char*) &val, sizeof(double));
	}
	
	/**
	 * Writes an short.
	 @param val the short value to be written
	*/
	public native void writeShort (short val) {
		
		this->writeBytes_char_p_short_int(this, (char*) &val, sizeof(short));
	}
	
	/**
	 * Writes an int.
	 @param val the int value to be written
	*/
	public native void writeInt (int val) {
		
		this->writeBytes_char_p_short_int(this, (char*) &val, sizeof(long int));
	}
	
	/**
	 * Writes a long.
	 @param val the long value to be written
	*/
	public native void writeLong (long val) {
		
		this->writeBytes_char_p_short_int(this, (char*) &val, sizeof(long long int));
	}
	
	/**
	 * Writes a String object. The specified string is
	 * finalized before the method returns.
	 @param val the String object to be written
	*/
	public void writeString (String val) {
		short len = native { strlen(val->value) };
		
		this.writeShort(len);
	
		this.writeBytes(val.value, len);
		
		val.finalize();
	}
	
	/**
	 * Writes an ANSI string.
	 @param val the ANSI string to be written
	*/
	public void writeChars (char* val) {
		
		short len = native { strlen(val) };
			
		this.writeBytes(val, len);
	}
	
	/**
	 * Writes an Object.  This object must be a
	 * subclass of serializable.
	 @param val the Object to be written
	*/
	public void writeObject (Object val) {
		((Serializable)val).serialize(this);
	}
	
	/**
	 * Reads a boolean.
	 @return the boolean value read
	*/
	public boolean readBoolean () {
		
		boolean val;
		
		val = this.readByte();
		
		return val;
	}
	
	/**
	 * Reads a char.
	 @return the char value read
	*/
	public char readChar () {
		
		char val;
		
		val = this.readByte();
		
		return val;
	}
	
	/**
	 * Reads a double.
	 @return the double value read
	*/
	public native double readDouble () {
		
		double val;
		
		this->readBytes_char_p_short_int(this, (char*) &val, sizeof(double));
		
		return val;
	}
	
	/**
	 * Reads a short.
	 @return the short value read
	*/
	public native short readShort () {
		
		int val;
		
		this->readBytes_char_p_short_int(this, (char*) &val, sizeof(short));
		
		return val;
	}
	
	/**
	 * Reads an int.
	 @return val the int value read
	*/
	public native int readInt () {
		
		long int val;
		
		this->readBytes_char_p_short_int(this, (char*) &val, sizeof(long int));
		
		return val;
	}
	
	/**
	 * Reads a long.
	 @return the long value read
	*/
	public native long readLong () {
		
		long long int val;
		
		this->readBytes_char_p_short_int(this, (char*) &val, sizeof(long long int));
		
		return val;
	}
	
	/**
	 * Reads a String object.
	 @return the String object read
	*/
	public String readString () {
	
		short len = this.readShort();
		char buffer[len];
		
		this.readBytes((char*)buffer, len);
		
		return new String(Character.copyValueOf((char*)buffer, 0, (int)len), false);
	}
	
	/**
	 * Reads an ANSI string.
	 @param len the length of the ANSI string
	 @return the ANSI string read
	*/
	public char* readChars (short len) {
		char buffer[len];
		
		this.readBytes((char*)buffer, len);
		
		return Character.copyValueOf((char*)buffer, 0, (int)len);
	}
	
	/**
	 * Reads a Object.
	 @return the Object read
	*/
	public Object readObject () {
		String name = this.readString();
		Serializable prototype;
		int i = 0;
		
		while (true) {
			native {
				prototype = ProtArray[i++];
				if (!strcmp(prototype->class, name->value)) {
					break;
				}
			}
		}
		
		name.finalize();
		
		return prototype.deserialize(this);
	}
}