/*
 * QUSOFT MICROSYSTMES
 * Moka
 * Copyright 2002 Frdric Brown
 */

/*
 *  This file is part of Moka API.
 *
 *  Moka API is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Moka API is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Moka API; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package moka.io;

import moka.lang.System;
import moka.lang.Character;
import moka.io.Gray;
import moka.io.Graph;

/**
 * The Graphics class encapsulates the display device operations
 * and provides methods to deal with bitmaps. Note that some methods has a static version. These methods are written with
 * an uppercase letter in this document to differenciate them from their instance counterparts, but they must be write with
 * a lowercase in a program. Ex: Graphics.drawTo(10, 10) (Althrought it is written DrawTo in this doccument)
 *
 * @author  Frdric Brown
 * @version 1.1, 2003-10-25
 * @since   MDK1.0a
 */
public class Graphics {

	/** The graphic port address. LCD (addr: 0x4C00,) by default.*/
	public void* addr;

	/** The maximum x coordinate.*/
	public short x_max;

	/** The maximum y coordinate.*/
	public short y_max;

	/** The grayscale level.*/
	//public short level;

	/** Draw with destination pixels turned off.*/
	public static final short REVERSE_ATTR = A_REVERSE;

	/** Draw with destination pixels turned on.*/
	public static final short NORMAL_ATTR = A_NORMAL;

	/** Draw with source pixels XORed with destination pixels.*/
	public static final short XOR_ATTR = A_XOR;

	/** Draw with destination pixels masked so that every other pixel turned off*/
	public static final short SHADED_ATTR = A_SHADED;

	/** Draw with source pixels replace destination pixels.*/
	public static final short REPLACE_ATTR = A_REPLACE;

	/** Draw with source pixels ORed with destination pixels.*/
	public static final short OR_ATTR = A_OR;

	/** AND the region to draw.*/
	public static final short AND_ATTR = A_AND;

	/** Draw a double thick line.*/
	public static final short THICK1_ATTR = A_THICK1;

	/** Draw the line using a vertical shading pattern.*/
	public static final short SHADE_V_ATTR = A_SHADE_V;

 	/** Draw the line using a horizontal shading pattern.*/
	public static final short SHADE_H_ATTR = A_SHADE_H;

	/** Draw the line using a negative slope diagonal shading pattern.*/
	public static final short SHADE_NS_ATTR = A_SHADE_NS;

	/** Draw the line using a positive slope diagonal shading pattern.*/
	public static final short SHADE_PS_ATTR = A_SHADE_PS;

	/** A universal SCR_RECT.*/
	public static SCR_RECT* uScrRect;

	/** The actual grayscale level (the gray variant the graphic instructions will draw). Legal value are Gray.OFF(grayscale disabled), Gray.BLACK, Gray.LIGHT_GRAY and Gray.DARK_GRAY.*/
	public static short level = Gray.OFF;

	/** Is in grayscale mode ? Should be only read*/
	public static boolean graymode = false;

	finally {
		
		native {
			#ifndef __MOKA__
				Graphics_uScrRect;
			#endif ;
		
		}
		
		if (Graphics.graymode) {
			native.GrayOff();
		}
	}

	/**
	 * Creates a new Graphics object using the default display device.
	 */
	public Graphics () {
		this.addr = (void*)0x4C00;

		//this.level = Gray.OFF;

		if (System.CALCULATOR == System.CALCULATOR_TI89) {
			this.x_max = 149;
			this.y_max = 93;
		}
		else {
			this.x_max = 239;
			this.y_max = 122;
		}
	}

	/**
	 * Sets the grayscale level.
	 *
	 * @param g the grayscale level. Legal values are : Gray.OFF(grayscale disabled), Gray.BLACK, Gray.LIGHT_GRAY and Gray.DARK_GRAY.
	 */
	public static native void setGray(short g) {
		Graphics_level = g;

		if (g == Gray_OFF) {
			if (Graphics_graymode) {
				GrayOff();
				Graphics_graymode = FALSE;
			}
		}
		else {
			if (!Graphics_graymode) {
				GrayOn();
				Graphics_graymode = TRUE;
			}
		}

		switch (g) {
			case Gray_LIGHT_GRAY:
			case Gray_BLACK: {
				GraySetAMSPlane (LIGHT_PLANE);
				break;
			}
			case Gray_DARK_GRAY: {
				GraySetAMSPlane (DARK_PLANE);
				break;
			}
		}
	}


	/**
	 * Creates a new Graphics object specifying the display device.
	 *
	 * @param addr The graphic port address
	 * @param x_max The maximum x coordinate
	 * @param y_max The maximum y coordinate
	 */
	public Graphics (void* addr, short x_max, short y_max) {
		this.addr = addr;

        //this.level = Gray.OFF;

		this.x_max = x_max;
		this.y_max = y_max;
	}

	/**
	 * Clears the Graphics' display device screen.  The pen
	 * position will remain intact.
	 * Changes the current graphic port to the one of the graphic object.
	 */
	public native void clrScr () {
		PortSet (this->addr, this->x_max, this->y_max);
		ClrScr();
	}

	/**
	 * Draws a char on the Graphics' display device screen.
	 * The following attributes are supported: Graphics.NORMAL_ATTR, Graphics.REVERSE_ATTR,
	 * Graphics.REPLACE_ATTR, Graphics.XOR_ATTR, Graphics.SHADED_ATTR.
	 * Changes the current graphic port to the one of the graphic object.
	 *
	 * @param x The x coordinate
	 * @param y The y coordinate
	 * @param c The char to draw
	 * @param attr The attribute
	 */
	public native void drawChar (short x, short y, char c, short attr) {
		PortSet (this->addr, this->x_max, this->y_max);

		DrawChar(x, y, c, attr);
	}

	/**
	 * Draws a pixel on the Graphics' display device screen.
	 * The following attributes are supported: Graphics.NORMAL_ATTR, Graphics.REVERSE_ATTR, Graphics.XOR_ATTR.
	 * Changes the current graphic port to the one of the graphic object.
	 *
	 * @param x The x coordinate
	 * @param y The y coordinate
	 * @param attr The attribute
	 */
	public native void drawPix (short x, short y, short attr) {
		PortSet (this->addr, this->x_max, this->y_max);
		DrawPix(x, y, attr);
	}

	/**
	 * Draws a line on the Graphics' display device screen.
	 * The following attributes are supported: Graphics.NORMAL_ATTR, Graphics.REVERSE_ATTR,
	 * Graphics.XOR_ATTR, Graphics.THICK1_ATTR, Graphics.SHADE_V_ATTR, Graphics.SHADE_H_ATTR,
	 * Graphics.SLOPE_NS_ATTR, Graphics.SLOPE_PS_ATTR.
	 * Changes the current graphic port to the one of the graphic object.
	 *
	 * @param x0 The x coordinate of the upper left corner
	 * @param y0 The y coordinate of the upper left corner
	 * @param x1 The x coordinate of the lower right corner
	 * @param y1 The y coordinate of the lower right corner
	 * @param attr The attribute
	 */
	public native void drawLine (short x0, short y0, short x1, short y1, short attr) {
		PortSet (this->addr, this->x_max, this->y_max);

		DrawLine(x0, y0, x1, y1, attr);
	}

	/**
	 * Draws an ANSI string on the Graphics' display device screen.
	 * The following attributes are supported: Graphics.NORMAL_ATTR, Graphics.REVERSE_ATTR,
	 * Graphics.REPLACE_ATTR, Graphics.XOR_ATTR, Graphics.SHADED_ATTR.
	 * Changes the current graphic port to the one of the graphic object.
	 *
	 * @param x The x coordinate
	 * @param y The y coordinate
	 * @param str The ANSI string to draw
	 * @param attr The attribute
	 */
	public native void draw (short x, short y, char* str, short attr) {
		PortSet (this->addr, this->x_max, this->y_max);

		DrawStr(x, y, str, attr);
	}

	/**
	 * Draws a String object on the Graphics' display device screen then finalizes the String object.
	 * The following attributes are supported: Graphics.NORMAL_ATTR, Graphics.REVERSE_ATTR,
	 * Graphics.REPLACE_ATTR, Graphics.XOR_ATTR, Graphics.SHADED_ATTR.
	 * Changes the current graphic port to the one of the graphic object.
	 *
	 * @param x The x coordinate
	 * @param y The y coordinate
	 * @param str The String to draw
	 * @param attr The attribute
	 */
	public native void draw (short x, short y, String str, short attr) {
		PortSet (this->addr, this->x_max, this->y_max);
		DrawStr(x, y, str->value, attr);

		str->finalize_(str);
	}

	/**
	 * Draws a short on the Graphics' display device screen.
	 * The following attributes are supported: Graphics.NORMAL_ATTR, Graphics.REVERSE_ATTR,
	 * Graphics.REPLACE_ATTR, Graphics.XOR_ATTR, Graphics.SHADED_ATTR.
	 * Changes the current graphic port to the one of the graphic object.
	 *
	 * @param x The x coordinate
	 * @param y The y coordinate
	 * @param num The short to draw
	 * @param attr The attribute
	 */
	public void draw (short x, short y, short num, short attr) {
		native {
			PortSet (this->addr, this->x_max, this->y_max);
		}
		String str = String.valueOf(num);

		Graphics.draw(x, y, str, attr);
	}

	/**
	 * Draws an int on the Graphics' display device screen.
	 * The following attributes are supported: Graphics.NORMAL_ATTR, Graphics.REVERSE_ATTR,
	 * Graphics.REPLACE_ATTR, Graphics.XOR_ATTR, Graphics.SHADED_ATTR.
	 * Changes the current graphic port to the one of the graphic object.
	 *
	 * @param x The x coordinate
	 * @param y The y coordinate
	 * @param num The int to draw
	 * @param attr The attribute
	 */
	public void draw (short x, short y, int num, short attr) {
		native {
			PortSet (this->addr, this->x_max, this->y_max);
		}
		String str = String.valueOf(num);

		Graphics.draw(x, y, str, attr);
	}

	/**
	 * Draws a long on the Graphics' display device screen.
	 * The following attributes are supported: Graphics.NORMAL_ATTR, Graphics.REVERSE_ATTR,
	 * Graphics.REPLACE_ATTR, Graphics.XOR_ATTR, Graphics.SHADED_ATTR.
	 * Changes the current graphic port to the one of the graphic object.
	 *
	 * @param x The x coordinate
	 * @param y The y coordinate
	 * @param num The long to draw
	 * @param attr The attribute
	 */
	public void draw (short x, short y, long num, short attr) {
		native {
			PortSet (this->addr, this->x_max, this->y_max);
		}
		String str = String.valueOf(num);

		Graphics.draw(x, y, str, attr);
	}

	/**
	 * Draws a double on the Graphics' display device screen.
	 * The following attributes are supported: Graphics.NORMAL_ATTR, Graphics.REVERSE_ATTR,
	 * Graphics.REPLACE_ATTR, Graphics.XOR_ATTR, Graphics.SHADED_ATTR.
	 * Changes the current graphic port to the one of the graphic object.
	 *
	 * @param x The x coordinate
	 * @param y The y coordinate
	 * @param num The double to draw
	 * @param attr The attribute
	 */
	public void draw (short x, short y, double num, short attr) {
		native {
			PortSet (this->addr, this->x_max, this->y_max);
		}
		String str = String.valueOf(num);

		Graphics.draw(x, y, str, attr);
	}

	/**
	 * Draws a char on the Graphics' display device screen.
	 * The following attributes are supported: Graphics.NORMAL_ATTR, Graphics.REVERSE_ATTR,
	 * Graphics.REPLACE_ATTR, Graphics.XOR_ATTR, Graphics.SHADED_ATTR.
	 * Changes the current graphic port to the one of the graphic object.
	 *
	 * @param x The x coordinate
	 * @param y The y coordinate
	 * @param c The char to draw
	 * @param attr The attribute
	 */
	public void draw (short x, short y, char c, short attr) {
		native {
			PortSet (this->addr, this->x_max, this->y_max);
		}
		Graphics.drawChar(x, y, c, attr);
	}

	/**
	 * Gets the status of a pixel on the Graphics' display device screen.
	 * Changes the current graphic port to the one of the graphic object.
	 *
	 * @param x The x coordinate
	 * @param y The y coordinate
	 * @return true if the pixel is on, false otherwize
	 */
	public native boolean getPix (short x, short y) {
		PortSet (this->addr, this->x_max, this->y_max);
		return GetPix(x, y);
	}

	/**
	 * Sets the virtual screen to the Graphics' display device screen.  All graphic commands which are built-in into TIOS
	 * does not expect that the video memory must be at 0x4C00, and that the video memory
	 * is always 240 x 128 pixels. Using portSet you can set up a virtual screen anywhere in
	 * a memory, and of any size.
	 */
	public native void portSet () {
		PortSet (this->addr, this->x_max, this->y_max);
	}

	/**
	 * Draws a ellipse on the Graphics' display device screen.
	 * The following attributes are supported: Graphics.NORMAL_ATTR, Graphics.REVERSE_ATTR,
	 * Graphics.XOR_ATTR.
	 * Changes the current graphic port to the one of the graphic object.
	 *
	 * @param x The x coordinate of the ellipse
	 * @param y The y coordinate of the ellipse
	 * @param a The horizontal semiaxe of the ellipse
	 * @param b The vertical semiaxe of the ellipse
	 * @param attr The attribute
	 */
	public native void drawEllipse (short x, short y, short a, short b, short attr) {
		PortSet (this->addr, this->x_max, this->y_max);
		DrawClipEllipse (x, y, a, b, Graphics_uScrRect, attr);
	}

	/**
	 * Draws a line from the current pen position (set by moveTo) on the Graphics' display device screen.
	 * Changes the current graphic port to the one of the graphic object.
	 *
	 * @param x The x coordinate
	 * @param y The y coordinate
	 */
	public native void drawTo (short x, short y) {
		PortSet (this->addr, this->x_max, this->y_max);
		DrawTo(x, y);
	}

	/**
	 * Saves the content of the LCD screen to a new LCD_BUFFER.
	 *
	 * @return A pointer to a newly allocated LCD_BUFFER.
	 */
	public native static void* lcdSave () {
		void* buffer = malloc(LCD_SIZE);

		LCD_save(buffer);

		return buffer;
	}

	/**
	 * Saves the content of the LCD screen to an existing LCD_BUFFER.
	 *
	 * @param buffer The LCD_BUFFER
	 * @return A pointer to the LCD_BUFFER.
	 */
	public native static void* lcdSave (void* buffer) {
		LCD_save(buffer);

		return buffer;
	}

	/**
	 * Restores the saved content of the LCD screen.
	 *
	 * @param buffer The saved LCD_BUFFER
	 */
	/*Alias implemented for lcdRestore (void* buffer).*/
	public abstract native static void lcdRestore (void* buffer) ;
	/*public native static void lcdRestore (void* buffer) {
		LCD_restore(buffer);
	}*/

	/**
	 * Sets the virtual screen.  All graphic commands which are built-in into TIOS
	 * does not expect that the video memory must be at 0x4C00, and that the video memory
	 * is always 240 x 128 pixels. Using portSet you can set up a virtual screen anywhere in
	 * a memory, and of any size.
	 *
	 * @param addr The memory address
	 * @param x_max The maximum x coordinate
	 * @param y_max The maximum x coordinate
	 */
	/*Alias implemented for portSet (void* addr, short x_max, short y_max).*/
	public abstract native static void portSet (void* addr, short x_max, short y_max);
	/*public native static void portSet (void* addr, short x_max, short y_max) {
		PortSet (addr, x_max, y_max);
	}*/

	/**
	 * Clears the current display device screen.  The pen
	 * position is moved to (0, 0). The main difference of this
	 * method vs the System class' one is its grayscale support.
	 */
	public static void clrscr () {
		Graphics.moveTo(0, 0);
		Graphics.clrScr();
	}

	/**
	 * Clears the current display device screen.  The pen
	 * position will remain intact.
	 */
	public native static void clrScr () {
		if (!Graphics_graymode) ClrScr();
		else {
			GraySetAMSPlane (LIGHT_PLANE);
			ClrScr();
			GraySetAMSPlane (DARK_PLANE);
			ClrScr();
		}
	}

	/**
	 * Draws a char on the current display device screen.
	 * The following attributes are supported: Graphics.NORMAL_ATTR, Graphics.REVERSE_ATTR,
	 * Graphics.REPLACE_ATTR, Graphics.XOR_ATTR, Graphics.SHADED_ATTR.
	 *
	 * @param x The x coordinate
	 * @param y The y coordinate
	 * @param c The char to draw
	 * @param attr The attribute
	 */
	public native static void drawChar (short x, short y, char c, short attr) {
		Graphics_setGray_short_int(Graphics_level);

		DrawChar(x, y, c, attr);

		if (Graphics_level != Gray_BLACK) return;
		GraySetAMSPlane (DARK_PLANE);

		DrawChar(x, y, c, attr);
	}

	/**
	 * Draws a pixel on the current display device screen.
	 * The following attributes are supported: Graphics.NORMAL_ATTR, Graphics.REVERSE_ATTR, Graphics.XOR_ATTR.
	 *
	 * @param x The x coordinate
	 * @param y The y coordinate
	 * @param attr The attribute
	 */
	public native static void drawPix (short x, short y, short attr) {
		Graphics_setGray_short_int(Graphics_level);

		DrawPix(x, y, attr);

		if (Graphics_level != Gray_BLACK) return;
		GraySetAMSPlane (DARK_PLANE);

		DrawPix(x, y, attr);
	}

	/**
	 * Draws a line on the current display device screen.
	 * The following attributes are supported: Graphics.NORMAL_ATTR, Graphics.REVERSE_ATTR,
	 * Graphics.XOR_ATTR, Graphics.THICK1_ATTR, Graphics.SHADE_V_ATTR, Graphics.SHADE_H_ATTR,
	 * Graphics.SLOPE_NS_ATTR, Graphics.SLOPE_PS_ATTR.
	 *
	 * @param x0 The x coordinate of the upper left corner
	 * @param y0 The y coordinate of the upper left corner
	 * @param x1 The x coordinate of the lower right corner
	 * @param y1 The y coordinate of the lower right corner
	 * @param attr The attribute
	 */
	public native static void drawLine (short x0, short y0, short x1, short y1, short attr) {
		Graphics_setGray_short_int(Graphics_level);

		DrawLine(x0, y0, x1, y1, attr);

		if (Graphics_level != Gray_BLACK) return;
		GraySetAMSPlane (DARK_PLANE);

		DrawLine(x0, y0, x1, y1, attr);
	}

	/**
	 * Draws a elipse on the current display device screen.
	 * The following attributes are supported: Graphics.NORMAL_ATTR, Graphics.REVERSE_ATTR,
	 * Graphics.XOR_ATTR.
	 *
	 * @param x The x coordinate of the elipse
	 * @param y The y coordinate of the elipse
	 * @param a The horizontal semiaxe of the elipse
	 * @param b The vertical semiaxe of the elipse
	 * @param attr The attribute
	 */
	public native static void drawElipse (short x, short y, short a, short b, short attr) {
		Graphics_setGray_short_int(Graphics_level);

		DrawClipEllipse (x, y, a, b, Graphics_uScrRect, attr);

		if (Graphics_level != Gray_BLACK) return;
		GraySetAMSPlane (DARK_PLANE);

		DrawClipEllipse (x, y, a, b, Graphics_uScrRect, attr);
	}

	/**
	 * Draws a line from the current pen position (set by moveTo) on the current display device screen.
	 *
	 * @param x The x coordinate
	 * @param y The y coordinate
	 */
	public native static void drawTo (short x, short y) {
		SCR_STATE ss;

		Graphics_setGray_short_int(Graphics_level);

		DrawTo(x, y);

		if (Graphics_level != Gray_BLACK) return;
		GraySetAMSPlane (DARK_PLANE);

		MoveTo(ss.CurX, ss.CurY);

		DrawTo(x, y);
	}

	/**
	 * Move the current pen position.
	 *
	 * @param x The x coordinate
	 * @param y The y coordinate
	 */
	/*Alias implemented for moveTo (short x, short y).*/
	public abstract native static void moveTo (short x, short y) ;
	/*public native static void moveTo (short x, short y) {
		MoveTo(x, y);
	}*/

	/**
	 * Draws an ANSI string on the current display device screen.
	 * The following attributes are supported: Graphics.NORMAL_ATTR, Graphics.REVERSE_ATTR,
	 * Graphics.REPLACE_ATTR, Graphics.XOR_ATTR, Graphics.SHADED_ATTR.
	 *
	 * @param x The x coordinate
	 * @param y The y coordinate
	 * @param str The ANSI string to draw
	 * @param attr The attribute
	 */
	public native static void draw (short x, short y, char* str, short attr) {
		Graphics_setGray_short_int(Graphics_level);

		DrawStr(x, y, str, attr);

		if (Graphics_level != Gray_BLACK) return;
		GraySetAMSPlane (DARK_PLANE);

		DrawStr(x, y, str, attr);
	}

	/**
	 * Draws a String object on the current display device screen then finalizes the String object.
	 * The following attributes are supported: Graphics.NORMAL_ATTR, Graphics.REVERSE_ATTR,
	 * Graphics.REPLACE_ATTR, Graphics.XOR_ATTR, Graphics.SHADED_ATTR.
	 *
	 * @param x The x coordinate
	 * @param y The y coordinate
	 * @param str The String to draw
	 * @param attr The attribute
	 */
	public native static void draw (short x, short y, String str, short attr) {
		Graphics_draw_short_int_short_int_char_p_short_int(x, y, str->value, attr);

		str->finalize_(str);
	}

	/**
	 * Draws a short on the current display device screen.
	 * The following attributes are supported: Graphics.NORMAL_ATTR, Graphics.REVERSE_ATTR,
	 * Graphics.REPLACE_ATTR, Graphics.XOR_ATTR, Graphics.SHADED_ATTR.
	 *
	 * @param x The x coordinate
	 * @param y The y coordinate
	 * @param num The short to draw
	 * @param attr The attribute
	 */
	public static void draw (short x, short y, short num, short attr) {
		String str = String.valueOf(num);

		Graphics.draw(x, y, str, attr);
	}

	/**
	 * Draws an int on the current display device screen.
	 * The following attributes are supported: Graphics.NORMAL_ATTR, Graphics.REVERSE_ATTR,
	 * Graphics.REPLACE_ATTR, Graphics.XOR_ATTR, Graphics.SHADED_ATTR.
	 *
	 * @param x The x coordinate
	 * @param y The y coordinate
	 * @param num The int to draw
	 * @param attr The attribute
	 */
	public static void draw (short x, short y, int num, short attr) {
		String str = String.valueOf(num);

		Graphics.draw(x, y, str, attr);
	}

	/**
	 * Draws a long on the current display device screen.
	 * The following attributes are supported: Graphics.NORMAL_ATTR, Graphics.REVERSE_ATTR,
	 * Graphics.REPLACE_ATTR, Graphics.XOR_ATTR, Graphics.SHADED_ATTR.
	 *
	 * @param x The x coordinate
	 * @param y The y coordinate
	 * @param num The long to draw
	 * @param attr The attribute
	 */
	public static void draw (short x, short y, long num, short attr) {
		String str = String.valueOf(num);

		Graphics.draw(x, y, str, attr);
	}

	/**
	 * Draws a double on the current display device screen.
	 * The following attributes are supported: Graphics.NORMAL_ATTR, Graphics.REVERSE_ATTR,
	 * Graphics.REPLACE_ATTR, Graphics.XOR_ATTR, Graphics.SHADED_ATTR.
	 *
	 * @param x The x coordinate
	 * @param y The y coordinate
	 * @param num The double to draw
	 * @param attr The attribute
	 */
	public static void draw (short x, short y, double num, short attr) {
		String str = String.valueOf(num);

		Graphics.draw(x, y, str, attr);
	}

	/**
	 * Draws a char on the current display device screen.
	 * The following attributes are supported: Graphics.NORMAL_ATTR, Graphics.REVERSE_ATTR,
	 * Graphics.REPLACE_ATTR, Graphics.XOR_ATTR, Graphics.SHADED_ATTR.
	 *
	 * @param x The x coordinate
	 * @param y The y coordinate
	 * @param c The char to draw
	 * @param attr The attribute
	 */
	public static void draw (short x, short y, char c, short attr) {
		Graphics.drawChar(x, y, c, attr);
	}

	/**
	 * Draws an ellipse on the current display device screen.
	 * The following attributes are supported: Graphics.NORMAL_ATTR, Graphics.REVERSE_ATTR,
	 * Graphics.XOR_ATTR.
	 * Changes the current graphic port to the one of the graphic object.
	 *
	 * @param x The x coordinate of the ellipse
	 * @param y The y coordinate of the ellipse
	 * @param a The horizontal semiaxe of the ellipse
	 * @param b The vertical semiaxe of the ellipse
	 * @param attr The attribute
	 */
	public static native void drawEllipse (short x, short y, short a, short b, short attr) {
		Graphics_setGray_short_int(Graphics_level);

		DrawClipEllipse (x, y, a, b, Graphics_uScrRect, attr);

		if (Graphics_level != Gray_BLACK) return;
		GraySetAMSPlane (DARK_PLANE);

		DrawClipEllipse (x, y, a, b, Graphics_uScrRect, attr);
	}

	/**
	 * Gets the status of a pixel on the current display device screen. Intented to
	 * be used while in grayscale mode.
	 *
	 * @param x The x coordinate
	 * @param y The y coordinate
	 * @return the status of the pixel (Gray.OFF if grayscale is turned off or the pixel is off, grayscale level otherwize).
	 */
	/*Alias implemented for getPix (short x, short y).*/
	public native static short getPixGray (short x, short y) {
		BOOL light;
		BOOL dark;
		short p = Gray_OFF;
		short g = Graphics_level;

		if (Graphics_graymode) {
			GraySetAMSPlane(LIGHT_PLANE);
			light = GetPix(x, y);

			GraySetAMSPlane(DARK_PLANE);
			dark = GetPix(x, y);

			if (light) {
				if (dark) {
					p = Gray_BLACK;
				}
				else {
					p = Gray_LIGHT_GRAY;
				}
			}
			else if (dark) {
				p = Gray_DARK_GRAY;
			}
		}

		Graphics_setGray_short_int(g);

		return p;
	}

	/**
	 * Gets the status of a pixel on the current display device screen.
	 *
	 * @param x The x coordinate
	 * @param y The y coordinate
	 * @return true if the pixel is on, false otherwize
	 */
	/*Alias implemented for getPix (short x, short y).*/
	public abstract native static boolean getPix (short x, short y) ;
	/*public native static boolean getPix (short x, short y) {
		return GetPix(x, y);
	}*/

	/**
	 * Converts a specified PIC file name to a bitmap.
	 * If the file string object risk to be used later, use a copy of the string object
     * instead of the original, the string being finalized before the method returns.
	 *
	 * @param file The PIC file name
	 * @return The converted bitmap or null in case of error.
	 */
	public native static BITMAP* bitmapFromVar(String file) {
		char *SymName = file->toVat_(file);
		BITMAP* bitmap;

		SYM_ENTRY *sym_entry = SymFindPtr (SymName, 0);

		/*if (!sym_entry) return null;
		if (peek (HToESI (sym_entry->handle)) != PIC_TAG) return NULL;*/

		if ((sym_entry) && (peek (HToESI (sym_entry->handle)) == PIC_TAG)) {
			bitmap = Graphics_cloneBitmap_BITMAP_p(HeapDeref (sym_entry->handle) + 2);
		}

		free ( SymName - strlen(file->value) - 1) ;

		file->finalize_(file);//Frees the memory allocated to the TIOS string

		return bitmap;
	}

	/**
	 * Saves a specified bitmap to a specified pic file.
	 * If the file string object risk to be used later, use a copy of the string object
	 * instead of the original, the string being finalized before the method returns.
	 *
	 * @param bmp The bitmap
	 * @param file The PIC file name
	 */
	public native static void bitmapToVar(BITMAP* bmp, String file) {
		//Code from Qs CLib BMPGET function
		short taille = bmp->NumRows*((bmp->NumCols+7)>>3)+4;//The length of the bitmap
		SYM_ENTRY *sym_entry;//A pointer on the TIOS VAT entry
		char *vat,//A pointer to the TIOS variable
				 *var;//Will contain the name of the variable, in TIOS string format

		var = file->toVat_(file);//Convert nom to TIOS string format

		sym_entry = DerefSym(SymAdd(var)) ;//Creates a new TIOS entry and assigns it to sym_entry

		vat = (unsigned char*)HeapDeref(sym_entry->handle = HeapAlloc(taille+3)) ;//Assigns the adress of the TIOS variable to vat
		//Writes the dimensions of the PIC to the PIC header
		vat[0] =  (taille + 1) / 256;
		vat[1] =  (taille + 1) % 256;
		memcpy(vat+2, bmp, taille) ;//Copy the bitmap data to the TIOS variable
		vat[taille+2] = 0xDF ;//Ends the TIOS variable by the PIC_VAR tag (DF)

		free(var - strlen(file->value) - 1);//Frees the memory allocated to the TIOS string

		file->finalize_(file);
	}

	/**
	 * Return a bitmap, taken from a specified zone of the screen.
	 * @param x Absolute horizontal coordinate
	 * @param y Absolute vertical coordinate
	 * @param width The width of the zone
	 * @param height The height of the zone
	 */
	public static BITMAP* getBitmap (short x, short y, short width, short height) {
		SCR_RECT scr = native {(SCR_RECT){{x, y, x+width-1, y+height-1}}};
		BITMAP* bmp = native.malloc(BitmapSize(&scr));

		native.BitmapGet(&scr, bmp);

		return bmp;
	}

	/**
	 * Puts a BITMAP on the screen. The following attributes are supported: Graphics.NORMAL_ATTR, Graphics.REVERSE_ATTR, Graphics.REPLACE_ATTR, Graphics.XOR_ATTR, Graphics.SHADED_ATTR, Graphics.OR_ATTR, Graphics.AND_ATTR.
	 * @param b The BITMAP
	 * @param x Absolute horizontal coordinate
	 * @param y Absolute vertical coordinate
	 * @param attr Display attribute
	 */
	public native static void putBitmap (BITMAP* b, short x, short y, short attr) {
		Graphics_setGray_short_int(Graphics_level);

		BitmapPut (x, y, b, Graphics_uScrRect, attr);

		if (Graphics_level != Gray_BLACK) return;
		GraySetAMSPlane (DARK_PLANE);

		BitmapPut (x, y, b, Graphics_uScrRect, attr);
	}

	/**
	 * Puts a set of TIOS on the screen. The following attributes are supported: Graphics.NORMAL_ATTR, Graphics.REVERSE_ATTR, Graphics.REPLACE_ATTR, Graphics.XOR_ATTR, Graphics.SHADED_ATTR, Graphics.OR_ATTR, Graphics.AND_ATTR.
	 * Be warned that this method will enable grayscale.
	 * @param d The BITMAP of the dark plane (layer 1)
	 * @param l The BITMAP of the light plane (layer 2)
	 * @param x Absolute horizontal coordinate
	 * @param y Absolute vertical coordinate
	 * @param attr Display attribute
	 */
	public native static void putGray (BITMAP* d, BITMAP* l, short x, short y, short attr) {
		short int graySave = Graphics_level;
		
		Graphics_setGray_short_int(Gray_DARK_GRAY);

		BitmapPut (x, y, d, Graphics_uScrRect, attr);

		if (Graphics_level != Gray_BLACK) return;
		GraySetAMSPlane (LIGHT_PLANE);

		BitmapPut (x, y, l, Graphics_uScrRect, attr);
		
		Graphics_level = graySave;
	}

	/**
	 * Puts a set of TIOS PIC variables on the screen. The following attributes are supported: Graphics.NORMAL_ATTR, Graphics.REVERSE_ATTR, Graphics.REPLACE_ATTR, Graphics.XOR_ATTR, Graphics.SHADED_ATTR, Graphics.OR_ATTR, Graphics.AND_ATTR.
	 * Be warned that this method will enable grayscale.
	 * @param d The PIC file name of the dark plane (layer 1)
	 * @param l The PIC file name of the light plane (layer 2)
	 * @param x Absolute horizontal coordinate
	 * @param y Absolute vertical coordinate
	 * @param attr Display attribute
	 */
	public static void putGray (String d, String l, short x, short y, short attr) {
		short graySave = Graphics.level;

		Graphics.setGray(Gray.DARK_GRAY);

		Graph.picVarPut(d, x, y, attr);

		Gray.setAMSPlane(Gray.LIGHT_PLANE);

		Graph.picVarPut(l, x, y, attr);
		
		Graphics.level = graySave;
	}

	/**
	 * Saves a zone of the screen to a TIOS' PIC variable.
	 * If the file string object risk to be used later, use a copy of the string object
	 * instead of the original, the string being finalized before the method returns.
	 * @param file The PIC file name
	 * @param x Absolute horizontal coordinate
	 * @param y Absolute vertical coordinate
	 * @param width The width of the zone
	 * @param height The height of the zone
	 */
	public static void getPicVar (String file, short x, short y, short width, short height) {
		BITMAP * bmp = Graphics.getBitmap(x, y, width, height);

		Graphics.bitmapToVar((BITMAP*)bmp, file);

		native.free(bmp);
	}

	/**
	 * Puts a TIOS PIC variable on the screen. The following attributes are supported: Graphics.NORMAL_ATTR, Graphics.REVERSE_ATTR, Graphics.REPLACE_ATTR, Graphics.XOR_ATTR, Graphics.SHADED_ATTR, Graphics.OR_ATTR, Graphics.AND_ATTR.
	 * If the file string object risk to be used later, use a copy of the string object
	 * instead of the original, the string being finalized before the method returns.
	 * @param file The PIC file name
	 * @param x Absolute horizontal coordinate
	 * @param y Absolute vertical coordinate
	 * @param attr Display attribute
	 */
	public static void picVarPut (String file, short x, short y, short attr) {
		char* name = file.toVat();

		Graphics.picVarPut(name, x, y, attr);

		native.free(name - strlen(file->value) - 1);//Frees the memory allocated to the TIOS string
		file.finalize();
	}

	/**
	 * Puts a TIOS PIC variable on the screen. The following attributes are supported: Graphics.NORMAL_ATTR, Graphics.REVERSE_ATTR, Graphics.REPLACE_ATTR, Graphics.XOR_ATTR, Graphics.SHADED_ATTR, Graphics.OR_ATTR, Graphics.AND_ATTR.
	 * @param SymName TIOS variable name, in vat string format
	 * @param x Absolute horizontal coordinate
	 * @param y Absolute vertical coordinate
	 * @param attr Display attribute
	 */
	public native static void picVarPut (char *SymName, short x, short y, short attr) {
		SYM_ENTRY *sym_entry = SymFindPtr (SymName, 0);
		if (!sym_entry) return;
		if (peek (HToESI (sym_entry->handle)) != PIC_TAG) return;

		Graphics_setGray_short_int(Graphics_level);

		BitmapPut (x, y, HeapDeref (sym_entry->handle) + 2, Graphics_uScrRect, attr);

		if (Graphics_level != Gray_BLACK) return;
		GraySetAMSPlane (DARK_PLANE);

		BitmapPut (x, y, HeapDeref (sym_entry->handle) + 2, Graphics_uScrRect, attr);
	}

	/**
	 * Returns a copy of the specified bitmap.
	 *
	 * @param bmp The bitmap
	 * @return A copy of the specified bitmap
	 */
	public native static BITMAP* cloneBitmap(BITMAP* bmp) {
		int i;
		SCR_RECT rec = (SCR_RECT){0, 0,  bmp->NumCols - 1, bmp->NumRows - 1};
		long size = BitmapSize(&rec);
		BITMAP* buffer = (BITMAP*)malloc(size);

		for (i = 0; i < size; i++) {
			((char*)buffer)[i] = ((char*)bmp)[i];
		}

		return buffer;
	}
}