/*
 * QUSOFT MICROSYSTMES
 * Moka
 * Copyright 2004 Frdric Brown
 */

/*
 *  This file is part of Moka API.
 *
 *  Moka API is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Moka API is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Moka API; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package moka.io;

//import moka.lang.System;
//import moka.lang.Character;
import moka.io.Graphics;

/**
 * The Graph class provides methods to deal with bitmaps and display. This class has many features of the Graphics class, but implements them
 * more efficiently.  However, these methods, instead of those of Graphics class, will not manage grayscale automatically.
 *
 * @author  Frdric Brown
 * @version 1.0, 2004-03-17
 * @since   MDK2.2
 */
public final class Graph {
	/** Draw with destination pixels turned off.*/
	public static final short REVERSE_ATTR = native.A_REVERSE;

	/** Draw with destination pixels turned on.*/
	public static final short NORMAL_ATTR = native.A_NORMAL;

	/** Draw with source pixels XORed with destination pixels.*/
	public static final short XOR_ATTR = native.A_XOR;

	/** Draw with destination pixels masked so that every other pixel turned off*/
	public static final short SHADED_ATTR = native.A_SHADED;

	/** Draw with source pixels replace destination pixels.*/
	public static final short REPLACE_ATTR = native.A_REPLACE;

	/** Draw with source pixels ORed with destination pixels.*/
	public static final short OR_ATTR = native.A_OR;

	/** AND the region to draw.*/
	public static final short AND_ATTR = native.A_AND;

	/** Draw a double thick line.*/
	public static final short THICK1_ATTR = native.A_THICK1;

	/** Draw the line using a vertical shading pattern.*/
	public static final short SHADE_V_ATTR = native.A_SHADE_V;

 	/** Draw the line using a horizontal shading pattern.*/
	public static final short SHADE_H_ATTR = native.A_SHADE_H;

	/** Draw the line using a negative slope diagonal shading pattern.*/
	public static final short SHADE_NS_ATTR = native.A_SHADE_NS;

	/** Draw the line using a positive slope diagonal shading pattern.*/
	public static final short SHADE_PS_ATTR = native.A_SHADE_PS;
	
	/** Draw with destination pixels turned off (alias).*/
	public static final short A_REVERSE = native.A_REVERSE;

	/** Draw with destination pixels turned on (alias).*/
	public static final short A_NORMAL = native.A_NORMAL;

	/** Draw with source pixels XORed with destination pixels (alias).*/
	public static final short A_XOR = native.A_XOR;

	/** Draw with destination pixels masked so that every other pixel turned off (alias)*/
	public static final short A_SHADED = native.A_SHADED;

	/** Draw with source pixels replace destination pixels (alias).*/
	public static final short A_REPLACE = native.A_REPLACE;

	/** Draw with source pixels ORed with destination pixels (alias).*/
	public static final short A_OR = native.A_OR;

	/** AND the region to draw (alias).*/
	public static final short A_AND = native.A_AND;

	/** Draw a double thick line (alias).*/
	public static final short A_THICK1 = native.A_THICK1;

	/** Draw the line using a vertical shading pattern (alias).*/
	public static final short A_SHADE_V = native.A_SHADE_V;

 	/** Draw the line using a horizontal shading pattern (alias).*/
	public static final short A_SHADE_H = native.A_SHADE_H;

	/** Draw the line using a negative slope diagonal shading pattern (alias).*/
	public static final short A_SHADE_NS = native.A_SHADE_NS;

	/** Draw the line using a positive slope diagonal shading pattern (alias).*/
	public static final short A_SHADE_PS = native.A_SHADE_PS;
	
	private Graph () {}

	/**
	 * Clears the current display device screen.
	 */
	public abstract static void clrscr () ;
	
	/**
	 * Clears the current display device screen.  The pen
	 * position will remain intact.
	 */
	public abstract static void clrScr () ; //ClrScr

	/**
	 * Draws a char on the current display device screen.
	 * The following attributes are supported: Graph.NORMAL_ATTR, Graph.REVERSE_ATTR,
	 * Graph.REPLACE_ATTR, Graph.XOR_ATTR, Graph.SHADED_ATTR.
	 *
	 * @param x The x coordinate
	 * @param y The y coordinate
	 * @param c The char to draw
	 * @param attr The attribute
	 */
	public abstract static void drawChar (short x, short y, char c, short attr) ;

	/**
	 * Draws a pixel on the current display device screen.
	 * The following attributes are supported: Graph.NORMAL_ATTR, Graph.REVERSE_ATTR, Graph.XOR_ATTR.
	 *
	 * @param x The x coordinate
	 * @param y The y coordinate
	 * @param attr The attribute
	 */
	public abstract static void drawPix (short x, short y, short attr) ;

	/**
	 * Draws a line on the current display device screen.
	 * The following attributes are supported: Graph.NORMAL_ATTR, Graph.REVERSE_ATTR,
	 * Graph.XOR_ATTR, Graph.THICK1_ATTR, Graph.SHADE_V_ATTR, Graph.SHADE_H_ATTR,
	 * Graph.SLOPE_NS_ATTR, Graph.SLOPE_PS_ATTR.
	 *
	 * @param x0 The x coordinate of the upper left corner
	 * @param y0 The y coordinate of the upper left corner
	 * @param x1 The x coordinate of the lower right corner
	 * @param y1 The y coordinate of the lower right corner
	 * @param attr The attribute
	 */
	public abstract static void drawLine (short x0, short y0, short x1, short y1, short attr) ;

	/**
	 * Draws a elipse on the current display device screen.
	 * The following attributes are supported: Graph.NORMAL_ATTR, Graph.REVERSE_ATTR,
	 * Graph.XOR_ATTR.
	 *
	 * @param x The x coordinate of the elipse
	 * @param y The y coordinate of the elipse
	 * @param a The horizontal semiaxe of the elipse
	 * @param b The vertical semiaxe of the elipse
	 * @param attr The attribute
	 */
	 /* DrawClipEllipse (x, y, a, b, Graph_uScrRect, attr); */
	public abstract static void drawEllipse (short x, short y, short a, short b, short attr) ;

	/**
	 * Draws a line from the current pen position (set by moveTo) on the current display device screen.
	 *
	 * @param x The x coordinate
	 * @param y The y coordinate
	 */
	public abstract static void drawTo (short x, short y) ;

	/**
	 * Move the current pen position.
	 *
	 * @param x The x coordinate
	 * @param y The y coordinate
	 */
	/*Alias implemented for moveTo (short x, short y).*/
	public abstract static void moveTo (short x, short y) ;

	/**
	 * Draws an ANSI string on the current display device screen.
	 * The following attributes are supported: Graph.NORMAL_ATTR, Graph.REVERSE_ATTR,
	 * Graph.REPLACE_ATTR, Graph.XOR_ATTR, Graph.SHADED_ATTR.
	 *
	 * @param x The x coordinate
	 * @param y The y coordinate
	 * @param str The ANSI string to draw
	 * @param attr The attribute
	 */
	/* DrawStr(x, y, str, attr); */
	public abstract static void draw (short x, short y, char* str, short attr) ;
	
	/**
	 * Draws a String object on the current display device screen then finalizes the String object.
	 * The following attributes are supported: Graph.NORMAL_ATTR, Graph.REVERSE_ATTR,
	 * Graph.REPLACE_ATTR, Graph.XOR_ATTR, Graph.SHADED_ATTR.
	 *
	 * @param x The x coordinate
	 * @param y The y coordinate
	 * @param str The String to draw
	 * @param attr The attribute
	 */
	public static void draw (short x, short y, String str, short attr) {
		Graph.draw(x, y, (char*)str.value, attr);

		str.finalize();
	}

	/**
	 * Draws a short on the current display device screen.
	 * The following attributes are supported: Graph.NORMAL_ATTR, Graph.REVERSE_ATTR,
	 * Graph.REPLACE_ATTR, Graph.XOR_ATTR, Graph.SHADED_ATTR.
	 *
	 * @param x The x coordinate
	 * @param y The y coordinate
	 * @param num The short to draw
	 * @param attr The attribute
	 */
	public static void draw (short x, short y, short num, short attr) {
		String str = String.valueOf(num);

		Graph.draw(x, y, str, attr);
	}

	/**
	 * Draws an int on the current display device screen.
	 * The following attributes are supported: Graph.NORMAL_ATTR, Graph.REVERSE_ATTR,
	 * Graph.REPLACE_ATTR, Graph.XOR_ATTR, Graph.SHADED_ATTR.
	 *
	 * @param x The x coordinate
	 * @param y The y coordinate
	 * @param num The int to draw
	 * @param attr The attribute
	 */
	public static void draw (short x, short y, int num, short attr) {
		String str = String.valueOf(num);

		Graph.draw(x, y, str, attr);
	}

	/**
	 * Draws a long on the current display device screen.
	 * The following attributes are supported: Graph.NORMAL_ATTR, Graph.REVERSE_ATTR,
	 * Graph.REPLACE_ATTR, Graph.XOR_ATTR, Graph.SHADED_ATTR.
	 *
	 * @param x The x coordinate
	 * @param y The y coordinate
	 * @param num The long to draw
	 * @param attr The attribute
	 */
	public static void draw (short x, short y, long num, short attr) {
		String str = String.valueOf(num);

		Graph.draw(x, y, str, attr);
	}

	/**
	 * Draws a double on the current display device screen.
	 * The following attributes are supported: Graph.NORMAL_ATTR, Graph.REVERSE_ATTR,
	 * Graph.REPLACE_ATTR, Graph.XOR_ATTR, Graph.SHADED_ATTR.
	 *
	 * @param x The x coordinate
	 * @param y The y coordinate
	 * @param num The double to draw
	 * @param attr The attribute
	 */
	public static void draw (short x, short y, double num, short attr) {
		String str = String.valueOf(num);

		Graph.draw(x, y, str, attr);
	}

	/**
	 * Draws a char on the current display device screen.
	 * The following attributes are supported: Graph.NORMAL_ATTR, Graph.REVERSE_ATTR,
	 * Graph.REPLACE_ATTR, Graph.XOR_ATTR, Graph.SHADED_ATTR.
	 *
	 * @param x The x coordinate
	 * @param y The y coordinate
	 * @param c The char to draw
	 * @param attr The attribute
	 */
	public abstract static void draw (short x, short y, char c, short attr) ;

	/**
	 * Gets the status of a pixel on the current display device screen.
	 *
	 * @param x The x coordinate
	 * @param y The y coordinate
	 * @return true if the pixel is on, false otherwize
	 */
	/*Alias implemented for getPix (short x, short y).*/
	public abstract static boolean getPix (short x, short y) ;
	
	/**
	 * Puts a TIOS PIC variable on the screen. The following attributes are supported: Graph.NORMAL_ATTR, Graph.REVERSE_ATTR, Graph.REPLACE_ATTR, Graph.XOR_ATTR, Graph.SHADED_ATTR, Graph.OR_ATTR, Graph.AND_ATTR.
	 * @param SymName TIOS variable name, in vat string format
	 * @param x Absolute horizontal coordinate
	 * @param y Absolute vertical coordinate
	 * @param attr Display attribute
	 */
	public native static void picVarPut (char *SymName, short x, short y, short attr) {
		SYM_ENTRY *sym_entry = SymFindPtr (SymName, 0);
		if (!sym_entry) return;
		if (peek (HToESI (sym_entry->handle)) != PIC_TAG) return;
		
		BitmapPut (x, y, HeapDeref (sym_entry->handle) + 2, Graphics_uScrRect, attr);
	}
	
	/**
	 * Puts a TIOS PIC variable on the screen. The following attributes are supported: Graph.NORMAL_ATTR, Graph.REVERSE_ATTR, Graph.REPLACE_ATTR, Graph.XOR_ATTR, Graph.SHADED_ATTR, Graph.OR_ATTR, Graph.AND_ATTR.
	 * If the file string object risk to be used later, use a copy of the string object
	 * instead of the original, the string being finalized before the method returns.
	 * @param file The PIC file name
	 * @param x Absolute horizontal coordinate
	 * @param y Absolute vertical coordinate
	 * @param attr Display attribute
	 */
	public static void picVarPut (String file, short x, short y, short attr) {
		char* name = file.toVat();

		Graph.picVarPut(name, x, y, attr);

		native.free(name - strlen(file->value) - 1);//Frees the memory allocated to the TIOS string
		file.finalize();
	}
	
	/**
	 * Puts a set of TIOS PIC variables on the screen. The following attributes are supported: Graphics.NORMAL_ATTR, Graphics.REVERSE_ATTR, Graphics.REPLACE_ATTR, Graphics.XOR_ATTR, Graphics.SHADED_ATTR, Graphics.OR_ATTR, Graphics.AND_ATTR.
	 * Be warned that this method needs grayscale to be enabled and will set the plane to Gray.LIGHT_PLANE.
	 * @param d The PIC file name of the dark plane (layer 1)
	 * @param l The PIC file name of the light plane (layer 2)
	 * @param x Absolute horizontal coordinate
	 * @param y Absolute vertical coordinate
	 * @param attr Display attribute
	 */
	public static void putGray (String d, String l, short x, short y, short attr) {
		Gray.setAMSPlane(Gray.DARK_PLANE);

		Graph.picVarPut(d, x, y, attr);

		Gray.setAMSPlane(Gray.LIGHT_PLANE);

		Graph.picVarPut(l, x, y, attr);
	}
}