/*
 * QUSOFT MICROSYSTMES
 * Moka
 * Copyright 2002-2003 Frdric Brown
 */

/*
 *  This file is part of Moka API.
 *
 *  Moka API is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Moka API is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Moka API; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package moka.io;

import moka.io.IOStream;
import moka.lang.Character;
import moka.lang.System;
import moka.lang.String;

/**
 * An abstract representation of file and directory pathnames.
 *
 * @author  Frdric Brown
 * @version 1.1, 2003-11-02
 * @since   MDK1.0a
 */
public class File extends IOStream {
	/** A char pointer used by the showVarLinkDialog method.*/
	private static char * varBuffer;

	/** Open for reading only (binary).*/
	public static final short MODE_OPEN_READ = 0;
	/** Create for writing (binary). If a file by that name already exists, it will be overwritten.*/
	public static final short MODE_WRITE = 1;
	/** Append (binary); open for writing at end of file, or create for writing if the file does not exist.*/
	public static final short MODE_APPEND = 2;
	/** Open an existing file for update (reading and writing) (binary).*/
	public static final short MODE_OPEN_UPDATE = 3;
	/** Create a new file for update (reading and writing) (binary). If a file by that name already exists, it will be overwritten.*/
	public static final short MODE_UPDATE = 4;
	/** Open for append (binary); open for update at the end of the file, or create if the file does not exist.*/
	public static final short MODE_UPDATE_APPEND = 5;

	/** Open for reading only (text).*/
	public static final short MODE_OPEN_READ_TEXT = 6;
	/** Create for writing (text). If a file by that name already exists, it will be overwritten.*/
	public static final short MODE_WRITE_TEXT = 7;
	/** Append (text); open for writing at end of file, or create for writing if the file does not exist.*/
	public static final short MODE_APPEND_TEXT = 8;
	/** Open an existing file for update (reading and writing) (text).*/
	public static final short MODE_OPEN_UPDATE_TEXT = 9;
	/** Create a new file for update (reading and writing) (text). If a file by that name already exists, it will be overwritten.*/
	public static final short MODE_UPDATE_TEXT = 10;
	/** Open for append (text); open for update at the end of the file, or create if the file does not exist.*/
	public static final short MODE_UPDATE_APPEND_TEXT = 11;

	/** The FILE structure encapsulated. */
	private FILE* file;

	/** The file mode. Should be only read.*/
	public short mode;

	/** The name of the file. Should be only read.*/
	public String name = null;

	/** The type of the file to appear in the VAR-LINK menu.*/
	public String type = null;

    /**
     * Creates a new <code>File</code> instance by using the given pathname string.
     * If the pathname string object risk to be used later, use a copy of the string object
     * instead of the original.
     *
     @param pathname A pathname string
     */
	public File (String pathname) {
		super();
		this.name = pathname;
		this.file = null;
		this.mode = File.MODE_OPEN_READ;
	}

	/**
	 * Shows the VAR-LINK dialog and return a file object, associated to the TIOS variable selected in the dialog.
	 *
	 * @return a file object associated to the TIOS variable selected in the dialog, or null if no file was selected.
	 */
	public static native File showVarLinkDialog() {
		EVENT ev;
		File_varBuffer = (char*)malloc(sizeof(char) * 20);
		TString* str;
		short len;

		File_varBuffer[0] = 0;

  		EVENT_HANDLER OldHandler = EV_captureEvents (VarLinkHandler);
  		memset (&ev, sizeof (ev), 0);
  		ev.Type = CM_KEYPRESS;
  		ev.extra.Key.Code = KEY_VARLNK;
  		EV_defaultHandler (&ev);
  		EV_captureEvents (OldHandler);

		if (strcmp("", File_varBuffer)) {
			str = TString_char_p_BOOL(File_varBuffer, FALSE);

			len = strlen(str->value);

			if (str->value[len-1] == '(') {
				str->delete_long_int_long_int(str, len-1, len);
			}

			return TFile_String(str);
		}
		else {
			free(File_varBuffer);
			return NULL;
		}
	}

	/**
	 * Exports the long argument to a TIOS variable.
	 * If the name string object risk to be used later, use a copy of the string object
     * instead of the original.
	 *
	 * name The name of the TIOS variable
	 * l The long to export
	 */
	public static void export (String name, long l) {
		File.exportExpr(name, String.valueOf(l));
	}

	/**
	 * Exports the int argument to a TIOS variable.
	 * If the name string object risk to be used later, use a copy of the string object
     * instead of the original.
	 *
	 * name The name of the TIOS variable
	 * i The int to export
	 */
	public static void export (String name, int i) {
		File.exportExpr(name, String.valueOf(i));
	}

	/**
	 * Exports the short argument to a TIOS variable.
	 * If the name string object risk to be used later, use a copy of the string object
     * instead of the original.
	 *
	 * name The name of the TIOS variable
	 * s The short to export
	 */
	public static void export (String name, short s) {
		File.exportExpr(name, String.valueOf(s));
	}

	/**
	 * Exports the byte argument to a TIOS variable.
	 * If the name string object risk to be used later, use a copy of the string object
     * instead of the original.
	 *
	 * name The name of the TIOS variable
	 * b The byte to export
	 */
	public static void export (String name, byte b) {
		File.exportExpr(name, String.valueOf(b));
	}

	/**
	 * Exports the double argument to a TIOS variable.
	 * If the name string object risk to be used later, use a copy of the string object
     * instead of the original.
	 *
	 * name The name of the TIOS variable
	 * d The double to export
	 */
	public static void export (String name, double d) {
		File.exportExpr(name, String.valueOf(d));
	}

	/**
	 * Exports the char argument to a TIOS variable.
	 * If the name string object risk to be used later, use a copy of the string object
     * instead of the original.
	 *
	 * name The name of the TIOS variable
	 * c The char to export
	 */
	public static void export (String name, char c) {
		File.export(name, String.valueOf(c));
	}

	/**
	 * Exports the boolean argument to a TIOS variable.
	 * If the name string object risk to be used later, use a copy of the string object
     * instead of the original.
	 *
	 * name The name of the TIOS variable
	 * b The boolean to export
	 */
	public static void export (String name, boolean b) {
		File.export(name, String.valueOf(b));
	}

	/**
	 * Exports the String argument to a TIOS variable.
	 * If the name and s string objects risk to be used later, use a copy of the string object
     * instead of the original.
	 *
	 * name The name of the TIOS variable
	 * s The String to export
	 */
	public static void export (String name, String s) {
		String str = "\"";

		str += s;

		str += "\"";

		File.exportExpr(name, str);
	}

	/**
	 * Exports an expression to a TIOS variable.
	 * If the name and expr string objects risk to be used later, use a copy of the string object
     * instead of the original.
	 *
	 * name The name of the TIOS variable
	 * expr The String to export
	 */
	public static void exportExpr (String name, String expr) {
		//expr.append((char)22);

		expr += ((char)22);

		expr += name;

		System.exec(expr);
	}

	/**
	 * Opens an existing binary file in the current I/O mode (read only by default).
	 */
	public void open() {

		this.open(this.mode);
	}

	/**
	 * Opens a binary file.
	 @param mode The file mode
	 */
	public native void open(short mode) {

		char* m = NULL;

		this->mode = mode;

		switch (mode) {
			case File_MODE_OPEN_READ: {
				m = (char*) "rb";
				break;
			}
			case File_MODE_WRITE: {
				m = (char*) "wb";
				break;
			}
			case File_MODE_APPEND: {
				m = (char*) "ab";
				break;
			}
			case File_MODE_OPEN_UPDATE: {
				m = (char*) "r+b";
				break;
			}
			case File_MODE_UPDATE: {
				m = (char*) "w+b";
				break;
			}
			case File_MODE_UPDATE_APPEND: {
				m = (char*) "a+b";
				break;
			}
			case File_MODE_OPEN_READ_TEXT: {
				m = (char*) "r";
				break;
			}
			case File_MODE_WRITE_TEXT: {
				m = (char*) "w";
				break;
			}
			case File_MODE_APPEND_TEXT: {
				m = (char*) "a";
				break;
			}
			case File_MODE_OPEN_UPDATE_TEXT: {
				m = (char*) "r+";
				break;
			}
			case File_MODE_UPDATE_TEXT: {
				m = (char*) "w+";
				break;
			}
			case File_MODE_UPDATE_APPEND_TEXT: {
				m = (char*) "a+";
				break;
			}
		}

		this->file = fopen(this->name->value, m);

		this->error = (this->file == NULL);

		this->opened = !this->error;

	}

	/**
	 * Closes the stream.
	 */
	public void close() {
		if (this.type) {
			native {
				fputc (0, this->file);
				fputs (this->type->value, this->file);
				fputc (0, this->file);
				fputc (OTH_TAG, this->file);
			}
		}
		this.error = native ( fclose(this->file) );

		this.file = null;

		this.opened = false;
	}

	/**
	 * Returns the number of bytes that can be read from this file.
	 *
	 * @return	the number of bytes that can be read from this file.
	 */
	public native int available () {
		long int num = (this->file->alloc - (this->file->fpos - (char*)this->file->base));
		
	 	return ((this->mode < 6) ? num : (num - 3));
	}

    /**
     * Deletes the file denoted by this abstract pathname.
     *
     * @return  <code>true</code> if and only if the file is
     *          successfully deleted; <code>false</code> otherwise
     */
    public native boolean delete() {
		BOOL ok;
		char* vat = this->name->toVat_(this->name);

		ok = SymDel (vat);

		free(vat - (strlen(this->name->value)) - 1);

		return ok;
    }

	/**
	 * Tests whether the file denoted by this abstract pathname exists.
	 *
     * @return  <code>true</code> if and only if the file denoted by this
     *          abstract pathname exists; <code>false</code> otherwise
	*/
	public native boolean exists () {

		BOOL exists = TRUE;
		char *vat;

		vat = this->name->toVat_(this->name);

		if (SymFind(vat).folder == HS_NULL.folder)
		{
			exists = FALSE;
		}

		free(vat - (strlen(this->name->value)) - 1);

		return exists;
	}

	/**
	 * Tests whether the file named by this abstract pathname is archived.
	 *
	 * @return true if and only if the file denoted by this abstract pathname is archived
	*/
	public native boolean isArchived () {
		HSym sym;
		SYM_ENTRY *sym_entry;
		char *vat;
		BOOL archived = FALSE;

		vat = this->name->toVat_(this->name);

		sym = SymFind(vat);

		if (sym.folder == HS_NULL.folder)
		{
			this->error = TRUE;
		}
		else {
			sym_entry = DerefSym(sym);

			this->error = (sym_entry == NULL);

			archived = (sym_entry->flags.bits.archived == TRUE);
		}

		free(vat - (strlen(this->name->value)) - 1);

		return archived;
	}

	/**
	 * Tests whether the file named by this abstract pathname is hidden.
	 *
	 * @return true if and only if the file denoted by this abstract pathname is hidden
	*/
	public native boolean isHidden () {
		HSym sym;
		SYM_ENTRY *sym_entry;
		char *vat;
		BOOL hidden = FALSE;

		vat = this->name->toVat_(this->name);

		sym = SymFind(vat);

		if (sym.folder == HS_NULL.folder)
		{
			this->error = TRUE;
		}
		else {
			sym_entry = DerefSym(sym);

			this->error = (sym_entry == NULL);

			hidden = (sym_entry->flags.bits.hidden == TRUE);
		}

		free(vat - (strlen(this->name->value)) - 1);

		return hidden;
	}

	/**
	 * Tests whether the file named by this abstract pathname is locked.
	 *
	 * @return true if and only if the file denoted by this abstract pathname is locked
	*/
	public native boolean isLocked () {
		HSym sym;
		SYM_ENTRY *sym_entry;
		char *vat;
		BOOL locked = FALSE;

		vat = this->name->toVat_(this->name);

		sym = SymFind(vat);

		if (sym.folder == HS_NULL.folder)
		{
			this->error = TRUE;
		}
		else {
			sym_entry = DerefSym(sym);

			this->error = (sym_entry == NULL);

			locked = (sym_entry->flags.bits.locked == TRUE);
		}

		free(vat - (strlen(this->name->value)) - 1);

		return locked;
	}

    /**
     * Returns the length of the file denoted by this abstract pathname.
     *
     * @return  The length, in bytes, of the file denoted by this abstract
     *          pathname
     */
    public int length() {

		int len;

		if (this.file) {
			len = native ( this->file->alloc );
		}
		else {
			this.open();
			len = native ( this->file->alloc );
			this.close();
		}

		return len;
    }

    /**
     * Creates the directory named by this abstract pathname.
     *
     * @return  <code>true</code> if and only if the directory was
     *          created; <code>false</code> otherwise
     */
    public native boolean mkdir() {
		BOOL ok;
		char* vat = this->name->toVat_(this->name);

		ok = (FolderAdd (vat) != H_NULL);

		free(vat - (strlen(this->name->value)) - 1);

		return ok;
    }

    /**
     * Renames the file denoted by this abstract pathname.
     * If the dest string object risk to be used later, use a copy of the string object
     * instead of the original.
     *
     * @param  dest  The new abstract pathname for the named file
     *
     * @return  <code>true</code> if and only if the renaming succeeded;
     *          <code>false</code> otherwise
     */
    public native boolean renameTo(String dest) {
		BOOL ok;
		long int len = strlen(this->name->value);
		char* vat = this->name->toVat_(this->name);
		char* newVat = dest->toVat_(dest);

		if ((ok = SymMove (vat, newVat))) {
			this->name->finalize_(this->name);
			this->name = dest;
		}

		free(vat - len - 1);
		free(newVat - (strlen(dest->value)) - 1);

		if (!ok) {
			dest->finalize_(dest);
		}

		return ok;


    }

    /**
     * Removes the directory named by this abstract pathname including
     * all files in it.
     *
     * @return  <code>true</code> if and only if the directory was
     *          removed; <code>false</code> otherwise
     */
    public native boolean rmdir() {
		BOOL ok;
		char* vat = this->name->toVat_(this->name);

		ok = FolderDel (vat);

		free(vat - (strlen(this->name->value)) - 1);

		return ok;
    }

	/**
	 * Sets the file-pointer offset, measured from the beginning of this file, at which the next read or write occurs.
	 *
	 * @param pos the offset position, measured in bytes from the beginning of the file, at which to set the file pointer.
	 */
	public native void seek (int pos) {
		this->error = fseek(this->file, pos, SEEK_SET);
	}

	/**
	 * Sets if the file named by this abstract pathname is archived (in the archive memory instead of the RAM).
	 @param b a flag indicating whether this file component is archived.
	 @return true if and only if the operation succeeded; false otherwise
	*/
	public native boolean setArchived (boolean b) {
		BOOL ok;
		char* vat = this->name->toVat_(this->name);

		if (b) {
			ok = EM_moveSymToExtMem (vat, HS_NULL);
		}
		else {
			ok = EM_moveSymFromExtMem (vat, HS_NULL);
		}

		free(vat - (strlen(this->name->value)) - 1);

		return ok;
	}

	/**
	 * Sets if the file named by this abstract pathname is hidden (invisible for the TI-BASIC programs).
	 @param b a flag indicating whether this file is Hidden.
	 @return true if and only if the operation succeeded; false otherwise
	*/
	public native boolean setHidden (boolean b) {
		HSym sym;
		SYM_ENTRY *sym_entry;
		char *vat;

		vat = this->name->toVat_(this->name);

		sym = SymFind(vat);

		if (sym.folder == HS_NULL.folder)
		{
			return FALSE;
		}

		sym_entry = DerefSym(sym);

		this->error = (sym_entry == NULL);

		sym_entry->flags.bits.hidden = b;

		free(vat - (strlen(this->name->value)) - 1);

		return TRUE;
	}

	/**
	 * Sets if the file named by this abstract pathname is locked (cannot be altered).
	 @param b a flag indicating whether this file is locked.
	 @return true if and only if the operation succeeded; false otherwise
	*/
	public native boolean setLocked (boolean b) {
		HSym sym;
		SYM_ENTRY *sym_entry;
		char *vat;

		vat = this->name->toVat_(this->name);

		sym = SymFind(vat);

		if (sym.folder == HS_NULL.folder)
		{
			return FALSE;
		}

		sym_entry = DerefSym(sym);

		this->error = (sym_entry == NULL);

		sym_entry->flags.bits.locked = b;

		free(vat - (strlen(this->name->value)) - 1);

		return TRUE;
	}

	/**
	 * Skips over n bytes of data from this file.
	 * @param n the number of bytes to be skipped.
	 */
	public native void skip (int n) {
		this->error = fseek(this->file, n, SEEK_CUR);
	}

    /**
     * Writes an 8 bit byte.
     *
     * @param	val the byte value to be written
     */
	public native void writeByte(char val) {
		if (fputc(val, this->file) == EOF) {
			this->error = TRUE;
		}
		else {
			this->error = FALSE;
		}
	}

    /**
     * Writes a buffer of bytes up to the specified length.
     *
     * @param	buffer the buffer of bytes to be written
     * @param	len the specified length
     */
	public native void writeBytes(char[] buffer, short len) {
		if (this->mode > 5) {
			short ind = 0;
			while (len-- > 0) {
				if (fputc(buffer[ind++], this->file) == EOF) {
					this->error = TRUE;
					return ;
				}
				
				this->error = FALSE;
			}
		}
		else {
			this->error = (fwrite(buffer, len, 1, this->file) != 1);
		}
	}

    /**
     * Reads an 8 bit byte.
     *
     * @return	the byte value read
     */
	public char readByte() {
		char val;

		val = native ( fgetc(this->file) );

		if (val == EOF) {
			this.error = true;
		}
		else {
			this.error = false;
		}

		return val;
	}

    /**
     * Copies to a buffer of bytes up to the specified length bytes read from the stream.
     *
     * @param	buffer the buffer of bytes where the bytes will be copied
     * @param	len the specified length
     * @return  the effective number of bytes read.
     */
	public native short readBytesCount(char[] buffer, short len) {
		if (this->mode < 6) {
			this->error = (fread(buffer, len, 1, this->file) != 1) ;
			
			if (!this->error) {
				return len;
			}
			else {
				return 0;
			}
		}
		else {
			long int pos = 0;
			short tmp;
			
			while (len-- > 0) {
				tmp = fgetc(this->file);
				if (tmp == EOF) {
					this->error = TRUE;
					return pos;
				}
				else if (((char)tmp) != '\r') {
					buffer[pos] = (char)tmp;
				}
				else {
					buffer[pos] = '\n';
					len--;
				}
				
				pos ++;
			}
			
			this->error = FALSE;
			return pos;
		}
	}

    /**
     * Copies to a buffer of bytes up to the specified length bytes read from the stream.
     *
     * @param	buffer the buffer of bytes where the bytes will be copied
     * @param	len the specified length
     */
	public void readBytes(char[] buffer, short len) {
		this.readBytesCount(buffer, len);
	}

    /**
     * Frees the memory of system ressources used by this File
     * object.  If the stream is opened, it will be closed first.
     */
	public void finalize() {
		if (this.opened) {
			this.close();
		}

		if (this.name) {
			this.name.finalize();
		}

		if (this.type) {
			this.type.finalize();
		}

		native {
			free(this);
		}
	}
}