/*
 * QUSOFT MICROSYSTMES
 * Moka
 * Copyright 2002 Frdric Brown
 */

/*
 *  This file is part of Moka API.
 *
 *  Moka API is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Moka API is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Moka API; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package moka.event;

import moka.lang.Object;

/**
 * The Interrupt interfaces for executing code while a thread is running.
 *
 * @author  Frdric Brown
 * @version 1.1, 2002-11-03
 * @since   MDK1.0a
 */
public abstract class Interrupt {

	/** A flag to indicate if this timer is registered. */
	private boolean registered;

	/** Setting this flag to true will stop the interrupts' execution.*/
	public static boolean busy = false;

	/** The array of interrupts.*/
	private static Interrupt * array;

	/** The number of interrupts.*/
	private static int size;

	/** The actual interrupts' index.*/
	private static volatile int index;

	/** The capacity of the interrupts' array.*/
	private static int capacity;

	/** The old int handler.*/
	private static INT_HANDLER oldInt5Handler = native.GetIntVec(AUTO_INT_5);

	static {
		Interrupt.size = 0;
		Interrupt.index = -1;
		Interrupt.capacity = 5;
		Interrupt.array = native.malloc( 5 * sizeof(TInterrupt*));

		native.SetIntVec(AUTO_INT_5, Interrupt_int5Handler);
	}

	finally {
		native.free(Interrupt_array);
		native.SetIntVec(AUTO_INT_5, Interrupt_oldInt5Handler);
	}

	/**
	 * Creates a new Interrupt.
	 *
	 * @since MDK2.1
	 */
	public Interrupt () {
		this.registered = false;
	}

	/**
	 * Called when an InterruptEvent occurs.  This method
	 * checks the current interrupt then increment the interrupts' index.
	 * @since MDK2.1
	 */
	/*private static void onInterrupt () {

		if (! Interrupt.busy) {
			if (Interrupt.index < 0) {
				return;
			}

			if (Interrupt.index >= Interrupt.size) {
				Interrupt.index = 0;
			}

			if (Interrupt.array[Interrupt.index].check()) {
				Interrupt.array[Interrupt.index].run();
			}

			Interrupt.index++;
		}
	}*/


	/**
	 * Registers the specified interrupt.
	 *
	 * @param item the interrupt to be registered.
	 * @since MDK2.1
	 */
	public static native void register (Interrupt item) {
		if (! item->registered) {
			item->registered = TRUE;

			Interrupt_busy = TRUE;

	 		TInterrupt** array;

			if (Interrupt_index < 0) {
				Interrupt_index = 0;
			}

	 		if (Interrupt_size < Interrupt_capacity) {
	 			Interrupt_array[Interrupt_size] = item;
	 		}
	 		else {
	 			array = malloc(sizeof(TInterrupt*) * (Interrupt_capacity + 10));
	 			Interrupt_capacity += 10;

	 			memcpy(array, Interrupt_array, (Interrupt_size) * sizeof(TInterrupt*));

	 			array[Interrupt_size] = item;

	 			free(Interrupt_array);

	 			Interrupt_array = array;
	 		}

			Interrupt_size++;

			Interrupt_busy = FALSE;
		}

		#if ! __MOKA__
			Interrupt_busy;
			Interrupt_index;
			Interrupt_size;
			check_();
			run_();
		#endif
	}

	/**
	 * Unregisters the specified interrupt.
	 *
	 * @param item the interrupt to be unregistered.
	 * @since MDK2.1
	 */
	public static native void unregister (Interrupt item) {
		if (item->registered) {
			item->registered = FALSE;
			Interrupt_busy = TRUE;

	    	long int i;

	    	for (i = 0; i < Interrupt_size; i++) {
	    		if (Interrupt_array[i] == item) {
	    			break;
	    		}
	    	}

	   		memmove(Interrupt_array + i, Interrupt_array + i + 1, (Interrupt_size - i - 1) * sizeof(TInterrupt*));
	   		Interrupt_size--;
	   		if (Interrupt_index == Interrupt_size) {
	   			Interrupt_index == 0;
	   		}

	   		if (Interrupt_size < 1) {
	   			Interrupt_index = -1;
	   		}

	   		Interrupt_busy = FALSE;
	   	}
	}

    /**
     * Frees the ressources used by this Interrupt.
     *
     * @since MDK2.1
     */
	public void finalize () {
		if (this.registered) {
			Interrupt.unregister(this);
		}
		native.free(this);
	}

	/**
	 * Used by a thread to verify if this interrupt should be executed.
	 *
	 * @return true if this interrupt should be executed
	 */
	public abstract boolean check ();

	/**
	 * Handles the code to execute when this object interrupts.
	 */
	public abstract void run ();
}