/*
 * QUSOFT MICROSYSTMES
 * Moka
 * Copyright 2002 Frdric Brown
 */

/*
 *  This file is part of Moka API.
 *
 *  Moka API is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Moka API is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Moka API; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package moka.lang;

import moka.lang.Object;
import moka.lang.String;

/**
 * The <code>System</code> class contains several useful class fields
 * and methods. It cannot be instantiated.
 * <p>
 * Among the facilities provided by the <code>System</code> class
 * are input and output functions and most of the calc-dependent and AMS-dependent constants.
 *
 * @author  Frdric Brown
 * @version 1.2, 2002-10-25
 * @since   MDK1.0a
 */
 public abstract final class System {

	/** The number of arguments passed to the program. Should be only read. */
	public static short argCount;

	/** The number of arguments passed to the program. Should be only read. */
	private static char[][] argVec;

	/**
	 * Return an newly allocated array of <code>System.argCount</code> <code>String</code> instances built uppon the arguments passed to the program.
	 *
	 * @return an newly allocated array of <code>System.argCount</code> <code>String</code> instances built uppon the arguments passed to the program
	 * @since MDK2.21
	 */
	private static String[] getMainArgs() {
		short i;
        String[] args;

        if (System_argCount) {
        	args = native.malloc(sizeof(TString *) * System_argCount) ;
        }
        else {
        	args = NULL;
        }

        for (i = 0; i < System.argCount; i++) {
                args[i] = new String((char*)System.argVec[i]) ;
        }

        return args;
	}

    /**
     * Prints an object.  In fact, prints the string produced by the
     * <code>toString</code> method.
     *
     * @param      obj   The <code>Object</code> to be printed
     */
    public static void print (Object obj) {
    	String str = obj.toString();

    	native {
    		printf(str->value);
    	}

    	str.finalize();
    }

    /**
     * Prints an object and then terminates the line.  In fact, prints the
     * string produced by the <code>toString</code> method.
     *
     * @param      obj   The <code>Object</code> to be printed
     */
    public static void println (Object obj) {
    	System.print(obj);
    	native {
    		printf("\n");
    	}
    }

    /**
     * Prints a String then finalize the String.
     *
     * @param      s   The <code>String</code> to be printed
     */
    public static void print (String s) {
    	native {
    		printf(s->value);
    	}
    	s.finalize();
    }

    /**
     * Prints a String and then terminates the line and finalize the String.
     *
     * @param      s   The <code>String</code> to be printed
     */
    public static void println (String s) {
    	System.print(s);
    	native {
    		printf("\n");
    	}
    }

    /**
     * Prints an ANSI string.
     *
     * @param      s   The <code>ANSI string</code> to be printed
     */
    public abstract static void print (char[] s) ;

    /**
     * Prints an ANSI string and then terminates the line.
     *
     * @param      s   The <code>ANSI string</code> to be printed
     */
    public abstract static void println (char[] s) ;

    /**
     * Prints a char.
     *
     * @param      c   The <code>char</code> to be printed
     */
    /*Alias implemented for print (char c).*/
    public abstract static void print (char c) ;
    /*public static void print (char c) {

    	native {
    		printf("%c", c);
    	}
    }*/

    /**
     * Prints a char and then terminates the line.
     *
     * @param      c   The <code>char</code> to be printed
     */
    public abstract static void println (char c) ;
    /*public static void println (char c) {
    	System.print(c);
    	native {
    		printf("\n");
    	}
    }*/

    /**
     * Prints a byte.
     *
     * @param      b   The <code>byte</code> to be printed
     */
    /*Alias implemented for print (byte b).*/
    public abstract static void print (byte b) ;
    /*public static void print (byte b) {

    	native {
    		printf("%i", (short int)b);
    	}
    }*/

    /**
     * Prints a byte and then terminates the line.
     *
     * @param      s   The <code>byte</code> to be printed
     */
    /*Alias implemented for println (byte b).*/
    public abstract static void println (byte b) ;
    /*public static void println (byte b) {
    	System.print(b);
    	native {
    		printf("\n");
    	}
    }*/

    /**
     * Prints a short.
     *
     * @param      s   The <code>short</code> to be printed
     */
    /*Alias implemented for print (short s).*/
    public abstract static void print (short s) ;
    /*public static void print (short s) {

    	native {
    		printf("%i", s);
    	}
    }*/

    /**
     * Prints a short and then terminates the line.
     *
     * @param      s   The <code>short</code> to be printed
     */
    public abstract static void println (short s) ;
    /*public static void println (short s) {
    	System.print(s);
    	native {
    		printf("\n");
    	}
    }*/

    /**
     * Prints an int.
     *
     * @param      i   The <code>int</code> to be printed
     */
    /*Alias implemented for print (int i).*/
    public abstract static void print (int i) ;
    /*public static void print (int i) {

    	native {
    		printf("%ld", i);
    	}
    }*/

    /**
     * Prints an int and then terminates the line.
     *
     * @param      i   The <code>int</code> to be printed
     */
    public abstract static void println (int i) ;
    /*public static void println (int i) {
    	System.print(i);
    	native {
    		printf("\n");
    	}
    }*/

    /**
     * Prints a double.
     *
     * @param      d   The <code>double</code> to be printed
     */
    public static void print (double d) {
    	String str = String.valueOf(d);

    	native {
    		printf(str->value);
    	}

    	str.finalize();
    }

    /**
     * Prints a double and then terminates the line.
     *
     * @param      d   The <code>double</code> to be printed
     */
    public static void println (double d) {
    	System.print(d);
    	native {
    		printf("\n");
    	}
    }
    
    /**
     * Allocate a block of memory from the memory heap.
     * If not enough space exists for the new block or the argument size is zero, it returns null.
     * The memory allocated with this method can be freed by System.free method.
     * This method supports the garbage collection feature (if the main class of the project implements
     * Use_GarbageCollection) instead of the native C equivalent.
     *
     * @param      size   the size (in bytes) of the block.
     * @return     A pointer to a block of memory, or null if 0 is specified for size or if there is not enough memory.
     */
    /* Alias implemented */
    public abstract static void* malloc (int size);

    /**
     * Deallocates a memory block previously allocated by a call of System.malloc.
     *
     * @param      ptr   pointer to the memory block.
     */
    /* Alias implemented */
    public abstract static void free (void* ptr);
 }