/*
 * QUSOFT MICROSYSTMES
 * Moka
 * Copyright 2002 Frdric Brown
 */

/*
 *  This file is part of Moka API.
 *
 *  Moka API is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Moka API is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Moka API; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package moka.lang;

import moka.lang.Object;
import moka.lang.Character;

/**
 * The <code>String</code> class represents character strings. All
 * string literals in Moka programs, such as <code>"abc"</code>, should be
 * implemented as instances of this class for easy manipulation.
 * <p>
 * Strings are not constant; their values can be changed after they
 * are created.
 * Because String objects are mutable they can't be shared. For example:
 * <p><blockquote><pre>
 *     String str = "abc";
 * </pre></blockquote><p>
 * is not equivalent to:
 * <p><blockquote><pre>
 *     char[] chars = {'a', 'b', 'c', '\0'};
 *     String str = new String(chars);
 * </pre></blockquote><p>
 * The class <code>String</code> includes methods for examining
 * individual characters of the sequence, for comparing strings, for
 * searching strings, for extracting substrings, and for creating a
 * copy of a string with all characters translated to uppercase or to
 * lowercase.
 * <p>
 * String conversions are implemented through the method
 * <code>toString</code>, defined by <code>Object</code> and
 * inherited by all classes in Moka.
 * @author  Frdric Brown
 * @version 1.011, 2002-08-29
 * @since   MDK1.0a
 */

 public class String {
	/** The value used for character storage. */
 	private char* value;

 	/** A flag used to determine if the value is given by a reference.*/
 	private boolean ref;

    /**
     * Initializes a newly created <code>String</code> object so that it
     * represents an empty character sequence.
     */
 	public native String() {
 		this->value = (char*) "";
 		this->ref = TRUE;
 	}

    /**
     * Initializes a newly created <code>String</code> object so that it
     * represents the same sequence of characters as the argument; in other
     * words, the newly created string is a copy of the argument string.
     *
     * @param   original   a <code>String</code>.
     */
 	public String(String original) {
 		this.value = Character.copyValueOf(original.value);
 		this.ref = false;
 	}

    /**
     * Initializes a newly created <code>String</code> object so that its
     * value makes reference to a given ANSI string.
     *
     * @param   ansiString   an ANSI string.
     */
 	public String(char* ansiString) {
 		this.value = ansiString;
 		this.ref = true;
 	}

    /**
     * Initializes a newly created <code>String</code> object so that its
     * value makes reference or not to a given ANSI string.
     *
     * @param   ansiString   an ANSI string.
     * @param   ref          true if the value is a reference, false otherwise.
     */
 	public String(char* ansiString, boolean ref) {
 		this.value = ansiString;
 		this.ref = ref;
 	}

    /**
     * Appends the string representation of the <code>Object</code>
     * argument to this string.
     * <p>
     * The argument is converted to a string as if by its method
     * <code>toString</code>, and the characters of that
     * string are then appended to this string.
     *
     * @param   obj   an <code>Object</code>.
     * @return  a reference to this <code>String</code> object.
     */
    public String append(Object obj) {
		return this.concat(obj.toString());
    }

    /**
     * Appends the string to this string.
     * <p>
     * The characters of the <code>String</code> argument are appended, in
     * order, to the contents of this string, increasing the
     * length of this string buffer by the length of the argument. If the string object 
     * risk to be used later,use a copy of the string object
	 * instead of the original, the string being finalized before the method returns.
     *
     * @param   str   a string.
     * @return  a reference to this <code>String</code>.
     */
    public String append(String str) {
    	return this.concat(str);
    }

    /**
     * Appends the characters of a subarray of the
     * <code>char</code> array argument to this string.
     * <p>
     * Characters of the character array <code>str</code>, starting at
     * index <code>offset</code>, are appended, in order, to the contents
     * of this string. The length of this string increases
     * by the value of <code>len</code>.
     *
     * @param   str      the characters to be appended.
     * @param   offset   the index of the first character to append.
     * @param   len      the number of characters to append.
     * @return  a reference to this <code>String</code> object.
     */
    public String append(char* str, int offset, int len) {
    	return this.concat(new String(Character.copyValueOf(str, offset, len), false));
    }

    /**
     * Appends the string representation of the <code>boolean</code>
     * argument to the string.
     * <p>
     * The argument is converted to a string as if by the method
     * <code>String.valueOf</code>, and the characters of that
     * string are then appended to this string.
     *
     * @param   b   a <code>boolean</code>.
     * @return  a reference to this <code>String</code>.
     */
    public String append(boolean b) {
    	return this.concat(String.valueOf(b));
    }

    /**
     * Appends the string representation of the <code>char</code>
     * argument to this string.
     * <p>
     * The argument is appended to the contents of this string buffer.
     * The length of this string buffer increases by <code>1</code>.
     *
     * @param   c   a <code>char</code>.
     * @return  a reference to this <code>String</code> object.
     */
    public String append(char c) {
    	return this.concat(String.valueOf(c));
    }

    /**
     * Appends the string representation of the <code>byte</code>
     * argument to this string.
     * <p>
     * The argument is converted to a string as if by the method
     * <code>String.valueOf</code>, and the characters of that
     * string are then appended to this string.
     *
     * @param   b   an <code>byte</code>.
     * @return  a reference to this <code>String</code> object.
     */
    public String append(byte b) {
		return this.concat(String.valueOf(b));
    }

    /**
     * Appends the string representation of the <code>short</code>
     * argument to this string.
     * <p>
     * The argument is converted to a string as if by the method
     * <code>String.valueOf</code>, and the characters of that
     * string are then appended to this string.
     *
     * @param   s   an <code>short</code>.
     * @return  a reference to this <code>String</code> object.
     */
    public String append(short s) {
		return this.concat(String.valueOf(s));
    }

    /**
     * Appends the string representation of the <code>int</code>
     * argument to this string.
     * <p>
     * The argument is converted to a string as if by the method
     * <code>String.valueOf</code>, and the characters of that
     * string are then appended to this string.
     *
     * @param   i   an <code>int</code>.
     * @return  a reference to this <code>String</code> object.
     */
    public String append(int i) {
		return this.concat(String.valueOf(i));
    }

    /**
     * Appends the string representation of the <code>double</code>
     * argument to this string.
     * <p>
     * The argument is converted to a string as if by the method
     * <code>String.valueOf</code>, and the characters of that
     * string are then appended to this string.
     *
     * @param   d   a <code>double</code>.
     * @return  a reference to this <code>String</code> object.
     */
    public String append(double d) {
		return this.concat(String.valueOf(d));
    }

    /**
     * Concatenates the specified string to the end of this string.
     * <p>
     * If the length of the argument string is <code>0</code>, then this
     * <code>String</code> object is returned.  If the value made reference
     * to an ANSI string, this reference is not kept. Otherwise, a the value of this
     * <code>String</code> object is modified, representing a character
     * sequence that is the concatenation of the character sequence
     * represented by this <code>String</code> object and the character
     * sequence represented by the argument string.  In both case, the String to
     * concatenate is finalized.<p>
     *
     * @param   str   the <code>String</code> that is concatenated to the end
     *                of this <code>String</code>.
     * @return  a string that represents the concatenation of this object's
     *          characters followed by the string argument's characters.
     */
    public String concat(String str) {
		int len0 = native ( strlen(str->value) );
		char* buffer;

		if (len0 == 0) {
			if (this.ref) {
				buffer = Character.copyValueOf(this.value);
				this.ref = false;
			}
			else {
				buffer = this.value;
			}
		}
		else {
			buffer = Character.concat(this.value, str.value);

			if (this.ref) {
				this.ref = false;
			}
			else {
				native {
					free(this->value);
				}
			}
		}

		str.finalize();

		this.value = buffer;

		return this;
    }

    /**
     * Returns the character at the specified index. An index ranges
     * from <code>0</code> to <code>length() - 1</code>. The first character
     * of the sequence is at index <code>0</code>, the next at index
     * <code>1</code>, and so on, as for array indexing.
     *
     * @param      index   the index of the character.
     * @return     the character at the specified index of this string.
     *             The first character is at index <code>0</code>.
     */
    public char charAt(int index) {
        return this.value[index];
    }

    /**
     * Removes the characters in a substring of this <code>String</code>.
     * The substring begins at the specified <code>start</code> and extends to
     * the character at index <code>end - 1</code>. If
     * <code>start</code> is equal to <code>end</code>, no changes are made.
     *
     * @param      start  The beginning index, inclusive.
     * @param      end    The ending index, exclusive.
     * @return     This string with the characters of the specified substring removed.
     */
    public String delete(int start, int end) {
		int len = native ( strlen(this->value) - (end - start) );
		char* buffer = native ( malloc(len + 1) );

		native {
			buffer[0] = 0;
			strncat(buffer, this->value, start);
			strcat(buffer, this->value + end);
		}

		if (this.ref) {
			this.ref = false;
		}
		else {
			native {
				free(this->value);
			}
		}

		this.value = buffer;

		return this;
    }

    /**
     * Copies characters from this string into the destination character
     * array.
     * <p>
     * The first character to be copied is at index <code>srcBegin</code>;
     * the last character to be copied is at index <code>srcEnd-1</code>
     * (thus the total number of characters to be copied is
     * <code>srcEnd-srcBegin</code>). The characters are copied into the
     * subarray of <code>dst</code> starting at index <code>dstBegin</code>
     * and ending at index:
     * <p><blockquote><pre>
     *     dstbegin + (srcEnd-srcBegin) - 1
     * </pre></blockquote>
     *
     * @param      srcBegin   index of the first character in the string
     *                        to copy.
     * @param      srcEnd     index after the last character in the string
     *                        to copy.
     * @param      dst        the destination array.
     * @param      dstBegin   the start offset in the destination array.
     */
    public native void getChars(int srcBegin, int srcEnd, char* dst, int dstBegin) {
    	memcpy(&dst[dstBegin], &this->value[srcBegin], srcEnd - srcBegin);
    }

    /**
     * Returns the index within this string of the first occurrence of the
     * specified substring, starting at the specified index.  The integer
     * returned is the index of the first occurrence of the
     * specified substring.
     * If no such value exists, then -1 is returned.
     *
     * @param   str         the substring for which to search.
     * @param   fromIndex   the index from which to start the search.
     * @return  the index within this string of the first occurrence of the
     *          specified substring, starting at the specified index.
     */
    public native int indexOf(String str, int fromIndex) {
       	char* begin = strstr(&this->value[fromIndex], str->value);

    	if (begin) {
    		return (long int)(begin - this->value);
    	}

    	return -1;
    }

    /**
     * Returns the index within this string of the last occurrence of the
     * specified substring, searching backward starting at the specified index.
     * The integer returned is the largest value <i>k</i> such that:
     * <blockquote><pre>
     *     k &lt;= Math.min(fromIndex, str.length()) && this.startsWith(str, k)
     * </pre></blockquote>
     * If no such value of <i>k</i> exists, then -1 is returned.
     *
     * @param   str         the substring to search for.
     * @param   fromIndex   the index to start the search from.
     * @return  the index within this string of the last occurrence of the
     *          specified substring.
     */
    public native int lastIndexOf(String str, int fromIndex) {
    	long int pos = -1;
    	long int next;
    	long int ind = 0;
    	char* begin;

    	do {
    		if (ind > fromIndex) {
    			break;
    		}
    		begin = strstr(&this->value[ind], str->value);
    		if (begin) {
    			next = (long int)(begin - this->value);
				if (next <= fromIndex) {
    				pos = next;
    			}
    			ind = next + 1;
    		}
    	}
    	while (begin);

    	return pos;
    }

    /**
     * Tests if two string regions are equal.
     * <p>
     * A substring of this <tt>String</tt> object is compared to a substring
     * of the argument other. The result is true if these substrings
     * represent identical character sequences. The substring of this
     * <tt>String</tt> object to be compared begins at index <tt>toffset</tt>
     * and has length <tt>len</tt>. The substring of other to be compared
     * begins at index <tt>ooffset</tt> and has length <tt>len</tt>.
     *
     * @param   toffset   the starting offset of the subregion in this string.
     * @param   other     the string argument.
     * @param   ooffset   the starting offset of the subregion in the string
     *                    argument.
     * @param   len       the number of characters to compare.
     * @return  <code>true</code> if the specified subregion of this string
     *          exactly matches the specified subregion of the string argument;
     *          <code>false</code> otherwise.
     */
    public native boolean regionMatches(int toffset, String other, int ooffset, int len) {
    	return strncmp(this->value + toffset, other->value + ooffset, len);
    }

    /**
     * Tests if two string regions are equal.
     * <p>
     * A substring of this <tt>String</tt> object is compared to a substring
     * of the argument <tt>other</tt>. The result is <tt>true</tt> if these
     * substrings represent character sequences that are the same, ignoring
     * case if and only if <tt>ignoreCase</tt> is true. The substring of
     * this <tt>String</tt> object to be compared begins at index
     * <tt>toffset</tt> and has length <tt>len</tt>.  The result is <tt>false</tt>
     * otherwize. The substring of
     * <tt>other</tt> to be compared begins at index <tt>ooffset</tt> and
     * has length <tt>len</tt>.
     *
     * @param   ignoreCase   if <code>true</code>, ignore case when comparing
     *                       characters.
     * @param   toffset      the starting offset of the subregion in this
     *                       string.
     * @param   other        the string argument.
     * @param   ooffset      the starting offset of the subregion in the string
     *                       argument.
     * @param   len          the number of characters to compare.
     * @return  <code>true</code> if the specified subregion of this string
     *          matches the specified subregion of the string argument;
     *          <code>false</code> otherwise. Whether the matching is exact
     *          or case insensitive depends on the <code>ignoreCase</code>
     *          argument.
     */
    public boolean regionMatches(boolean ignoreCase, int toffset, String other, int ooffset, int len) {
		int i;
		int dif = ooffset - toffset;

		for (i = toffset; i < toffset + len; i++) {
			if (!(this.value[i] == other.value[i + dif]) && (!ignoreCase || !(this.value[i] == (other.value[i + dif] + 32)) && !(this.value[i] == (other.value[i + dif] - 32)))) {
				return false;
			}
		}

		return true;
	}

    /**
     * Returns a new string resulting from replacing all occurrences of
     * <code>oldChar</code> in this string with <code>newChar</code>.
     * <p>
     * If the character <code>oldChar</code> does not occur in the
     * character sequence represented by this <code>String</code> object,
     * then a reference to this <code>String</code> object is returned.
     * Otherwise, this string object will be modified to
     * represents a character sequence identical to the character sequence
     * represented by this <code>String</code> object, except that every
     * occurrence of <code>oldChar</code> is replaced by an occurrence
     * of <code>newChar</code>.
     *
     * @param   oldChar   the old character.
     * @param   newChar   the new character.
     * @return  this string with every
     *          occurrence of <code>oldChar</code> replaced by <code>newChar</code>.
     */
    public String replace(char oldChar, char newChar) {
    	int i;
    	int len = native { strlen(this->value) };

    	if (this.ref) {
    		this.ref = false;
    		this.value = Character.copyValueOf(this.value);
    	}

    	for (i = 0; i < len; i++) {
    		if (this.value[i] == oldChar) {
    			this.value[i] = newChar;
    		}
    	}

    	return this;
    }

    /**
     * Returns a new string resulting from replacing all occurrences of
     * <code>oldStr</code> substring in this string with <code>newStr</code>.
     * <p>
     * If the substring <code>oldStr</code> does not occur in the
     * character sequence represented by this <code>String</code> object,
     * then a reference to this <code>String</code> object is returned.
     * Otherwise, this string object will be modified to
     * represents a character sequence identical to the character sequence
     * represented by this <code>String</code> object, except that every
     * occurrence of <code>oldStr</code> substring is replaced by an occurrence
     * of <code>newStr</code>.
     *
     * @param   oldChar   the old character.
     * @param   newChar   the new character.
     * @return  this string with every
     *          occurrence of <code>oldStr</code> replaced by <code>newStr</code>.
     */
    public native String replace(String oldStr, String newStr) {
    	long int len = strlen(this->value);
    	long int len0 = strlen(oldStr->value);
    	long int len1 = strlen(newStr->value);
    	long int dif = len1 - len0;
    	long int prevInd = 0;
    	long int ind = 0;
    	long int indBuf = 0;
    	long int nbr = 0;
    	char* begin;
    	char* buffer;

    	do {
    		begin = strstr(&this->value[ind], oldStr->value);
    		if (begin) {
				ind = (long int)(begin - this->value) + len0;
				nbr++;
    		}
    	}
    	while (begin);

    	buffer = malloc(len + dif * nbr + 1);

    	ind = 0;

    	do {
    		begin = strstr(&this->value[ind], oldStr->value);
    		if (begin) {
				ind = (long int)(begin - this->value);
				memcpy(&buffer[indBuf], &this->value[prevInd], ind - prevInd);
				indBuf += ind - prevInd;
				memcpy(&buffer[indBuf], newStr->value, len1);
				ind += len0;
				indBuf += len1;
				prevInd = ind;
    		}
    	}
    	while (begin);

    	if (this->ref) {
    		this->ref = FALSE;
    	}
    	else {
    		free(this->value);
    	}

    	memcpy(&buffer[indBuf], &this->value[prevInd], len - prevInd + 1);

    	this->value = buffer;

    	return this;
    }

    /**
     * Tests if this string starts with the specified prefix beginning
     * a specified index.
     *
     * @param   prefix    the prefix.
     * @param   toffset   where to begin looking in the string.
     * @return  <code>true</code> if the character sequence represented by the
     *          argument is a prefix of the substring of this object starting
     *          at index <code>toffset</code>; <code>false</code> otherwise.
     */
 	public native boolean startsWith(String prefix, int toffset) {
       	char* begin = strstr(&this->value[toffset], prefix->value);

    	if ((long int)(begin - this->value) == toffset) {
    		return TRUE;
    	}

    	return FALSE;
 	}

    /**
     * Returns a new string that is a substring of this string. The
     * substring begins at the specified <code>beginIndex</code> and
     * extends to the character at index <code>endIndex - 1</code>.
     * Thus the length of the substring is <code>endIndex-beginIndex</code>.
     *
     * @param      beginIndex   the beginning index, inclusive.
     * @param      endIndex     the ending index, exclusive.
     * @return     the specified substring.
     */
    public native String substring(int beginIndex, int endIndex) {
    	TString* str = TString_();
    	long int len = endIndex-beginIndex;

    	str->value = malloc(len + 1);
		str->ref = FALSE;

		if (len > 0) {
			memcpy(str->value, &this->value[beginIndex], len);
		}

		str->value[len] = '\0';


    	return str;
    }

    /**
     * Converts all of the characters in this <code>String</code> to lower
     * case.
     *
     * @return  the <code>String</code>, converted to lowercase.
     */
    public String toLowerCase() {
    	int len = native ( strlen(this->value) );
    	int i;
    	char c;

    	if (this.ref) {
    		this.ref = false;
    		this.value = Character.copyValueOf(this.value);
    	}

		native {
			for (i = 0; i < len; i++) {
				c = this->value[i];
				if (c >= 'A' && c <= 'Z' || (unsigned char)c >= 192 && (unsigned char)c <= 214 || (unsigned char)c >= 216 && (unsigned char)c <= 221) {
					this->value[i] = c + 32;
				}
			}
		}

    	return this;
    }

	/**
     * Returns a copy of this string.
     *
     * @return  a copy of the string.
     */
 	public String toString() {
 		return new String(this);
 	}

    /**
     * Converts all of the characters in this <code>String</code> to upper
     * case.
     *
     * @return  the <code>String</code>, converted to uppercase.
     */
    public String toUpperCase() {
    	int len = native ( strlen(this->value) );
    	int i;
    	char c;

    	if (this.ref) {
    		this.ref = false;
    		this.value = Character.copyValueOf(this.value);
    	}

    	native {
			for (i = 0; i < len; i++) {
				c = this->value[i];
				if (c >= 'a' && c <= 'z' || (unsigned char)c >= 224 && (unsigned char)c <= 246 || (unsigned char)c >= 248 && (unsigned char)c <= 253) {
					this->value[i] = c - 32;
				}
			}
		}

    	return this;
    }

    /**
     * Returns this string, minus the leading and trailing whitespace.
     *
     * @return  This string with leading and trailing white
     *          space removed, or this string if it has no leading or
     *          trailing white space.
     */
    public String trim() {
    	int len = native { strlen(this->value) };
    	int indB = 0;
    	int indE = len - 1;
    	String str;

		while (this.value[indB] == ' ' && indB < len) indB++;
		while (this.value[indE] == ' ' && indE >= 0) indE--;

		if (indB == 0 && indE == len - 1) {
			return this;
		}

		str = this.substring(indB, indE + 1);

		if (this.ref) {
			this.ref = false;
		}
		else {
			native {
				free(this->value);
			}
		}

		this.value = str.value;

		str.finalize();

		return this;
    }

    /**
     * Tests if this string ends with the specified suffix.
     *
     * @param   suffix   the suffix.
     * @return  <code>true</code> if the character sequence represented by the
     *          argument is a suffix of the character sequence represented by
     *          this object; <code>false</code> otherwise. Note that the
     *          result will be <code>true</code> if the argument is the
     *          empty string or is equal to this <code>String</code> object.
     */
    public native boolean endsWith(String suffix) {
		return !strcmp(&this->value[strlen(this->value) - strlen(suffix->value)], suffix->value);
    }

    /**
     * Compares this string to the specified object.
     * The result is <code>true</code> if and only if the argument is not
     * <code>null</code> and is a <code>String</code> object that represents
     * the same sequence of characters as this object.
     *
     * @param   anObject   the object to compare this <code>String</code>
     *                     against.
     * @return  <code>true</code> if the <code>String </code>are equal;
     *          <code>false</code> otherwise.
     */
    public boolean equals(Object object) {
    	if (this == object) {
    		return true;
    	}

    	if (object instanceof String) {
    		native {
    			return !strcmp(this->value, ((TString*)object)->value);
    		}
    	}

    	return false;
    }

    /**
     * Compares this <code>String</code> to another <code>String</code>,
     * ignoring case considerations.  Two strings are considered equal
     * ignoring case if they are of the same length, and corresponding
     * characters in the two strings are equal ignoring case.
     *
     * @param   anotherString   the <code>String</code> to compare this
     *                          <code>String</code> against.
     * @return  <code>true</code> if the argument is not <code>null</code>
     *          and the <code>String</code>s are equal,
     *          ignoring case; <code>false</code> otherwise.
     */
    public boolean equalsIgnoreCase(String anotherString) {
		int len = native ( strlen(this->value) );
		int len0 = native ( strlen(anotherString->value) );
		int i;

		if (len - len0 != 0) {
			return false;
		}

		for (i = 0; i < len; i++) {
			if (!(this.value[i] == anotherString.value[i]) && !(this.value[i] == (anotherString.value[i] + 32)) && !(this.value[i] == (anotherString.value[i] - 32))) {
				return false;
			}
		}

		return true;
    }

    /**
     * Frees the memory of system ressources used by this String
     * object.  If the value of the String do not makes reference to
     * an ANSI String, the ANSI string memory will be freed, otherwise no.
     *
     */
 	public void finalize () {
 		if (!this.ref) {
 			native {
 				free(this->value);
 			}
 		}
 		native {
 			free(this);
 		}
 	}

    /**
     * Compares two strings lexicographically.
     * The comparison is based on the ASCII value of each character in
     * the strings. The character sequence represented by this
     * <code>String</code> object is compared lexicographically to the
     * character sequence represented by the argument string. The result is
     * a negative integer if this <code>String</code> object
     * lexicographically precedes the argument string. The result is a
     * positive integer if this <code>String</code> object lexicographically
     * follows the argument string. The result is zero if the strings
     * are equal; <code>compareTo</code> returns <code>0</code> exactly when
     * the {@link #equals(Object)} method would return <code>true</code>.
     * <p>
     * This is the definition of lexicographic ordering. If two strings are
     * different, then either they have different characters at some index
     * that is a valid index for both strings, or their lengths are different,
     * or both. If they have different characters at one or more index
     * positions, let <i>k</i> be the smallest such index; then the string
     * whose character at position <i>k</i> has the smaller value, as
     * determined by using the &lt; operator, lexicographically precedes the
     * other string. In this case, <code>compareTo</code> returns the
     * difference of the two character values at position <code>k</code> in
     * the two string -- that is, the value:
     * <blockquote><pre>
     * this.charAt(k)-anotherString.charAt(k)
     * </pre></blockquote>
     * @param   anotherString   the <code>String</code> to be compared.
     * @return  the value <code>0</code> if the argument string is equal to
     *          this string; a value less than <code>0</code> if this string
     *          is lexicographically less than the string argument; and a
     *          value greater than <code>0</code> if this string is
     *          lexicographically greater than the string argument.
     */
 	public native short compareTo(String anotherString) {
 		return strcmp(this->value, anotherString->value);
 	}

    /**
     * Returns the length of this string.
     * The length is equal to the number of
     * characters in the string.
     *
     * @return  the length of the sequence of characters represented by this
     *          object.
     */
    public native int length() {
        return strlen(this->value);
    }

    /**
     * Compares two strings lexicographically.
     * The comparison is based on the ASCII value of each character in
     * the strings. The character sequence represented by this
     * <code>String</code> object is compared lexicographically to the
     * character sequence represented by the argument ANSI string. The result is
     * a negative integer if this <code>String</code> object
     * lexicographically precedes the argument ANSI string. The result is a
     * positive integer if this <code>String</code> object lexicographically
     * follows the argument string. The result is zero if the strings
     * are equal.
     * <p>
     * This is the definition of lexicographic ordering. If two strings are
     * different, then either they have different characters at some index
     * that is a valid index for both strings, or their lengths are different,
     * or both. If they have different characters at one or more index
     * positions, let <i>k</i> be the smallest such index; then the string
     * whose character at position <i>k</i> has the smaller value, as
     * determined by using the &lt; operator, lexicographically precedes the
     * other string. In this case, <code>strcmp</code> returns the
     * difference of the two character values at position <code>k</code> in
     * the two string -- that is, the value:
     * <blockquote><pre>
     * this.charAt(k)-ansiString[k]
     * </pre></blockquote>
     * @param   ansiString   the <code>ANSI String</code> to be compared.
     * @return  the value <code>0</code> if the argument ANSI string is equal to
     *          this string; a value less than <code>0</code> if this string
     *          is lexicographically less than the ANSI string argument; and a
     *          value greater than <code>0</code> if this string is
     *          lexicographically greater than the ANSI string argument.
     */
 	public native short strcmp(char* ansiString) {
 		return strcmp(this->value, ansiString);
 	}

    /**
     * Returns the string representation of the <code>char</code> argument.
     *
     * @param   c   an <code>char</code>.
     * @return  a string representation of the <code>char</code> argument.
     */
    public static String valueOf(char c) {
        native {
        	char buf[] = {c, '\0'} ;
        }
        return new String(Character.copyValueOf((char*)buf), false);
    }

    /**
     * Returns the string representation of the <code>boolean</code> argument.
     *
     * @param   b   a <code>boolean</code>.
     * @return  if the argument is <code>true</code>, a string equal to
     *          <code>"true"</code> is returned; otherwise, a string equal to
     *          <code>"false"</code> is returned.
     */
    public static String valueOf(boolean b) {
	return b ? "true" : "false";
    }

    /**
     * Returns the string representation of the <code>byte</code> argument.
     *
     * @param   b   a <code>byte</code>.
     * @return  a string representation of the <code>byte</code> argument.
     */
    public native static String valueOf(byte b) {
		char vec[4];

		sprintf(vec, "%i", (short int)b);

		return TString_char_p_BOOL(strcpy(malloc(strlen(vec) + 1), vec), FALSE);
    }

    /**
     * Returns the string representation of the <code>short</code> argument.
     *
     * @param   s   an <code>short</code>.
     * @return  a string representation of the <code>short</code> argument.
     */
    public native static String valueOf(short s) {
		char vec[7];

		sprintf(vec, "%i", s);

		return TString_char_p_BOOL(strcpy(malloc(strlen(vec) + 1), vec), FALSE);
    }

    /**
     * Returns the string representation of the <code>int</code> argument.
     *
     * @param   i   an <code>int</code>.
     * @return  a string representation of the <code>int</code> argument.
     */
    public native static String valueOf(int i) {
		char vec[12];

		sprintf(vec, "%li", i);

		return TString_char_p_BOOL(strcpy(malloc(strlen(vec) + 1), vec), FALSE);
    }

    /**
     * Returns the string representation of the <code>double</code> argument.
     *
     * @param   d   an <code>double</code>.
     * @return  a string representation of the <code>double</code> argument.
     */
    public native static String valueOf(double d) {
		char vec[22];
		sprintf(vec, "%f", d);

		return TString_char_p_BOOL(strcpy(malloc(strlen(vec) + 1), vec), FALSE);
    }

    /**
     * Returns the string representation of the ANSI string
     * argument. The contents of the character array are copied; subsequent
     * modification of the ANSI string does not affect the newly
     * created string.
     *
     * @param   data   a <code>char</code> array.
     * @return  a newly allocated string representing the same sequence of
     *          characters contained in the ANSI string argument.
     */
    public static String valueOf(char* ansiString) {
    	return new String(Character.copyValueOf(ansiString), false);
    }

    /**
     * Returns the string representation of the <code>char</code> array
     * argument. The contents of the character array are copied; subsequent
     * modification of the character array does not affect the newly
     * created string.
     *
     * @param   data   a <code>char</code> array.
     * @param   length   the length of the <code>char</code> array.
     * @return  a newly allocated string representing the same sequence of
     *          characters contained in the character array argument.
     */
    public static String valueOf(char* data, int length) {
    	return new String(Character.copyValueOf(data, 0, length), false);
    }
}