/*
 * QUSOFT MICROSYSTMES
 * Moka
 * Copyright 2002 Frdric Brown
 */

/*
 *  This file is part of Moka API.
 *
 *  Moka API is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Moka API is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Moka API; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package moka.lang;

import moka.lang.Object;
import moka.lang.String;

/**
 * The <code>Character</code> class wraps a value of the primitive
 * type <code>char</code> in an object. An object of type
 * <code>Character</code> contains a single field whose type is
 * <code>char</code>.
 * <p>
 * In addition, this class provides several methods for determining
 * a character's category (lowercase letter, digit, etc.) and for converting
 * characters from uppercase to lowercase and vice versa. Plus, this class offers
 * several method for ANSI string manipulation.
 * <p>
 * Character information is based on the ASCII.
 * @author  Frdric Brown
 * @version 1.0, 2002-08-29
 * @since   MDK1.0a
 */
public class Character {

    /**
     * The value of the <code>Character</code>.
     */
	private char value;

    /**
     * Constructs a newly allocated <code>Character</code> object that
     * represents the primitive <code>char</code> argument.
     *
     * @param   value   the value to be represented by the <code>char</code>.
     */
    public Character(char value) {
		this.value = value;
    }

    /**
     * Compares this object against the specified object.
     * The result is <code>true</code> if and only if the argument is not
     * <code>null</code> and is a <code>Character</code> object that
     * represents the same <code>char</code> value as this object.
     *
     * @param   obj   the object to compare with.
     * @return  <code>true</code> if the objects are the same;
     *          <code>false</code> otherwise.
     */
    public boolean equals(Object obj) {
        if (obj instanceof Character) {
            	return this.value == ((Character)obj).value;
        }
        return false;
    }
    /**
     * Returns a <code>String</code> object representing this
     * <code>Character</code>'s value.  The result is a string of
     * length 1 whose sole component is the primitive
     * <code>char</code> value represented by this
     * <code>Character</code> object.
     *
     * @return  a string representation of this object.
     */
    public String toString() {
        return String.valueOf(this.value);
    }

    /**
     * Returns the value of this <code>Character</code> object.
     * @return  the primitive <code>char</code> value represented by
     *          this object.
     */
    public char charValue() {
        return this.value;
    }

    /**
     * Concatenates another ANSI string to the end of the specified ANSI string.
     * <p>
     * If the length of the argument string is <code>0</code>, then this
     * ANSI String object is copied then returned. Otherwise, a new
     * ANSI String object is created, representing a character
     * sequence that is the concatenation of the character sequence
     * represented by this ANSI String object and the character
     * sequence represented by the argument string.<p>
     *
     * @param   ansiString   the ANSI String that will be followed by the other ANSI string.
     * @param   anotherAnsiString   the ANSI String that is concatenated to the end
     *                of the specified ANSI String.
     * @return  an ANSI string that represents the concatenation of the specified ANSI string
     *          characters followed by the other ANSI string argument's characters.
     */
	public native static char* concat(char* ansiString, char* anotherAnsiString) {
		long int len0 = strlen(ansiString);
		long int len1 = strlen(anotherAnsiString);
		char* str = malloc(len0 + len1 + 1);
		strcpy(str, ansiString);
		return strcat(str, anotherAnsiString);
	}

    /**
     * Returns a copy of the ANSI string
     * argument. The contents of the character array are copied; subsequent
     * modification of the character array does not affect the newly
     * created ANSI string.
     *
     * @param   data   an ANSI string.
     * @return  a newly allocated  ANSI string representing the same sequence of
     *          characters contained in the ANSI string argument.
     */
	public native static char* copyValueOf(char* ansiString) {
		return strcpy(malloc(strlen(ansiString) + 1), ansiString);
	}

    /**
     * Returns the ANSI string representation of a specific subarray of the
     * <code>char</code> array argument.
     * <p>
     * The <code>offset</code> argument is the index of the first
     * character of the subarray. The <code>count</code> argument
     * specifies the length of the subarray. The contents of the subarray
     * are copied; subsequent modification of the character array does not
     * affect the newly created string.
     *
     * @param   data     the character array.
     * @param   offset   the initial offset into the value of the
     *                  ANSI string.
     * @param   count    the length of the value of the <code>String</code>.
     * @return  a string representing the sequence of characters contained
     *          in the subarray of the character array argument.
     */
	public native static char* copyValueOf(char* data, int offset, int count) {
		char* ansiString;

		ansiString = malloc(count + 1);
    	memcpy(ansiString, &data[offset], count);
    	ansiString[count] = '\0';

    	return ansiString;
	}

	/**
	 * Converts a String object into a TIOS VAT String then finalizes the String.
	 *
	 @param string The String object
	 @return The string converted in TIOS VAT string
	 */
	public native static char* toVat(String string)
	{
		int len = strlen(string->value);
		char *vat = malloc(len + 2);

		vat[0] = '\0';

		vat = strcpy(&vat[1], string->value) + len;

		string->finalize_(string);

		return vat;
	}

    /**
     * Determines if the specified character is a digit.
	 *
     * @param   ch   the character to be tested.
     * @return  <code>true</code> if the character is a digit;
     *          <code>false</code> otherwise.
     */
    public static boolean isDigit(char ch) {
    	return (ch >= '0' && ch <= '9');
    }

    /**
     * Determines if the specified character is a letter.
     *
     * @param   ch   the character to be tested.
     * @return  <code>true</code> if the character is a letter;
     *          <code>false</code> otherwise.
     */
    public static boolean isLetter(char ch) {
    	return Character.isLowerCase(ch) || Character.isUpperCase(ch);
    }

   /**
     * Determines if the specified character is a lowercase character.
     * <p>
     * The following are examples of lowercase characters:
     * <p><blockquote><pre>
     * a b c d e f g h i j k l m n o p q r s t u v w x y z
     * </pre></blockquote>
     *
     * @param   ch   the character to be tested.
     * @return  <code>true</code> if the character is lowercase;
     *          <code>false</code> otherwise.
     */
    public native static boolean isLowerCase(char ch) {

    	return ch >= 'a' && ch <= 'z' || (unsigned char)ch >= 224 && (unsigned char)ch <= 246 || (unsigned char)ch >= 248 && (unsigned char)ch <= 253;
    }

   /**
     * Determines if the specified character is an uppercase character.
     * <p>
     * The following are examples of uppercase characters:
     * <p><blockquote><pre>
     * A B C D E F G H I J K L M N O P Q R S T U V W X Y Z
     * </pre></blockquote>
     *
     * @param   ch   the character to be tested.
     * @return  <code>true</code> if the character is uppercase;
     *          <code>false</code> otherwise.
     */
    public native static boolean isUpperCase(char ch) {

    	return ch >= 'A' && ch <= 'Z' || (unsigned char)ch >= 192 && (unsigned char)ch <= 214 || (unsigned char)ch >= 216 && (unsigned char)ch <= 221;
    }

    /**
     * Returns a new ANSI string that is a substring of the specified ANSI string. The
     * substring begins at the specified <code>beginIndex</code> and
     * extends to the character at index <code>endIndex - 1</code>.
     * Thus the length of the substring is <code>endIndex-beginIndex</code>.
     * <p>
     *
     * @param      ansiString   the specified ANSI string.
     * @param      beginIndex   the beginning index, inclusive.
     * @param      endIndex     the ending index, exclusive.
     * @return     the specified substring.
     */
    public native static char* substring(char* ansiString, int beginIndex, int endIndex) {
		return strncpy(malloc(endIndex-beginIndex + 1), &ansiString[beginIndex], endIndex-beginIndex);
    }

    /**
     * Converts the character argument to lowercase using case
     * mapping information from the TI-92 Plus Character Codes chart.
     *
     * @param   ch   the character to be converted.
     * @return  the lowercase equivalent of the character, if any;
     *          otherwise, the character itself.
     */
    public native static char toLowerCase(char ch) {

		if (ch >= 'A' && ch <= 'Z') {
    		ch = ch + 32;
    	}

    	return ch;
    }

    /**
     * Converts the character argument to uppercase using case mapping
     * information from the TI-92 Plus Character Codes chart.
     *
     * @param   ch   the character to be converted.
     * @return  the uppercase equivalent of the character, if any;
     *          otherwise, the character itself.
     */
    public native static char toUpperCase(char ch) {

    	if (ch >= 'a' && ch <= 'z') {
    		ch = ch - 32;
    	}

    	return ch;
    }
}