(*
 * QUSOFT MICROSYSTMES
 * Moka
 * Copyright 2003 Frdric Brown
 *)

(*
 *  This file is part of Moka.
 *
 *  Moka is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.

 *  Moka is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.

 *  You should have received a copy of the GNU General Public License
 *  along with Moka; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *)

(*
 * Reflect unit contains a very large number of classes used to instantiate
 * abstract entities (such as Classes, Methods, etc.) during the conversion
 * process.
 *)

unit Reflect;

interface

Uses Util;

type
  TFileLinked = class;
  TPack = class;
  TClasse = class;
  TMethode = class;
  TAttribute = class;
  TParameter = class;
  TMethodVector = class;
  TClassVector = class;
  TAttributeVector = class;
  TParameterVector = class;

  TFileLinked = class

  private

  public
    fileText: AnsiString;
    words: TStringVector;
    outText: AnsiString;
    constructor Create;
    procedure writeString(s: AnsiString);
    procedure writeLine(s: AnsiString);
    procedure nextLine();
  end;

  TPack = class (TFileLinked)

  private

  public
    name: AnsiString;
    classes: TClassVector;
    constructor Create(name: AnsiString);
  end;

  TClasse = class (TFileLinked)

  private

  public
    pack: TPack;
    sup: TClasse;
    supName: AnsiString;
    ifsName: TStringVector;
    fileName: AnsiString;

    name: AnsiString;
    meth: TMethodVector;
    attr: TAttributeVector;
   // cons: TAttributeVector;
    inst: Boolean;
    inclu: TStringVector;
    ifs: TClassVector;
    visibility: Longint;

    used: Boolean;
    inter: Boolean;
    fin: Boolean;
    abst: Boolean;

    constructor Create(name: AnsiString);
    procedure setUsed (b: Boolean);
    procedure setInst (b: Boolean);
    function hasValuedStat(): Boolean;
    function hasValuedAttr(): Boolean;
    function hasStat(): Boolean;
    function ext(sup: AnsiString): Boolean;
    function implAll(inte: AnsiString): Boolean;
    function impl(inte: AnsiString): Boolean;
    function instof(classe: AnsiString): Boolean;
    function defined(mName: AnsiString; params: TParameterVector): Boolean;
    function returnAll(mSign: AnsiString): TMethode;
    function return(mSign: AnsiString): TMethode;
    function getMVec(str: AnsiString): TMethodVector;
  end;

  TType = class

  private

  public
    strMoka: AnsiString;
    strC: AnsiString;
    strP: AnsiString;
    constructor Create (strMoka: AnsiString; strC: AnsiString);
    function est (s: AnsiString): Boolean; dynamic;
    function format(s: AnsiString): AnsiString; dynamic;
  end;

  TMokaBoolean = class (TType)

  private

  public
    constructor Create;
    function est (s: AnsiString): Boolean; override;
    function format (s: AnsiString): AnsiString; override;
  end;

  TMokaChar = class (TType)

  private

  public
    constructor Create;
    function est (s: AnsiString): Boolean; override;
    function format (s: AnsiString): AnsiString; override;
  end;

  TMokaByte = class (TType)

  private

  public
    constructor Create;
    function est (s: AnsiString): Boolean; override;
    function format (s: AnsiString): AnsiString; override;
  end;

  TMokaUByte = class (TType)

  private

  public
    constructor Create;
    function est (s: AnsiString): Boolean; override;
    function format (s: AnsiString): AnsiString; override;
  end;

  TMokaShort = class (TType)

  private

  public
    constructor Create;
    function est (s: AnsiString): Boolean; override;
    function format (s: AnsiString): AnsiString; override;
  end;

  TMokaUShort = class (TType)

  private

  public
    constructor Create;
    function est (s: AnsiString): Boolean; override;
    function format (s: AnsiString): AnsiString; override;
  end;

  TMokaInt = class (TType)

  private

  public
    constructor Create;
    function est (s: AnsiString): Boolean; override;
    function format (s: AnsiString): AnsiString; override;
  end;

  TMokaUInt = class (TType)

  private

  public
    constructor Create;
    function est (s: AnsiString): Boolean; override;
    function format (s: AnsiString): AnsiString; override;
  end;

  TMokaLong = class (TType)

  private

  public
    constructor Create;
    function est (s: AnsiString): Boolean; override;
    function format (s: AnsiString): AnsiString; override;
  end;

  TMokaULong = class (TType)

  private

  public
    constructor Create;
    function est (s: AnsiString): Boolean; override;
    function format (s: AnsiString): AnsiString; override;
  end;

  TMokaFloat = class (TType)

  private

  public
    constructor Create;
    function est (s: AnsiString): Boolean; override;
    function format (s: AnsiString): AnsiString; override;
  end;

  TMokaDouble = class (TType)

  private

  public
    constructor Create;
    function est (s: AnsiString): Boolean; override;
    function format (s: AnsiString): AnsiString; override;
  end;

  TMokaString = class (TType)

  private

  public
    constructor Create;
    function est (s: AnsiString): Boolean; override;
    function format (s: AnsiString): AnsiString; override;
  end;

  TMokaANSIString = class (TType)

  private

  public
    constructor Create;
    function est (s: AnsiString): Boolean; override;
    function format (s: AnsiString): AnsiString; override;
  end;

  TMokaNull = class (TType)

  private

  public
    constructor Create;
    function est (s: AnsiString): Boolean; override;
    function format (s: AnsiString): AnsiString; override;
  end;

  TTyped = class

  private

  public
    name: AnsiString;
    rType: AnsiString;
    rPtr: AnsiString;
    comment: AnsiString;
    visibility: Longint;
    text: AnsiString;
    order: Longint;
    procedure writeString(s: AnsiString);
    procedure writeLine(s: AnsiString);
    procedure nextLine();
  end;

  TMethode = class (TTyped)

  private

  public
    stat: Boolean;
    fin: Boolean;
    abst: Boolean;
    nat: Boolean;
    used: Boolean;
    recur: Boolean;
    params: TParameterVector;
    locals: TStringVector;
    words: TStringVector;
    outp: TStringVector;
    constructor Create(comment: AnsiString); overload;
    constructor Create(rType: AnsiString; rPtr: AnsiString; name: AnsiString; stat: Boolean; abst: Boolean; nat: Boolean; fin: Boolean; comment: AnsiString; visibility: Longint; text: AnsiString); overload;
    function sameAs(m: TMethode): Boolean;
    function copy(): TMethode;
    function listArgs(addNames: Boolean; cl: TClasse): AnsiString;
    function listArgsConst(addTypes: Boolean; addNames: Boolean): AnsiString;
    function getMorph(): AnsiString;
    function getSign(cl: TClasse): AnsiString;
  end;

  TAttribute = class (TTyped)

  private

  public
    stat: Boolean;
    value: TStringVector;
    outp: TStringVector;
    used: Boolean;
    fin: Boolean;
    vol: Boolean;
    recur: Boolean;
    constructor Create(comment: AnsiString); overload;
    constructor Create(rType: AnsiString; rPtr: AnsiString; name: AnsiString; stat: Boolean; fin: Boolean; vol: Boolean; comment: AnsiString; visibility: Longint); overload;
    function copy(): TAttribute;
  end;

  TParameter = class (TTyped)

  private

  public
    constructor Create(rType: AnsiString; rPtr: AnsiString; name: AnsiString);
  end;

  TPackageVector = class (TVector)

  private

  public
    function contains(str: AnsiString): Boolean;
    function returns(str: AnsiString): TPack;
    function getPack(pos: Longint): TPack;
  end;

  TMethodVector = class (TVector)

  private

  public
    function getMeth(pos: Longint): TMethode;
    function contains(str: AnsiString): Boolean; overload;
    function contains(m: TMethode): Boolean; overload;
    function return(str: AnsiString): TMethode;
    function getMVec(str: AnsiString): TMethodVector;
  end;

  TTypeVector = class (TVector)

  private

  public
    function getType(pos: Longint): TType;
  end;

  TAttributeVector = class (TVector)

  private

  public
    function getAttr(pos: Longint): TAttribute;
    function contains(str: AnsiString): Boolean;
    function return(s: AnsiString): TAttribute;
  end;

  TClassVector = class (TVector)

  private

  public
    function getClass(pos: Longint): TClasse;
    function contains(str: AnsiString): Boolean;
    function return(str: AnsiString): TClasse;
  end;

  TParameterVector = class (TVector)

  private

  public
    function getParam(pos: Longint): TParameter;
    function copy(): TParameterVector;
    function contains(str: AnsiString): Boolean;
    function return(str: AnsiString): TParameter;
    function getMorph: AnsiString;
  end;

implementation

uses Funcs, Globals;

procedure TTyped.writeString(s: AnsiString);
begin
  text := text + s;
end;

procedure TTyped.writeLine(s: AnsiString);
begin
  text := text + s + newLine;
end;

procedure TTyped.nextLine();
begin
  text := text + newLine;
end;

function TMethode.getMorph: AnsiString;
var
  str: AnsiString;
  i: Longint;
  n: Longint;
  p: TParameter;
begin
  str := '_';

  for i := 0 to params.size - 1 do
  begin
    if str <> '_' then
    begin
      str := str + '_';
    end;

    p := params.getParam(i);

    if isClass(p.rType) then
    begin
      str := str + p.rType;
    end
    else
    begin
      str := str + replace(convertType(p.rType), ' ', '_');
    end;

    for n := 1 to count('*', p.rPtr) do
    begin
      str := str + '_p';
    end;

  end;

  getMorph := str;
end;

function TMethode.getSign(cl: TClasse): AnsiString;
var
  str: AnsiString;
begin
  str := '';

  if stat or cl.inter then
  begin
    str := cl.name + '_' + self.name + self.getMorph;
  end
  else
  begin
    str := self.name + self.getMorph;
  end;

  (*if stat then
  begin
    str := cl.name + '_';
  end;

  if rType <> '' then
  begin
    str := str + name;
  end
  else
  begin
    str := name + '_';
  end;

  str := str + '_';

  for i := 0 to params.size - 1 do
  begin
    if str <> '_' then
    begin
      str := str + '_';
    end;

    p := params.getParam(i);

    if isClass(p.rType) then
    begin
      str := str + p.rType;
    end
    else
    begin
      str := str + replace(convertType(p.rType), ' ', '_');
    end;

    for n := 1 to count('*', p.rPtr) do
    begin
      str := str + '_p';
    end;

  end; *)

  getSign := str;
end;

function TMethode.listArgsConst(addTypes: Boolean; addNames: Boolean): AnsiString;
var
  str: AnsiString;
  i: Longint;
  p: TParameter;
begin
  str := '';

  for i := 0 to params.size - 1 do
  begin
    p := params.getParam(i);
    if str <> '' then
    begin
      str := str + ', ';
    end;

  if addTypes then
  begin
      if isClass(p.rType) then
      begin
        str := str + 'struct ' + p.rType + ' * ';
      end
      else
      begin
        str := str + convertType(p.rType) + ' ';
      end;

      str := str + p.rPtr;
    end;

    if addNames then
    begin
      str := str + p.name + ' ';
    end;
  end;

  listArgsConst := str; 
end;

function TMethode.listArgs(addNames: Boolean; cl: TClasse): AnsiString;
var
  str: AnsiString;
  i: Longint;
  p: TParameter;
begin
  str := '';

  if not stat then
  begin
    if addNames then
    begin
      if (cl.inter) then
      begin
        str := 'struct Object * this ';
      end
      else
      begin
        str := 'struct ' + cl.name + ' * this ';
      end;
    end
    else
    begin
      if (cl.inter) then
      begin
        str := 'struct Object * ';
      end
      else
      begin
        str := 'struct ' + cl.name + ' * ';
      end;
    end;
  end;

  for i := 0 to params.size - 1 do
  begin
    p := params.getParam(i);
    if str <> '' then
    begin
      str := str + ', ';
    end;
    if isClass(p.rType) then
    begin
        str := str + 'struct ' + p.rType + ' * ';
    end
    else
    begin
      str := str + convertType(p.rType) + ' ';
    end;

    str := str + p.rPtr;

    if addNames then
    begin
      str := str + p.name + ' ';
    end;
  end;

  listArgs := str; 
end;

function TType.est(s: AnsiString): Boolean;
begin
  est := true;
end;

function TType.format(s: AnsiString): AnsiString;
begin
  format := s;
end;

constructor TType.Create(strMoka: AnsiString; strC: AnsiString);
begin
  self.strMoka := strMoka;
  self.strC    := strC;
end;

constructor TMokaBoolean.Create;
begin
  strMoka := 'boolean';
  strC := 'BOOL';
end;

function TMokaBoolean.est(s: AnsiString): Boolean;
begin
  if (s = 'true') or (s = 'false') then
  begin
    est := true;
  end
  else
  begin
    est := false;
  end;
end;

function TMokaBoolean.format(s: AnsiString): AnsiString;
begin
  if (s = 'true') then
  begin
    format := 'TRUE';
  end
  else
  begin
    format := 'FALSE';
  end;
end;

constructor TMokaChar.Create;
begin
  strMoka := 'char';
  strC := 'char';
end;

function TMokaChar.est(s: AnsiString): Boolean;
var
  i: Longint;
  len: Longint;
begin
  len := Length(s);

  if (len < 2) then
  begin
    est := false;
    exit;
  end;

  for i := 1 to len do
  begin
    if ((i = 1) or (i = len)) and (s[i] <> '''') then
    begin
      est := false;
      exit;
    end;
  end;

  est := true;
end;

function TMokaChar.format(s: AnsiString): AnsiString;
begin
  format := s;
end;

constructor TMokaByte.Create;
begin
  strMoka := 'byte';
  strC := 'byte';
end;

function TMokaByte.est(s: AnsiString): Boolean;
begin
  if isDigit(s[1]) and ((s[Length(s)] =  'b') or (s[Length(s)] = 'B')) then
  begin
    est := true;
  end
  else
  begin
    est := false;
  end;
end;

function TMokaByte.format(s: AnsiString): AnsiString;
var
  str: AnsiString;
  i: Longint;
  size: Longint;
begin
  size := Length(s) - 1;
  str := '';

  for i := 1 to size do
  begin
    str := str + s[i];
  end;

  format := str;
end;

constructor TMokaUByte.Create;
begin
  strMoka := 'ubyte';
  strC := 'unsigned byte';
end;

function TMokaUByte.est(s: AnsiString): Boolean;
begin
  if (Length(s) >= 3) and isDigit(s[1]) and (((s[Length(s)-1] = 'u') and (s[Length(s)] = 'b')) or ((s[Length(s)-1] = 'U') and (s[Length(s)] = 'B'))) then
  begin
    est := true;
  end
  else
  begin
    est := false;
  end;
end;

function TMokaUByte.format(s: AnsiString): AnsiString;
var
  str: AnsiString;
  i: Longint;
  size: Longint;
begin
  size := Length(s) - 2;
  str := '';

  for i := 1 to size do
  begin
    str := str + s[i];
  end;

  format := str;
end;

constructor TMokaShort.Create;
begin
  strMoka := 'short';
  strC := 'short int';
end;

function TMokaShort.est(s: AnsiString): Boolean;
begin
  if isDigit(s[1]) and ((s[Length(s)] =  's') or (s[Length(s)] = 'S')) then
  begin
    est := true;
  end
  else
  begin
    est := false;
  end;
end;

function TMokaShort.format(s: AnsiString): AnsiString;
var
  str: AnsiString;
  i: Longint;
  size: Longint;
begin
  size := Length(s) - 1;
  str := '';

  for i := 1 to size do
  begin
    str := str + s[i];
  end;

  format := str;
end;

constructor TMokaUShort.Create;
begin
  strMoka := 'ushort';
  strC := 'unsigned short int';
end;

function TMokaUShort.est(s: AnsiString): Boolean;
begin
  if (Length(s) >= 3) and isDigit(s[1]) and (((s[Length(s)-1] = 'u') and (s[Length(s)] = 's')) or ((s[Length(s)-1] = 'U') and (s[Length(s)] = 'S'))) then
  begin
    est := true;
  end
  else
  begin
    est := false;
  end;
end;

function TMokaUShort.format(s: AnsiString): AnsiString;
var
  str: AnsiString;
  i: Longint;
  size: Longint;
begin
  size := Length(s) - 2;
  str := '';

  for i := 1 to size do
  begin
    str := str + s[i];
  end;

  format := str;
end;

constructor TMokaInt.Create;
begin
  strMoka := 'int';
  strC := 'long int';
end;

function TMokaInt.est(s: AnsiString): Boolean;
var
  i: Longint;
begin
  for i := 1 to Length(s) do
  begin
    if not isDigit(s[i]) then
    begin
      est := false;
      exit;
    end;
  end;

  est := true;
end;

function TMokaInt.format(s: AnsiString): AnsiString;
begin
  format := s;
end;

constructor TMokaUInt.Create;
begin
  strMoka := 'uint';
  strC := 'unsigned long int';
end;

function TMokaUInt.est(s: AnsiString): Boolean;
begin
  if isDigit(s[1]) and ((s[Length(s)] =  'u') or (s[Length(s)] = 'U')) then
  begin
    est := true;
  end
  else
  begin
    est := false;
  end;
end;

function TMokaUInt.format(s: AnsiString): AnsiString;
var
  str: AnsiString;
  i: Longint;
  size: Longint;
begin
  size := Length(s) - 1;
  str := '';

  for i := 1 to size do
  begin
    str := str + s[i];
  end;

  format := str;
end;

constructor TMokaLong.Create;
begin
  strMoka := 'long';
  strC := 'long long int';
end;

function TMokaLong.est(s: AnsiString): Boolean;
begin
  if isDigit(s[1]) and ((s[Length(s)] =  'l') or (s[Length(s)] = 'L')) then
  begin
    est := true;
  end
  else
  begin
    est := false;
  end;
end;

function TMokaLong.format(s: AnsiString): AnsiString;
var
  str: AnsiString;
  i: Longint;
  size: Longint;
begin
  size := Length(s) - 1;
  str := '';

  for i := 1 to size do
  begin
    str := str + s[i];
  end;

  format := str;
end;

constructor TMokaULong.Create;
begin
  strMoka := 'ulong';
  strC := 'unsigned long long int';
end;

function TMokaULong.est(s: AnsiString): Boolean;
begin
  if (Length(s) >= 3) and isDigit(s[1]) and (((s[Length(s)-1] = 'u') and (s[Length(s)] = 's')) or ((s[Length(s)-1] = 'l') and (s[Length(s)] = 'L'))) then
  begin
    est := true;
  end
  else
  begin
    est := false;
  end;
end;

function TMokaULong.format(s: AnsiString): AnsiString;
var
  str: AnsiString;
  i: Longint;
  size: Longint;
begin
  size := Length(s) - 2;
  str := '';

  for i := 1 to size do
  begin
    str := str + s[i];
  end;

  format := str;
end;

constructor TMokaFloat.Create;
begin
  strMoka := 'float';
  strC := 'float';
end;

function TMokaFloat.est(s: AnsiString): Boolean;
begin
  if isDigit(s[1]) and ((s[Length(s)] =  'f') or (s[Length(s)] = 'F')) then
  begin
    est := true;
  end
  else
  begin
    est := false;
  end;
end;

function TMokaFloat.format(s: AnsiString): AnsiString;
var
  str: AnsiString;
  i: Longint;
  size: Longint;
begin
  size := Length(s) - 1;
  str := '';

  for i := 1 to size do
  begin
    str := str + s[i];
  end;

  format := str;
end;

constructor TMokaDouble.Create;
begin
  strMoka := 'double';
  strC := 'double';
end;

function TMokaDouble.est(s: AnsiString): Boolean;
begin
  if isDigit(s[1]) and (count('.', s) > 0) and not ((s[Length(s)] =  'f') or (s[Length(s)] = 'F')) then
  begin
    est := true;
  end
  else
  begin
    est := false;
  end;
end;

function TMokaDouble.format(s: AnsiString): AnsiString;
begin
  format := s;
end;

constructor TMokaString.Create;
begin
  strMoka := '';
  strC := 'String';
end;

function TMokaString.est(s: AnsiString): Boolean;
var
  i: Longint;
  len: Longint;
begin
  len := Length(s);

  if (len < 2) then
  begin
    est := false;
    exit;
  end;

  for i := 1 to len do
  begin
    if ((i = 1) or (i = len)) and (s[i] <> '"') then
    begin
      est := false;
      exit;
    end;
  end;

  est := true;
end;

function TMokaString.format(s: AnsiString): AnsiString;
begin
  format := 'TString_char_p ( ( char * ) ' + s + ' )';
end;

constructor TMokaANSIString.Create;
begin
  strMoka := '';
  strC := 'char *';
end;

function TMokaANSIString.est(s: AnsiString): Boolean;
var
  i: Longint;
  len: Longint;
begin
  len := Length(s);

  if (len < 3) then
  begin
    est := false;
    exit;
  end;

  for i := 1 to len do
  begin
    if ((i = 1) or (i = len-1)) and (s[i] <> '"') then
    begin
      est := false;
      exit;
    end;
    if ((i = len)) and (s[i] <> 'a') and (s[i] <> 'A') then
    begin
      est := false;
      exit;
    end;
  end;

  est := true;
end;

function TMokaANSIString.format(s: AnsiString): AnsiString;
var
  str: AnsiString;
  i: Longint;
  size: Longint;
begin
  size := Length(s) - 1;
  str := '';

  for i := 1 to size do
  begin
    str := str + s[i];
  end;

  format := str;
end;

constructor TMokaNull.Create;
begin
  strMoka := '';
  strC := '';
end;

function TMokaNull.est(s: AnsiString): Boolean;
begin
  est := (s = 'null');
end;

function TMokaNull.format(s: AnsiString): AnsiString;
begin
  format := 'NULL';
end;

constructor TFileLinked.Create;
begin
  outText := '';
end;

procedure TFileLinked.writeString(s: AnsiString);
begin
  outText := outText + s;
end;

procedure TFileLinked.writeLine(s: AnsiString);
begin
  outText := outText + s + newLine;
end;

procedure TFileLinked.nextLine();
begin
  outText := outText + newLine;
end;

constructor TPack.Create(name: AnsiString);
begin
  self.name := name;
  classes   := TClassVector.Create;
  words     := nil;
  fileText  := '';
  outText   := '';
end;

constructor TParameter.Create(rType: AnsiString; rPtr: AnsiString; name: AnsiString);
begin
  self.rType := rType;
  self.rPtr  := rPtr;
  self.name  := name;
  comment    := ''; 
end;

function TMethode.sameAs(m: TMethode): Boolean;
var
  i: Longint;
begin
  if (m.name <> name) then
  begin
    sameAs := false;
    exit;
  end;

  if (m.params.size <> params.size) then
  begin
    sameAs := false;
    exit;
  end;

  for i := 0 to params.size - 1 do
  begin
    if (params.getParam(i).rType <> m.params.getParam(i).rType) or (params.getParam(i).rPtr <> m.params.getParam(i).rPtr) then
    begin
      sameAs := false;
      exit;
    end;
  end;
  
  sameAs := true;
end;

function TAttributeVector.return(s: AnsiString): TAttribute;
var
  i: Longint;
  a: TAttribute;
begin
  for i := 0 to size - 1 do
  begin
    a := getAttr(i);
    if (a.name = s) then
    begin
      return := a;
      exit;
    end;
  end;

  return := nil;
end;

function TAttribute.copy(): TAttribute;
var
  a: TAttribute;
begin
  a := TAttribute.Create(rType, rPtr, name, stat, fin, vol, comment, visibility);

  a.value := value;
  a.used  := used;

  copy := a;
end;

constructor TAttribute.Create(comment: AnsiString);
begin
  self.Create('', '', '', false, false, false, comment, vDefault);
end;

constructor TAttribute.Create(rType: AnsiString; rPtr: AnsiString; name: AnsiString; stat: Boolean; fin: Boolean; vol: Boolean; comment: AnsiString; visibility: Longint);
begin
  self.rType := rType;
  self.rPtr  := rPtr;
  self.name  := name;
  self.stat  := stat;
  self.fin   := fin;
  self.vol   := vol;
  used       := true;
  value      := TStringVector.Create;
  self.comment    := comment;
  self.visibility := visibility;
  outp := TStringVector.Create;
  text := '';
end;

function TMethode.copy(): TMethode;
var
  m: TMethode;
begin
  m := TMethode.Create(rType, rPtr, name, stat, abst, nat, fin, comment, visibility, text);

  m.used := used;

  m.params := params;

  m.locals := locals;

  m.words   := words.copy;

  m.outp    := outp.copy;

  copy := m;
end;

constructor TMethode.Create(comment: AnsiString);
begin
  self.Create('', '', '', false, false, false, false, comment, vDefault, '');
end;

constructor TMethode.Create(rType: AnsiString; rPtr: AnsiString; name: AnsiString; stat: Boolean; abst: Boolean; nat: Boolean; fin: Boolean; comment: AnsiString; visibility: Longint; text: AnsiString);
begin
  self.rType := rType;
  self.rPtr  := rPtr;
  self.name  := name;
  self.stat  := stat;
  self.abst  := abst;
  used       := true;
  params     := TParameterVector.Create;
  locals     := TStringVector.Create;
  words      := TStringVector.Create(50, 100);
  self.comment    := comment;
  self.fin        := fin;
  self.visibility := visibility;
  self.text := text;
  outp       := TStringVector.Create(50, 100);
end;

function TPackageVector.contains(str: AnsiString): Boolean;
var
  size: Longint;
  i: Longint;
begin
    size := self.size - 1;

    for i := 0 to size do
    begin
      if (str = getPack(i).name) then
      begin
        contains := true;
        exit;
      end;
    end;

    contains := false;
end;

function TPackageVector.returns(str: AnsiString): TPack;
var
  size: Longint;
  i: Longint;
begin
    size := self.size - 1;

    for i := 0 to size do
    begin
      if (str = getPack(i).name) then
      begin
        returns := getPack(i);
        exit;
      end;
    end;

    returns := nil;
end;

function TTypeVector.getType(pos: Longint): TType;
begin
    getType := TType(get(pos));
end;

function TPackageVector.getPack(pos: Longint): TPack;
begin
    getPack := TPack(get(pos));
end;

function TParameterVector.getParam(pos: Longint): TParameter;
begin
    getParam := TParameter(get(pos));
end;

function TParameterVector.getMorph: String;
var
  str: AnsiString;
  i: Longint;
  n: Longint;
  p: TParameter;
begin
  str := '_';

  for i := 0 to size - 1 do
  begin
    if str <> '_' then
    begin
      str := str + '_';
    end;

    p := getParam(i);

    if isClass(p.rType) then
    begin
      str := str + p.rType;
    end
    else
    begin
      str := str + replace(convertType(p.rType), ' ', '_');
    end;

    for n := 1 to count('*', p.rPtr) do
    begin
      str := str + '_p';
    end;

  end;

  getMorph := str;
end;

function TParameterVector.copy(): TParameterVector;
var
  vec: TParameterVector;
  i: Longint;
begin
  vec := TParameterVector.Create;

  for i := 0 to size - 1 do
  begin
    vec.add(getParam(i));
  end;

  copy := vec;
end;

function TParameterVector.contains(str: AnsiString): Boolean;
var
  size: Longint;
  i: Longint;
begin
    size := self.size - 1;

    for i := 0 to size do
    begin
      if (str = getParam(i).name) then
      begin
        contains := true;
        exit;
      end;
    end;

    contains := false;
end;

function TParameterVector.return(str: AnsiString): TParameter;
var
  i: Longint;
  p: TParameter;
begin
  for i := 0 to size - 1 do
  begin
    p := getParam(i);
    if (p.name = str) then
    begin
      return := p;
      exit;
    end;
  end;

  return := nil;
end;

function TClassVector.getClass(pos: Longint): TClasse;
begin
    getClass := TClasse(get(pos));
end;

function TClassVector.return(str: AnsiString): TClasse;
var
  size: Longint;
  i: Longint;
begin
    size := self.size - 1;

    for i := 0 to size do
    begin
      if (str = getClass(i).name) then
      begin
        return := getClass(i);
        exit;
      end;
    end;

    return := nil;
end;

function TClassVector.contains(str: AnsiString): Boolean;
var
  size: Longint;
  i: Longint;
begin
    size := self.size - 1;

    for i := 0 to size do
    begin
      if (str = getClass(i).name) then
      begin
        contains := true;
        exit;
      end;
    end;

    contains := false;
end;

function TAttributeVector.getAttr(pos: Longint): TAttribute;
begin
    getAttr := TAttribute(get(pos));
end;

function TAttributeVector.contains(str: AnsiString): Boolean;
var
  size: Longint;
  i: Longint;
begin
    size := self.size - 1;

    for i := 0 to size do
    begin
      if (str = getAttr(i).name) then
      begin
        contains := true;
        exit;
      end;
    end;

    contains := false;
end;

function TMethodVector.getMeth(pos: Longint): TMethode;
begin
    getMeth := TMethode(get(pos));
end;

function TMethodVector.contains(m: TMethode): Boolean;
var
  size: Longint;
  i: Longint;
begin
    size := self.size - 1;

    for i := 0 to size do
    begin
      if (m.sameAs(getMeth(i))) then
      begin
        contains := true;
        exit;
      end;
    end;

    contains := false;
end;

function TMethodVector.contains(str: AnsiString): Boolean;
var
  size: Longint;
  i: Longint;
begin
    size := self.size - 1;

    for i := 0 to size do
    begin
      if (str = getMeth(i).name) then
      begin
        contains := true;
        exit;
      end;
    end;

    contains := false;
end;

function TMethodVector.return(str: AnsiString): TMethode;
var
  size: Longint;
  i: Longint;
begin
    size := self.size - 1;

    for i := 0 to size do
    begin
      if (str = getMeth(i).name) then
      begin
        return := getMeth(i);
        exit;
      end;
    end;

    return := nil;
end;

function TMethodVector.getMVec(str: AnsiString): TMethodVector;
var
  mv: TMethodVector;
  size: Longint;
  i: Longint;
begin
    mv := TMethodVector.Create;
    size := self.size - 1;

    for i := 0 to size do
    begin
      if (str = getMeth(i).name) then
      begin
        mv.add(getMeth(i));
      end;
    end;

    getMVec := mv;
end;

constructor TClasse.Create(name: AnsiString);
begin
  self.name  := name;
  self.sup   := nil;
  words      := TStringVector.Create(50, 100);
  inclu      := TStringVector.Create;
  fileText   := '';
  outText    := '';
  supName    := '';
  pack       := nil;
  visibility := vDefault;
  fin        := false;
  used       := true;
  inter      := false;
  abst       := false;
  ifsName    := TStringVector.Create;
  fileName   := '';
  ifs        := TClassVector.Create;
  meth       := TMethodVector.Create;
  attr       := TAttributeVector.Create;
end;

procedure TClasse.setUsed (b: Boolean);
begin
  used := b;
  if used and (sup <> nil) then
  begin
    sup.setUsed(true);
  end;
end;

procedure TClasse.setInst (b: Boolean);
begin
  inst := b;
  if inst and (sup <> nil) then
  begin
    sup.setInst(true);
  end;
end;

function TClasse.hasValuedStat(): Boolean;
var
  i: Longint;
  size: Longint;
begin
  if (attr.size < 1) then
  begin
    hasValuedStat := false;
    exit;
  end;

  size := attr.size - 1;

  for i := 0 to size do
  begin
    if attr.getAttr(i).stat and (attr.getAttr(i).value.size > 0) then
    begin
      hasValuedStat := true;
      exit;
    end;
  end;

  hasValuedStat := false;
end;

function TClasse.hasValuedAttr(): Boolean;
var
  i: Longint;
  size: Longint;
begin
  if (attr.size < 1) then
  begin
    hasValuedAttr := false;
    exit;
  end;

  size := attr.size - 1;

  for i := 0 to size do
  begin
    if not attr.getAttr(i).stat and (attr.getAttr(i).value.size > 0) then
    begin
      hasValuedAttr := true;
      exit;
    end;
  end;

  hasValuedAttr := false;
end;

function TClasse.hasStat(): Boolean;
var
  i: Longint;
  size: Longint;
begin
  if (attr.size < 1) then
  begin
    hasStat := false;
    exit;
  end;

  size := attr.size - 1;

  for i := 0 to size do
  begin
    if attr.getAttr(i).stat then
    begin
      hasStat := true;
      exit;
    end;
  end;

  hasStat := false;
end;

function TClasse.ext(sup: AnsiString): Boolean;
var
  cl: TClasse;
begin
  cl := self;

  while (cl <> nil) do
  begin
    if (cl.name = sup) then
    begin
      ext := true;
      exit;
    end;
    cl := cl.sup;
  end;

  ext := false;
end;

function TClasse.getMVec(str: AnsiString): TMethodVector;
var
  mv: TMethodVector;
  size: Longint;
  i: Longint;
  cl: TClasse;
begin
    mv := TMethodVector.Create;

    cl := self;


    while (cl <> nil) do
    begin
      size := cl.meth.size - 1;
      for i := 0 to size do
      begin
        if (str = cl.meth.getMeth(i).name) then
        begin
          mv.add(cl.meth.getMeth(i));
        end;
      end;
      cl := cl.sup;
    end;

    getMVec := mv;
end;

function TClasse.implAll(inte: AnsiString): Boolean;
var
  i: Longint;
  size: Longint;
  cl: TClasse;
begin
  cl := self;

  while (cl <> nil) do
  begin
    size := cl.ifs.size - 1;

    for i := 0 to size do
    begin
      if (cl.ifs.getClass(i).name = inte) then
      begin
        implAll := true;
        exit;
      end;
    end;

    cl := cl.sup;
  end;

  implAll := false;
end;

function TClasse.impl(inte: AnsiString): Boolean;
var
  i: Longint;
  size: Longint;
begin
  size := ifs.size - 1;

  for i := 0 to size do
  begin
    if (ifs.getClass(i).name = inte) then
    begin
      impl := true;
      exit;
    end;
  end;

  impl := false;
end;

function TClasse.instof(classe: AnsiString): Boolean;
var
  cl: TClasse;
begin
  cl := self;

  while (cl <> nil) do
  begin
    if (cl.name = classe) then
    begin
      instof := true;
      exit;
    end;
    cl := cl.sup;
  end;

  instof := false;
end;

function TClasse.returnAll(mSign: AnsiString): TMethode;
var
  i: Longint;
  m: TMethode;
  cl: TClasse;
begin
  cl := self;
  while (cl <> nil) do
  begin
    for i := 0 to cl.meth.size - 1 do
    begin
      m := cl.meth.getMeth(i);
      if ((m.name + m.getMorph) = mSign) then
      begin
        returnAll := m;
        exit;
      end;
    end;
    cl := cl.sup;
  end;
  
  returnAll := nil;
end;

function TClasse.return(mSign: AnsiString): TMethode;
var
  i: Longint;
  m: TMethode;
begin
  for i := 0 to meth.size - 1 do
  begin
    m := meth.getMeth(i);
    if ((m.name + m.getMorph) = mSign) then
    begin
      return := m;
      exit;
    end;
  end;

  return := nil;
end;

function TClasse.defined(mName: AnsiString; params: TParameterVector): Boolean;
var
  m: TMethode;
  i: Longint;
  size: Longint;
  pSize: Longint;
  n: Longint;
begin
  size  := meth.size - 1;
  pSize := params.size - 1;

  for i := 0 to size do
  begin
    m := meth.getMeth(i);
    if (m.name = mName) then
    begin
      if (params.size <> m.params.size) then
      begin
        continue;
      end;
      if (params.size = 0) then
      begin
        defined := true;
        exit;
      end;
      for n := 0 to pSize do
      begin
        if not ((params.getParam(n).rType = m.params.getParam(n).rType) and (params.getParam(n).rPtr = m.params.getParam(n).rPtr)) then
        begin
          break;
        end;
        if (n = pSize) then
        begin
          defined := true;
          exit;
        end;
      end;
    end;
  end;

  defined := false;
end;

end.
