(*
 * QUSOFT MICROSYSTMES
 * Moka
 * Copyright 2003 Frdric Brown
 *)

(*
 *  This file is part of Moka.
 *
 *  Moka is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.

 *  Moka is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.

 *  You should have received a copy of the GNU General Public License
 *  along with Moka; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *)

(*
 * Decoder unit contains the decode function called to decode the java source code.
 * The java source code is translated into many objects defined by the classes of
 * the Reflect unit.
 *)

unit Decoder;

interface

uses Util, Reflect;

procedure decode(mainFileName: AnsiString);
procedure importDir(dirStr: AnsiString; str: AnsiString; vecImp: TStringVector);
function packToPath(strPack: AnsiString): AnsiString;
function classToPath(strClass: AnsiString): AnsiString;
function addComments(cClass: TClasse; indCl: Longint): Longint; overload;
procedure decodeCurrentClass;
procedure addInterface(cClass: TClasse; iClass: TClasse);

implementation

uses SysUtils, Funcs, Globals, OutputStream, Tokenizer, Converter;

procedure decode(mainFileName: AnsiString);
var
  dPack: TPack;
  vecImp: TStringVector;
  inclu: AnsiString;
  i: Longint;
  fileName: AnsiString;
  sClass: TClasse;
  iInt: TClasse;
  cPack: TPack;
  n: Longint;
  indCl: Longint;
  indImp: Longint;
  str: AnsiString;
  dirStr: AnsiString;
  time0: TTimeStamp;
  time1: TTimeStamp;
  m: TMethode;
  a: TAttribute;
  pv: TParameterVector;
  buf: TStringVector;
begin

  cClass := nil;
  cMeth  := nil;

  vecCl  := TClassVector.Create;
  vecPk  := TPackageVector.Create;

  vecImp := TStringVector.Create;

  vecImp.add(mainFileName);

  dPack := TPack.Create('default');

  vecPk.add(dPack);

  time0 := DateTimeToTimeStamp(Time());
  Write('Reading source files : ');

  os.println('');

  indImp := 0;
  while (indImp < vecImp.size) do
  begin
    fileName := vecImp.getStr(indImp);

    {if vecDone.contains(extractClass(fileName)) then
    begin
      indImp := indImp + 1;
      continue;
    end;}

    os.println('Reading: ' + fileName);

    if not FileExists(fileName) then
    begin
      pushError(1, 'Error: The file specified doesn''t exists (' + fileName + ').');
    end;

    cClass := TClasse.Create(extractClass(fileName));


    if FileExists(dirProj + pathSeparator + extractClass(fileName) + '.h') then
    begin
      //cClass.inclu := '#include "' + extractClass(fileName) + '.h"';
      cClass.inclu.add(extractDirectory(fileName) + pathSeparator + extractClass(fileName) + '.h');
      os.println('Included ' + extractClass(fileName) + '.h');
    end
    else if FileExists(extractDirectory(fileName) + pathSeparator + extractClass(fileName) + '.h') then
    begin
      //cClass.inclu := '#include <' + extractDirectory(fileName) + pathSeparator + extractClass(fileName) + '.h>';
      cClass.inclu.add(extractDirectory(fileName) + pathSeparator + extractClass(fileName) + '.h');
      os.println('Included ' + extractDirectory(fileName) + pathSeparator + extractClass(fileName) + '.h');
      //cClass.inclu := '#include "' + extractClass(fileName) + '.h"';
    end;

    if FileExists(dirProj + pathSeparator + extractClass(fileName) + '.c') then
    begin
      cClass.inclu.add(extractDirectory(fileName) + pathSeparator + extractClass(fileName) + '.c');
      os.println('Included ' + extractClass(fileName) + '.c');
    end
    else if FileExists(extractDirectory(fileName) + pathSeparator + extractClass(fileName) + '.c') then
    begin
      cClass.inclu.add(extractDirectory(fileName) + pathSeparator + extractClass(fileName) + '.c');
      os.println('Included ' + extractDirectory(fileName) + pathSeparator + extractClass(fileName) + '.c');
    end;

    if (mainCl = nil) then
    begin
      mainCl := cClass;
    end;

    cClass.fileName := fileName;

    vecCl.add(cClass);

    cClass.fileText := readFileAsString(fileName);

    cClass.words := getStringVector(cClass.fileText);

    os.println(#9 + 'Class      : ' + cClass.name);

    indCl := 0;
    indCl := addComments(cClass, indCl);

    if (cClass.words.getStr(indCl) = 'package') then
    begin
      str := '';
      indCl := indCl + 1;
      while (indCl < cClass.words.size) and (cClass.words.getStr(indCl) <> ';') do
      begin
        str := str + cClass.words.getStr(indCl);
        indCl := indCl + 1;
      end;

      if vecPk.contains(str) then
      begin
        cPack := vecPk.returns(str);
      end
      else
      begin
        cPack := TPack.Create(str);
        vecPk.add(cPack);
      end;
      indCl := indCl + 1;
    end
    else
    begin
      cPack := dPack;
      str   := 'default';
    end;

    cPack.classes.add(cClass);
    cClass.pack := cPack;

    os.println(#9 + 'Package    : ' + str);

    if (cClass.name = extractClass(mainFileName)) then
    begin
      if (cPack.name = 'default') then
      begin
        dirStr := dirProj;
      end
      else
      begin
        dirStr := dirProj + pathSeparator + packToPath(cPack.name);
        if not DirectoryExists(dirStr) then
        begin
          dirStr := apiDir + pathSeparator;
          dirStr := dirStr + packToPath(cPack.name);
        end;
      end;
      importDir(dirStr, str, vecImp);
    end;

    indCl := addComments(cClass, indCl);

    while (cClass.words.getStr(indCl) = 'import') do
    begin
      str := '';
      indCl := indCl + 1;
      while (indCl < cClass.words.size) and (cClass.words.getStr(indCl) <> ';') and ((indCl + 1 < cClass.words.size) and (cClass.words.getStr(indCl+1) <> '*')) do
      begin
        str := str + cClass.words.getStr(indCl);
        indCl := indCl + 1;
      end;
      indCl := indCl + 1;

      if (indCl < cClass.words.size) and (cClass.words.getStr(indCl) = '*') then
      begin
        dirStr := dirProj + pathSeparator + packToPath(str);
        if not DirectoryExists(dirStr) then
        begin
          dirStr := apiDir + pathSeparator;
          dirStr := dirStr + packToPath(str);
        end;
        importDir(dirStr, str, vecImp);
        indCl := indCl + 2;
      end
      else
      begin
        dirStr := dirProj + pathSeparator + classToPath(str);
        if not FileExists(dirStr) then
        begin
        dirStr := apiDir + pathSeparator;
        dirStr := dirStr + classToPath(str);
        end;
        if not (vecImp.contains(dirStr)) then
        begin
          os.println(#9 + 'Importing class ' + str);
          if not FileExists(dirStr) then
          begin
            pushError(2, 'Error: Unable to locate the ' + str + ' class (' + dirStr + ').');
          end;
          vecImp.add(dirStr);
        end;
      end;
      indCl := addComments(cClass, indCl);
    end;

    if not (est(cClass.words.getStr(indCl), vecCDecla)) then
    begin
      pushError(1, 'Error: class modifier, ''public'', ''class'' or ''interface'' expected in the declaration of class ' + cClass.name + ' but ''' + cClass.words.getStr(indCl) + ''' found (' + fileName + ').');
    end;

    os.print(#9 + 'Identity   : ');

    while (indCl < cClass.words.size) and est(cClass.words.getStr(indCl), vecCMod) do
    begin
      os.print(cClass.words.getStr(indCl) + ' ');
      if (cClass.words.getStr(indCl) = 'public') then
      begin
        cClass.visibility := vPublic;
      end
      else if (cClass.words.getStr(indCl) = 'abstract') then
      begin
        cClass.abst := true;
      end
      else if (cClass.words.getStr(indCl) = 'final') then
      begin
        cClass.fin := true;
      end;
      indCl := indCl + 1;
    end;

    if (cClass.words.getStr(indCl) = 'interface') then
    begin
      os.print('interface ');
      cClass.inter := true;
    end
    else
    begin
      os.print('class ');
    end;

    indCl := indCl + 1;

    if (cClass.name <> cClass.words.getStr(indCl)) then
    begin
      pushError(1, 'Error: ' + cClass.words.getStr(indCl) + ' class must be defined in a file named ''' + cClass.words.getStr(indCl) + '.java'' (' + fileName + ').');
    end;

    indCl := indCl + 1;

    if (cClass.name <> 'Object') then
    begin
      os.print('extending ');
    end;

    if (cClass.words.getStr(indCl) = 'extends') then
    begin
      cClass.supName := cClass.words.getStr(indCl + 1);
      indCl := indCl + 2;
    end
    else
    begin
      if (cClass.name = 'Object') then
      begin
        cClass.supName := '';
      end
      else
      begin
        cClass.supName := 'Object';
      end;
    end;

    os.println(cClass.supName);

    if (cClass.words.getStr(indCl) = 'implements') then
    begin
      if cClass.inter then
      begin
        pushError(1, 'Error : ''{'' expected in the declaration of interface ' + cClass.name + ' but ''' + cClass.words.getStr(indCl) + ''' found (' + fileName + ').');
      end;

      indCl := indCl + 1;

      os.print(#9 + 'Interfaces : ');
      os.print(cClass.words.getStr(indCl));
      cClass.ifsName.add(cClass.words.getStr(indCl));

      while (indCl + 1 < cClass.words.size) and (cClass.words.getStr(indCl + 1) = ',') do
      begin
        indCl := indCl + 2;
        os.print(', ' + cClass.words.getStr(indCl));
        cClass.ifsName.add(cClass.words.getStr(indCl));
      end;

      os.println('');

      indCl := indCl + 1;
    end;

    if (cClass.words.getStr(indCl) <> '{') then
    begin
      pushError(1, 'Error : ''extends'', ''implements'' or ''{'' expected in the declaration of class ' + cClass.name + ' but ''' + cClass.words.getStr(indCl) + ''' found (' + fileName + ').');
    end;

    indImp := indImp + 1;
  end;

  if not vecCl.contains('Object') then
  begin
    pushError(1, 'Error : Class Object is missing.');
  end;

  for i := 0 to vecCl.size - 1 do
  begin
    cClass := vecCl.getClass(i);
    if (cClass.name = 'Object') then
    begin
      continue;
    end;

    if cClass.supName = '' then
    begin
      cClass.sup := nil;
      continue;
    end;
    sClass := vecCl.return(cClass.supName);
    if (sClass <> nil) then
    begin
      cClass.sup := sClass;
    end
    else
    begin
      pushError(1, 'Error : Superclass ' + cClass.supName + ' of class ' + cClass.name + ' is missing (' + cClass.fileName + ').');
    end;
    for n := 0 to cClass.ifsName.size - 1 do
    begin
      iInt := vecCl.return(cClass.ifsName.getStr(n));
      if (iInt <> nil) then
      begin
        cClass.ifs.add(iInt);
      end
      else
      begin
        pushError(1, 'Error : Interface ' + cClass.ifsName.getStr(n) + ' implemented by class ' + cClass.name + ' is missing (' + cClass.fileName + ').');
      end;
    end;
  end;

  time1 := DateTimeToTimeStamp(Time());
  Writeln('done in ' + intToStr(time1.Time - time0.Time) + 'ms.');

  time0 := DateTimeToTimeStamp(Time());
  Write('Decoding classes : ');

  for i := 0 to vecCl.size - 1 do
  begin
    cClass := vecCl.getClass(i);
    decodeCurrentClass();
  end;

  time1 := DateTimeToTimeStamp(Time());
  Writeln('done in ' + intToStr(time1.Time - time0.Time) + 'ms.');

  time0 := DateTimeToTimeStamp(Time());
  Write('Converting source code : ');

  os.println('');

  for i := 0 to vecCl.size - 1 do
  begin
    cClass := vecCl.getClass(i);
    os.println('Interfacing class ' + cClass.name);
    for n := 0 to cClass.ifs.size - 1 do
    begin
      os.println(#9 + 'Interfacing ' + ' with interface ' + cClass.ifs.getClass(n).name);
      addInterface(cClass, cClass.ifs.getClass(n));
    end;
  end;

  for i := 0 to vecCl.size - 1 do
  begin
    cClass := vecCl.getClass(i);
    os.println('Convecting code of class ' + cClass.name);
    {for n := 0 to cClass.ifs.size - 1 do
    begin
      os.println(#9 + 'Interfacing ' + ' with interface ' + cClass.ifs.getClass(n).name);
      addInterface(cClass, cClass.ifs.getClass(n));
    end;}
    for n := 0 to cClass.attr.size - 1 do
    begin
      os.println(#9 + 'Converting value of attribute ' + cClass.attr.getAttr(n).name);
      if (cClass.attr.getAttr(n).value.size > 0) then
      begin
        cClass.attr.getAttr(n).outp := convertAttr(cClass, cClass.attr.getAttr(n));
        cClass.attr.getAttr(n).text := cClass.attr.getAttr(n).outp.toString;
      end
      else
      begin
        os.println(#9 + #9 + 'This attribute does not have an initial value');
      end;
    end;
    for n := 0 to cClass.meth.size - 1 do
    begin
      if not cClass.inter then
      begin
        os.println(#9 + 'Converting code of method ' + cClass.meth.getMeth(n).name);
        if (cClass.meth.getMeth(n).nat) then
        begin
          os.println(#9 + #9 + 'This method is native');
        end
        else
        begin
          cClass.meth.getMeth(n).outp := convertMeth(cClass, cClass.meth.getMeth(n));
          cClass.meth.getMeth(n).text := cClass.meth.getMeth(n).outp.toText();
        end;
      end;
      if (cClass.meth.getMeth(n).name = 'finalize') and mainCl.ifs.contains('Use_GarbageCollection') then
      begin
        buf := TStringVector.Create;
        (*
        buf.add('if');
        buf.add('(');
        buf.add('!');
        buf.add('strcmp');
        buf.add('(');
        buf.add('this');
        buf.add('->');
        buf.add('class');
        buf.add(',');
        buf.add('"');
        buf.add(cClass.name);
        buf.add('"');
        buf.add(')');
        buf.add(')');
        *)
        buf.add('GarbageCollector_rem_Object');
        buf.add('(');
        buf.add('this');
        buf.add(')');
        buf.add(';');

        cClass.meth.getMeth(n).text := #9 + buf.toString + newLine + cClass.meth.getMeth(n).text;
      end;
    end;
    m := TMethode.Create('/* Instanciates a new ' + cClass.name + '. */');
    m.name := cClass.name;
    if not cClass.inter and not cClass.meth.contains(m) and not (cClass.abst and cClass.fin) then
    begin
      os.println(#9 + 'Constructor is missing, creating a default one');

      cClass.meth.add(m);
      if cClass.sup <> nil then
      begin
        m.text := #9 + cClass.sup.name + '__ ( ( T' + cClass.sup.name + ' * ) this ) ; ';
      end;
    end;
    if cClass.ext('Serializable') then
    begin
      pv := TParameterVector.Create;
      pv.add(TParameter.Create('IOStream', '', 'ios'));
      if not cClass.defined('serialize', pv) then
      begin
        str := #9 + 'ios -> writeString_String ( ios , TString_char_p ( this -> class ) ) ; ' + newLine;

        sClass := cClass;

        while sClass <> nil do
        begin
          for n := 0 to sClass.attr.size - 1 do
          begin
            a := sClass.attr.getAttr(n);
            if isClass(a.rType) then
            begin
              if (a.rPtr = '') then
              begin
                if (a.rType = 'String') then
                begin
                  str := str + #9 + 'ios -> writeBoolean_BOOL ( ios , this -> ' + a.name + ' != NULL ) ; ' + newLine;
                  str := str + #9 + 'if ( this -> ' + a.name + ' ) { ' + newLine;
                  str := str + #9 + #9 + 'ios -> writeString_String ( ios, this -> ' + a.name + ' -> toString_ ( this -> ' + a.name + ' ) ) ; ' + newLine;
                  str := str + #9 + '} ' + newLine;
                end
                else if vecCl.return(a.rType).ext('Serializable') then
                begin
                  str := str + #9 + 'ios -> writeBoolean_BOOL ( ios , this -> ' + a.name + ' != NULL ) ; ' + newLine;
                  str := str + #9 + 'if ( this -> ' + a.name + ' ) { ' + newLine;
                  str := str + #9 + #9 + 'this -> ' + a.name + ' -> serialize_IOStream ( this -> ' + a.name + ' , ios ) ; ' + newLine;
                  str := str + #9 + '} ' + newLine;
                end;
              end
              else if (a.rPtr = '') then
              begin
                str := str + #9 + 'ios -> writeBoolean_BOOL ( ios , FALSE ) ; ' + newLine;
              end;
            end
            else
            begin
              str := str + #9 + 'ios -> writeBytes_char_p_short_int ( ios , ( char * ) & this -> ' + a.name + ' , sizeof ( ' + convertType(a.rType) + ' ) ) ; ' + newLine;
            end;
          end;
          sClass := sClass.sup;
        end;
        m := TMethode.Create('/* Used in the serialization process. */');
        m.rType := 'void';
        m.name := 'serialize';
        m.params := pv;
        m.nat := true;
        m.text := str;
        cClass.meth.add(m);
      end;
      if not cClass.defined('deserialize', pv) then
      begin
        str := #9 + 'T' + cClass.name + ' * obj = T' + cClass.name + '_ ( ) ; ' + newLine;

        sClass := cClass;

        while sClass <> nil do
        begin
          for n := 0 to sClass.attr.size - 1 do
          begin
            a := sClass.attr.getAttr(n);
            if isClass(a.rType) then
            begin
              if (a.rPtr = '') then
              begin
                if (a.rType = 'String') then
                begin
                  str := str + #9 + 'if ( ios -> readBoolean_ ( ios ) ) { ' + newLine;
                  str := str + #9 + #9 + 'obj -> ' + a.name + ' = ios -> readString_ ( ios ) ; ' + newLine;
                  str := str + #9 + '} ' + newLine;
                  str := str + #9 + 'else { ' + newLine;
                  str := str + #9 + #9 + 'obj -> ' + a.name + ' = NULL ; ' + newLine;
                  str := str + #9 + '} ' + newLine;
                end
                else if vecCl.return(a.rType).ext('Serializable') then
                begin
                  str := str + #9 + 'if ( ios -> readBoolean_ ( ios ) ) { ' + newLine;
                  str := str + #9 + #9 + 'obj -> ' + a.name + ' = ( T' + a.rType + ' * ) ios -> readObject_ ( ios ) ; ' + newLine;
                  str := str + #9 + '} ' + newLine;
                  str := str + #9 + 'else { ' + newLine;
                  str := str + #9 + #9 + 'obj -> ' + a.name + ' = NULL ; ' + newLine;
                  str := str + #9 + '} ' + newLine;
                end;
              end;
            end
            else if (a.rPtr = '') then
            begin
              str := str + #9 + 'ios -> readBytes_char_p_short_int ( ios , ( char * ) & obj -> ' + a.name + ', sizeof ( ' + convertType(a.rType) + ' ) ) ; ' + newLine;
            end;
          end;
          sClass := sClass.sup;
        end;
        str := str + #9 + 'return ( TObject * ) obj ; ';
        m := TMethode.Create('/* Used in the deserialization process. */');
        m.rType := 'Object';
        m.name := 'deserialize';
        m.params := pv;
        m.nat := true;
        m.text := str;
        cClass.meth.add(m);
      end;
    end;
  end;

  (*if mainCl.ifs.contains('Use_GarbageCollection') then
  begin
    cClass := vecCl.return('System');
    cMeth := cClass.return('free_void_p');
    if (cMeth <> nil) then
    begin
      cMeth.text := 'GarbageCollector_remPtr_void_p ( ptr ) ;' + newLine + cMeth.text;
    end;
    cMeth := cClass.return('malloc_long_int');
    if (cMeth <> nil) then
    begin
      cMeth.text := 'void * ptr = malloc ( size ) ;' + newLine + 'GarbageCollector_addPtr_void_p ( ptr ) ;' + newLine + 'return ptr ;';
    end;
  end; *)
  time1 := DateTimeToTimeStamp(Time());
  Writeln('done in ' + intToStr(time1.Time - time0.Time) + 'ms.');
end;

procedure addInterface(cClass: TClasse; iClass: TClasse);
var
  i: Longint;
  a: TAttribute;
  m: TMethode;
begin
  while iClass <> nil do
  begin
    for i := 0 to iClass.attr.size - 1 do
    begin
      a := iClass.attr.getAttr(i);
      if not a.stat then
      begin
        os.println(#9 + 'Adding attribute ' + a.name);
        if overloads(a, cClass)  then
        begin
          os.println(#9 + #9 + 'This attribute is already present in class ' + cClass.name);
        end
        else
        begin
          cClass.attr.add(a);
        end;
      end;
    end;

    for i := 0 to iClass.meth.size - 1 do
    begin
      m := iClass.meth.getMeth(i);
      if not m.stat then
      begin
        os.println(#9 + 'Adding method ' + m.name);
        if overloads(m, cClass)  then
        begin
          os.println(#9 + #9 + 'Class ' + cClass.name + ' overrides this method.');
        end
        else
        begin
          cClass.meth.add(m.copy);
        end;
      end;
    end;
    iClass := iClass.sup;
  end;
end;

procedure decodeCurrentClass;
var
  comment: AnsiString;
  indCl: Longint;
  order: Longint;
begin
  indCl := 0;
  cClass.words := getBlock(cClass.words, 0);

  os.println('Decoding ' + cClass.name + ' class (' + cClass.fileName + ')');

  order := 0;

  while (indCl < cClass.words.size) do
  begin
    comment := '';

    while (indCl < cClass.words.size) and isComment(cClass.words.getStr(indCl)) do
    begin
      if (comment <> '') then
      begin
        comment := newLine;
      end;
      comment := comment + (cClass.words.getStr(indCl));
      indCl := indCl + 1;
    end;

    if (indCl < cClass.words.size) then
    begin
      if est(cClass.words.getStr(indCl), vecStatic) and (cClass.words.getStr(indCl + 1) = '{') then
      begin
        addStatic(cClass, indCl, comment);
        indCl := nextInstIndex(cClass.words, indCl);
      end
      else if (nextIndex(cClass.words, '(', indCl) < 0) or ((nextIndex(cClass.words, '=', indCl) > 0) and (nextIndex(cClass.words, '(', indCl) > nextIndex(cClass.words, '=', indCl))) or ((nextIndex(cClass.words, ';', indCl) > 0) and (nextIndex(cClass.words, '(', indCl) > nextIndex(cClass.words, ';', indCl))) then
      begin
        addAttribute(cClass, indCl, comment, order);
        order := order + 1;
        indCl := nextInstIndex(cClass.words, indCl);
      end
      else
      begin
        addMethod(cClass, indCl, comment, order);
        order := order + 1;
        indCl := nextInstIndex(cClass.words, indCl);
      end;
    end;

  end;
end;

function addComments(cClass: TClasse; indCl: Longint): Longint;
begin
  while (indCl < cClass.words.size) and isComment(cClass.words.getStr(indCl)) do
  begin
    cClass.writeLine(cClass.words.getStr(indCl));
    indCl := indCl + 1;
  end;

  addComments := indCl;
end;

function classToPath(strClass: AnsiString): AnsiString;
var
  dirStr: AnsiString;
  n: Longint;
begin
  dirStr := '';
        for n := 1 to Length(strClass) do
        begin
          if (strClass[n] = '.') then
          begin
            dirStr := dirStr + pathSeparator;
          end
          else
          begin
            dirStr := dirStr + strClass[n];
          end;
        end;

  classToPath := dirStr + '.java';
end;

function packToPath(strPack: AnsiString): AnsiString;
var
  dirStr: AnsiString;
  n: Longint;
begin
  dirStr := '';
        for n := 1 to Length(strPack) do
        begin
          if (strPack[n] = '.') then
          begin
            dirStr := dirStr + pathSeparator;
          end
          else
          begin
            dirStr := dirStr + strPack[n];
          end;
        end;

  packToPath := dirStr;
end;

procedure importDir(dirStr: AnsiString; str: AnsiString; vecImp: TStringVector);
var
  n: Longint;
  vecStr: TStringVector;
begin
      os.println(#9 + 'Importing all classes of the ' + str + ' package ...');
      if not DirectoryExists(dirStr) then
      begin
        pushError(2, 'Error: Unable to locate the ' + str + ' package classes (' + dirStr + ').');
      end;
      vecStr := getFilesVector(dirStr, '*.java');
      for n := 0 to vecStr.size - 1 do
      begin
        str := dirStr + pathSeparator;
        str := str + vecStr.getStr(n);
        if not (vecImp.contains(str)) and not vecCl.contains(extractClass(str)) then
        begin
          os.println(#9 + #9 + str);
          vecImp.add(str);
        end;
      end;
end;

end.
