/******************************************************************************
*
* project name:  TICT-Explorer
* file name:     viewhex.c
* initial date:  07/01/2001
* author:        thomas.nussbaumer@gmx.net
*
* description:   Picture Viewer
*                String Variable Viewer
*                Hex-Editor (taken from TIGCC Tools Suite)
*
* Hex-Editor from TIGCC Tools Suite integrated by Cody Cutrer
* [programmer066@hotmail.com]
*
*
* $Id: viewers.c,v 1.9 2002/10/01 09:06:49 tnussb Exp $
*
******************************************************************************/

//=============================================================================
//=============================================================================
//
// integrated picture viewer
//
//=============================================================================
//=============================================================================
void ViewPIC(HANDLE handle) {
    unsigned char* src;
    unsigned short offx;
    unsigned short offy;

    if (!(src=SaveHeapLock(handle))) return;

    offy = (PIX_HEIGHT - ((BITMAP*)(src+2))->NumRows)/2;
    offx = (PIX_WIDTH - ((BITMAP*)(src+2))->NumCols)/2;

    memset(LCD_MEM,0,LCD_SIZE);
    BitmapPut(offx,offy,src+2,&fullscreen,A_REPLACE);
    GetUserInput();

    SaveHeapUnLock();
}


//=============================================================================
//=============================================================================
//
// integrated text viewer
//
//=============================================================================
//=============================================================================
#define TEXT_NR_LINES      C89_92(10,12)
#define TEXT_NR_CHARS      C89_92(20,30)
#define ASCII_PER_PAGE     (TEXT_NR_CHARS*TEXT_NR_LINES)

unsigned char* StartOfPage(unsigned short pagenr,
                           unsigned char* start,
                           unsigned char* end)
{
    unsigned short line_count = 0;
    unsigned char* tmp        = start;
    unsigned short char_count = 0;
    unsigned short page_count = 0;

    while (page_count != pagenr && tmp <= end) {
        if (*tmp == 0x0d) {
            line_count++;
            char_count=0;
            tmp++;
            if (tmp>end) break;
        }
        else {
            char_count++;
            if (char_count==TEXT_NR_CHARS) {
                line_count++;
                char_count=0;
            }
        }

        if (line_count == TEXT_NR_LINES) {
            page_count++;
            line_count=0;
        }
        tmp++;
    }

    if (tmp > end) return NULL;

    return tmp;
}



void ViewTEXT(SYM_ENTRY* symptr) {
    unsigned char* src;
    unsigned char* end;
    unsigned short size;
    unsigned short input;
    short          redraw = 1;
    unsigned short pages;
    long           active_page = 0;

    if (!(src=SaveHeapLock(symptr->handle))) return;

    size = *(unsigned short*)(src) + 2;

    src+=5;
    size-=7;

    end = src+size;

    // the minimum length of a page is TEXT_NR_LINES (1 character per line)
    for (pages = 0;pages<65520/TEXT_NR_LINES;pages++) {
        if (!StartOfPage(pages,src,end)) break;
    }

    FontSetSys(F_6x8);

    do {
        if (redraw) {
            unsigned char* act = StartOfPage((unsigned short)active_page,src,end);
            short x,y;
            unsigned char c;

            memset(LCD_MEM,0,LCD_SIZE);

            for (y=0;y<TEXT_NR_LINES;y++) {
                for (x=0;x<TEXT_NR_CHARS;x++) {
                    c = *act++;
                    if (c==0xd) {
                        act++;
                        break;
                    }
                    if (c) DrawChar(x*8,y*10,c,A_REPLACE);
                    if (act>=end) break;
                }
                if (act>=end) break;
            }
            redraw = 0;
        }

        input = GetUserInput();
        if (input == KEY_UP) {
            active_page--;
            if (active_page < 0) active_page = pages-1;
            redraw = 1;
        }
        else if (input == KEY_DOWN) {
            active_page++;
            if (active_page >= pages) active_page = 0;
            redraw = 1;
        }
        else if (input == KEY_UP2ND) {   // 2nd + UP
            active_page = 0;
            redraw = 1;
        }
        else if (input == KEY_DOWN2ND) { // 2nd + DOWN
            active_page = pages-1;
            redraw = 1;
        }
    }
    while (input != KEY_ESC);

    FontSetSys(F_4x6);

    SaveHeapUnLock();
    DrawStatusLine(NULL);
}


//=============================================================================
//=============================================================================
//
// integrated hexeditor (taken from TIGCC Tools Suite)
//
//=============================================================================
//=============================================================================
#define TTHEDIT_VERSION "1.10"

#define NR_ROWS    C89_92(15,20)
#define NR_COLUMNS C89_92(8,12)
#define NR_BYTES   (NR_ROWS * NR_COLUMNS)

/*===========================================================================*/
/* gets input key with additional features (APD timer/idle mode/repeat fix)  */
/*===========================================================================*/
static inline short GetInputKey(void) {
#if defined(USE_TI89)
    unsigned short input = toupper(GetUserInput());

    switch(input) {
        case '=': input = 'A'; break;
        case '(': input = 'B'; break;
        case ')': input = 'C'; break;
        case ',': input = 'D'; break;
        case '/': input = 'E'; break;
        case '|': input = 'F'; break;
        // map also alpha-keys to numbers (if alpha-lock is on)
        case 'V': input = '0';   break;
        case 'Q': input = '1';   break;
        case 'R': input = '2';   break;
        case 'S': input = '3';   break;
        case 'L': input = '4';   break;
        case 'M': input = '5';   break;
        case 'N': input = '6';   break;
        case 'G': input = '7';   break;
        case 'H': input = '8';   break;
        case 'I': input = '9';   break;
        default:  break;
    }
    return input;
#else
    return toupper(GetUserInput());
#endif
}



/*===========================================================================*/
/* using XOR and a simple sprite for cursor drawing                          */
/*===========================================================================*/
void XOR_Cursor(short x,short y) {
    long addr = ((long)LCD_MEM)+(y<<5)-(y<<1)+((x>>3)&0xfffe);
    long mask = (long)(0xF8)<<(24-(x&15));

    // unrolled loop for more speed ...
    *(long*)addr^=mask;addr+=30;*(long*)addr^=mask;addr+=30;
    *(long*)addr^=mask;addr+=30;*(long*)addr^=mask;addr+=30;
    *(long*)addr^=mask;addr+=30;*(long*)addr^=mask;addr+=30;
    *(long*)addr^=mask;
}


/*===========================================================================*/
/* draws info bar                                                            */
/*===========================================================================*/
void DrawInfoBar(short locked_flag, unsigned short varsize,short save_flag)
{
    const char* savestr;
    char        tempstr[100];

    if (locked_flag)    savestr = "Locked";
    else if (save_flag) savestr = "Yes";
    else                savestr = "No";

    FillFullLines(C89_92(100,128) - 7,7,0x0);
    sprintf(tempstr, "Size: %x(%u) Modified:%s",varsize,varsize,savestr);
    FastStringXY(C89_92(130,210),C89_92(94,122),"[X]:Help");
    FastStringXY(1,C89_92(94,122),tempstr);
#if defined(USE_TI89)
asm("movem.l %d0/%a0,-(%sp)\n"
    "movea.l #0x4c00,%a0\n"
    "moveq   #6,%d0\n"
    "lea     2790(%a0),%a0\n"
"__loop__:\n"
    "not.l   (%a0)+\n"
    "not.l   (%a0)+\n"
    "not.l   (%a0)+\n"
    "not.l   (%a0)+\n"
    "not.l   (%a0)+\n"
    "lea     10(%a0),%a0\n"
    "dbf     %d0,__loop__\n"
    "movem.l (%sp)+,%d0/%a0");
#else
asm("movem.l %d0/%a0,-(%sp)\n"
    "movea.l #0x4c00,%a0\n"
    "moveq   #6,%d0\n"
    "lea     3630(%a0),%a0\n"
"__loop__:\n"
    "not.l   (%a0)+\n"
    "not.l   (%a0)+\n"
    "not.l   (%a0)+\n"
    "not.l   (%a0)+\n"
    "not.l   (%a0)+\n"
    "not.l   (%a0)+\n"
    "not.l   (%a0)+\n"
    "not.w   (%a0)+\n"
    "dbf     %d0,__loop__\n"
    "movem.l (%sp)+,%d0/%a0");
#endif
}



/*===========================================================================*/
/* outputs character in the right screen part                                */
/*===========================================================================*/
#define PrintRightSideChar(x,y,c) \
  ({char __s[2] = {c,0}; FastStringXY(x,y,__s);})



/*===========================================================================*/
/* outputs hexcode in the left screen part                                   */
/*===========================================================================*/
static inline void PrintLeftSideNumber(short x,short y,unsigned c) {
    static const unsigned char hex[] = "0123456789ABCDEF";
    char s[3];
    s[0] = hex[(c >> 4) & 0x0f];
    s[1] = hex[c & 0x0f];
    s[2] = 0;
    FastStringXY(x,y,s);
}

/*===========================================================================*/
/* prints address (format=%04X)                                              */
/*===========================================================================*/
static inline void PrintAddress(short x,short y,unsigned short c) {
    static const unsigned char hex[] = "0123456789ABCDEF";
    char s[5];

    s[3] = hex[c & 0x0f]; c>>=4;
    s[2] = hex[c & 0x0f]; c>>=4;
    s[1] = hex[c & 0x0f]; c>>=4;
    s[0] = hex[c];
    s[4] = 0;
    FastStringXY(x,y,s);
}

/*===========================================================================*/
/* outputs helppage                                                          */
/*===========================================================================*/
static inline void DrawHelpPageHexEditor(void) {
    static const char* help[] = {
        "TIGCC Tools Hex Editor v "TTHEDIT_VERSION,
        "",
        "by tenordave@hotmail.com",
        "&   thomas.nussbaumer@gmx.net",
        "&   xdanger1@caramail.com",
        "",
        "Arrow Keys: Move Cursor",
        "ESC: Cancel / Quit & Save",
        "F1: Page Down",
        "F2: Page Up",
        "F3: Last Page",
        "F4: First Page",
        "F5: Boss Key *DOES NOT SAVE!*",
        "0-9 & A-F: Change current number",
        "",
        "--- Press any key to return ---"};

    short i;
    for (i=0;i<16;i++) FastStringXY(0, i*6, help[i]);
}



/*===========================================================================*/
/* main function                                                             */
/*===========================================================================*/
void ViewHEX(SYM_ENTRY* symptr) {
    unsigned char* src;
    unsigned char* hdata;
    short          save_flag    = 0;
    short          updatescreen = 1;
    short          changeflag   = 0;
    short          curx         = 0; // cursor x coordinate
    short          cury         = 0; // cursor y coordinate
    short          curab        = 0; // var for high or low part of byte
    short          cursorx      = 0; // where to draw the cursor in x
    short          cursory      = 0; // where to draw the cursor in y
    unsigned short varplace     = 0; // Where we are in the variable
    HANDLE         symptr_handle;
    unsigned short varsize;
    short          key;
    short          i,j;
    short          locked_flag;
    short          archived_flag;
    LCD_BUFFER     lcd;

    symptr_handle = symptr->handle;
    locked_flag   = symptr->flags.bits.locked;
    archived_flag = symptr->flags.bits.archived;

    if (!(src=SaveHeapLock(symptr_handle))) return;

    //------------------------------------------------------
    // get size of variable and generate length string (hex)
    //------------------------------------------------------
    varsize = (src[0]<<8 | src[1]);

    //------------------------------------------------------------------
    // skip "length of variable" bytes and prefix first byte of variable
    //------------------------------------------------------------------
    src+=2;

    //------------------------------------------------------------------
    // generate a local copy of the variable content so we can "undo"
    // all modifications simple
    //------------------------------------------------------------------
    if (!(hdata = malloc(varsize))) {
        SaveHeapUnLock();
        ST_helpMsg(GetMsg(MSG_ERR_OUTOFMEM)); //"ERROR: out of memory");
        return;
    }

    memcpy(hdata, src, varsize);

    //-----------------
    // MAIN LOOP ...
    //-----------------
    while (1) {
        //-------------------------------------------
        // if necessary -> update the complete screen
        //-------------------------------------------
        if (updatescreen) {
            ClearScreen();
            //-----------------
            // for all rows ...
            //-----------------
            for (i=0;i<NR_ROWS;i++) {
                //draw where we are in the file (hex output)

                if (varplace+i*NR_COLUMNS<varsize) {
                    PrintAddress(0,i*6+1,varplace+i*NR_COLUMNS);
                }

                //---------------------------------------
                // draw all bytes of a row ...
                //---------------------------------------
                for (j=0;j<NR_COLUMNS;j++) {
                    if (varplace+j+i*NR_COLUMNS<varsize) {
                        PrintLeftSideNumber(20+10*j,i*6+1,hdata[varplace+i*NR_COLUMNS+j]);
                        PrintRightSideChar(C89_92(105,145)+j*6,i*6+1,hdata[varplace+i*NR_COLUMNS+j]);
                    }
                }
            }
            //-----------------------------------------------------
            //draw info bar in one fell swoop and reset update flag
            //-----------------------------------------------------
            DrawInfoBar(locked_flag,varsize,save_flag);
            updatescreen = 0;
        }

        cursorx = 20+(curx<<3)+(curx<<1)+(curab<<2);
        cursory = (cury<<2)+(cury<<1);
        XOR_Cursor(cursorx,cursory);

        key = GetInputKey();

        if (key==264) {// ESC
            break;
        }
        else if (key=='X') {
            LCD_save(lcd);
            ClearScreen();
            DrawHelpPageHexEditor();
            GetUserInput();
            LCD_restore(lcd);
        }

        XOR_Cursor(cursorx,cursory);

        //---------------------------------
        // treatment key for PAGE DOWN (F1)
        //---------------------------------
        if (key==268 && varplace+NR_BYTES<varsize) {
            curx=0,cury=0,varplace+=NR_BYTES,updatescreen=1;
        }
        //---------------------------------
        // treatment key for PAGE UP (F2)
        //---------------------------------
        else if (key==269) {
            updatescreen=1;
            if(varplace >= (unsigned short)NR_BYTES) varplace-=NR_BYTES;
            else varplace=0;
        }
        //----------------------------------
        // treatment key for FIRST PAGE (F4)
        //----------------------------------
        else if (key==271 && varplace!=0) {
            varplace=0,updatescreen=1,curx=0,cury=0,curab=0;
        }
        //----------------------------------
        // treatment key for LAST PAGE (F3)
        //----------------------------------
        else if (key==270 && varplace!=varsize-(varsize%NR_BYTES)) {
            varplace=varsize-(varsize%NR_BYTES),updatescreen=1,curx=0,cury=0,curab=0;
        }
        //----------------------------------
        // treatment key for QUICK EXIT (F5)
        //----------------------------------
        else if (key==272) {
            save_flag=0; break;
        }

        //-------------------------------------
        // keys 0-9 and a-f and A-F (edit keys)
        //-------------------------------------
        else if (((key>='0' && key<='9')  ||
                  (key>='A' && key<='F')) && !locked_flag && !archived_flag)
        {
            unsigned char value;

            if (key >='0' && key<='9') {
                value = key - '0';
            }
            else {
                value = key - 'A' + 10;
            }

            if (curab==0) {
                hdata[varplace+cury*NR_COLUMNS+curx] &=0x0F;
                hdata[varplace+cury*NR_COLUMNS+curx] |= (value <<4);
            }
            else {
                hdata[varplace+cury*NR_COLUMNS+curx] &=0xF0;
                hdata[varplace+cury*NR_COLUMNS+curx] |= (value & 0x0F);
            }
            save_flag=1;
            if (save_flag && !changeflag) changeflag=1,updatescreen=1;
            PrintLeftSideNumber(20+10*curx,cury*6+1,hdata[varplace+cury*NR_COLUMNS+curx]);

            for (i=0;i<NR_COLUMNS && varplace+cury*NR_COLUMNS+i<varsize;i++) {
                PrintRightSideChar(C89_92(105,145)+i*6,cury*6+1,hdata[varplace+cury*NR_COLUMNS+i]);
            }

            DrawInfoBar(locked_flag,varsize,save_flag);
            //--------------------------------------
            // TRICK: simulate a cursor right action
            //--------------------------------------
            key = KEY_RIGHT;
        }

        //------------------------------
        // treatment key RIGHT
        //------------------------------
        if (key==KEY_RIGHT) {
            if (curab==0) curab++;
            else if (curx < NR_COLUMNS-1 && varplace+curx+cury*NR_COLUMNS+1<varsize) curx++,curab--;

            //-------------------------------------------------------------
            // TRICK:
            // simulate a cursor DOWN action + set cursor to first position
            //-------------------------------------------------------------
            else if (curx == NR_COLUMNS-1 && varplace+(cury+1)*NR_COLUMNS<varsize) {
                key = KEY_DOWN;
                curx = 0,curab--;
            }
        }
        //------------------------------
        // treatment key LEFT
        //------------------------------
        else if (key==KEY_LEFT) {
            if(curab==1)    curab--;
            else if(curx>0) curx--,curab++;
            else if (curx == 0 && (cury>0 || varplace>0)) {
                key = KEY_UP;
                curx = NR_COLUMNS-1,curab++;
            }
        }

        //------------------------------
        // treatment key UP
        //------------------------------
        if (key==KEY_UP && (cury>0 || varplace>0)) {
            cury--;
            if(cury<0) {
                cury=NR_ROWS-1,updatescreen=1;
                if(varplace>=NR_BYTES) varplace-=NR_BYTES;
                else varplace=0;
            }
        }
        //------------------------------
        // treatment key DOWN
        //------------------------------
        else if(key==KEY_DOWN && varplace+(cury+1)*NR_COLUMNS<varsize) {
            if (varplace+curx+(cury+1)*NR_COLUMNS<varsize) cury++;
            if (cury>NR_ROWS-1) {
                cury=0;
                curx=0;
                varplace+=NR_BYTES;
                updatescreen=1;
            }
        }

    } //end main while loop

    //---------------------------------------------------------------
    // if anything changed query user if he wants to save the changes
    //---------------------------------------------------------------
    if (save_flag) {
        LCD_save(lcd);
        if (DlgMessage("Variable Data Changed!", "Save Changes?", BT_CANCEL, BT_OK)==13) {
            memcpy(src, hdata, varsize);
        }
        LCD_restore(lcd);
    }

    //-------------------------------------------------------------------
    // restore original screen, free allocated memory and unlock variable
    //-------------------------------------------------------------------
    free(hdata);
    SaveHeapUnLock();
}


//=============================================================================
// Revision History
//=============================================================================
//
// $Log: viewers.c,v $
// Revision 1.9  2002/10/01 09:06:49  tnussb
// ALPHA lock handled now correctly while hexediting
//
// Revision 1.8  2002/09/10 11:28:43  tnussb
// changes up to v1.30 Beta 4 / examine history.txt for details
//
// Revision 1.7  2002/03/15 15:16:39  tnussb
// (1) Thanx to Cody Cutrer the hexviewer is now replaced by the hexeditor of
//     the TIGCC Tools Suite
// (2) optimizations of the initial hexeditor integration
// (3) Textviewer on TI92p uses now fullscreen
//
// Revision 1.6  2002/02/25 12:54:46  tnussb
// minor size optimizations and beautifying
//
// Revision 1.5  2002/02/07 18:01:19  tnussb
// generic commit
//
// Revision 1.4  2001/02/04 13:11:28  Thomas Nussbaumer
// changes up to version 1.00 RC2 (see history.txt)
//
// Revision 1.3  2001/01/26 21:04:44  Thomas Nussbaumer
// changes for version 0.80 [see history.txt]
//
//

