/******************************************************************************
*
* project name:  TICT-Explorer
* file name:     vatutils.c
* initial date:  17/01/2001
* author:        thomas.nussbaumer@gmx.net
*
* description:   contains some utils to make operating the VAT easier
*
* $Id: vatutils.c,v 1.5 2002/02/25 12:54:46 tnussb Exp $
*
******************************************************************************/


/*===========================================================================*/
/* generates TIOS filename from C string                                     */
/* returns pointer to end of TIOS filename                                   */
/*===========================================================================*/
char* GenerateTIOSName(const char* cname,char* buffer) {
    buffer[0] = 0;
    return (strcpy(buffer + 1, cname) + strlen(cname));
}


/*===========================================================================*/
/* returns SYM_ENTRY pointer of symbol specified by given name               */
/* NOTE: name can be of any kind (C or TIOS). if name[0] is 0 this routine   */
/*       believes its a tios name otherwise its a C name.                    */
/*===========================================================================*/
SYM_ENTRY* GetSymByName(const char* name) {
    char buf[50];
    HSym hs;

    if (*name) hs = SymFind(GenerateTIOSName(name,buf));  /* cname    */
    else       hs = SymFind(name);                        /* tiosname */

    if (!hs.folder) return NULL;
    else            return DerefSym(hs);
}


#if !defined(MINIMIZE_VATUTILS)
/*===========================================================================*/
/* returns SYM_ENTRY pointer of symbol specified by given folder and name    */
/* NOTE: folder and name MUST be C strings                                   */
/*===========================================================================*/
SYM_ENTRY* GetSymByFolderAndName(char* folder,char* name) {
    char buf[50];

    sprintf(buf,"%s\\%s",folder,name);
    return GetSymByName(buf);
}
#endif

#if !defined(MINIMIZE_VATUTILS)
/*===========================================================================*/
/* returns 0 if file referenced name is locked or doesn't exist              */
/* NOTE:  name can be of any kind (C or TIOS). if name[0] is 0 this routine  */
/*        believes its a tios name otherwise its a C name                    */
/* NOTE2: archived files are returned as locked by the AMS                   */
/*===========================================================================*/
short IsHeapLocked(char* name) {
    SYM_ENTRY* symentry = GetSymByName(name);

    if (!symentry || !symentry->handle) {
        return 0;
    }
    else {
        return HeapGetLock(symentry->handle);
    }
}
#endif

#if !defined(MINIMIZE_VATUTILS)
/*===========================================================================*/
/* returns 1 if file referenced name is a TWIN symbol or doesn't exist       */
/* NOTE:  name can be of any kind (C or TIOS). if name[0] is 0 this routine  */
/*        believes its a tios name otherwise its a C name                    */
/*===========================================================================*/
short IsTwinSymbol(char* name) {
    SYM_ENTRY* symentry = GetSymByName(name);

    if (!symentry || !symentry->flags.bits.twin) return 0;
    else                                         return 1;
}
#endif

/*===========================================================================*/
/* returns 1 if entry is in use                                              */
/*===========================================================================*/
short IsEntryInUse(SYM_ENTRY* symentry) {
    if (symentry && symentry->handle) {
        if (symentry->flags.bits.twin ||
            (!symentry->flags.bits.archived && HeapGetLock(symentry->handle)))
        {
            return 1;
        }
    }

    return 0;
}


#if !defined(MINIMIZE_VATUTILS)
/*===========================================================================*/
/* returns 1 if file is in use                                               */
/* NOTE:  name can be of any kind (C or TIOS). if name[0] is 0 this routine  */
/*        believes its a tios name otherwise its a C name                    */
/*===========================================================================*/
short IsInUse(char* name) {
    return IsEntryInUse(GetSymByName(name));
}
#endif

/*===========================================================================*/
/* tries to lock handle h and returns pointer to memory if okay, otherwise 0 */
/* NOTE: this function sets __was_locked__ to 1 if handle was already locked */
/* NOTE2: CANNOT BE USED NESTED                                              */
/*===========================================================================*/
volatile HANDLE __handle_to_unlock__ = 0;

#define SAVEHEAPLOCK_RESET() {__handle_to_unlock__ = 0;}

unsigned char* SaveHeapLock(HANDLE h) {
    if (!h) {
        return NULL;
    }
    else if (HeapGetLock(h)) {
        __handle_to_unlock__ = 0;
        return HeapDeref(h);
    }
    else {
        __handle_to_unlock__ = h;
        return HLock(h);
    }
}


/*===========================================================================*/
/* unlocks a handle                                                          */
/* NOTE: this function should only be called after a previous SaveHeapLock() */
/* NOTE2: CANNOT BE USED NESTED                                              */
/*===========================================================================*/
void SaveHeapUnLock(void) {
    // check if we use this function only in pairs
    if (__handle_to_unlock__) HeapUnlock(__handle_to_unlock__);
    __handle_to_unlock__ = 0;
}



//=============================================================================
// Revision History
//=============================================================================
//
// $Log: vatutils.c,v $
// Revision 1.5  2002/02/25 12:54:46  tnussb
// minor size optimizations and beautifying
//
// Revision 1.4  2002/02/07 18:01:18  tnussb
// generic commit
//
// Revision 1.3  2001/02/04 13:11:28  Thomas Nussbaumer
// changes up to version 1.00 RC2 (see history.txt)
//
// Revision 1.2  2001/01/26 21:04:44  Thomas Nussbaumer
// changes for version 0.80 [see history.txt]
//
// Revision 1.1  2001/01/17 23:37:39  Thomas Nussbaumer
// initial version
//
//
