/******************************************************************************
*
* project name:  TICT-Explorer
* file name:     extension.c
* initial date:  06/01/2001
* author:        thomas.nussbaumer@gmx.net
*
* description:   handles file to extension mapping
*
* $Id: extension.c,v 1.11 2002/09/23 09:36:35 tnussb Exp $
*
******************************************************************************/

#include "tts/ttarchive.h"  // ttarchive stuff

//-----------------------------------------------------------------------------
// DON'T CHANGE THE TYPE DEFINES, THEY ARE STORDED ALSO IN THE CONFIGFILE !!
//-----------------------------------------------------------------------------
#define TYPE_NONE        0
#define TYPE_NOSTUB_EXE  1
#define TYPE_PPG_EXE     2
#define TYPE_DOORS_EXE   3
#define TYPE_DOORS_LIB   4
#define TYPE_PIC         5
#define TYPE_EBOOK       6
#define TYPE_QUILL       7
#define TYPE_SCOTTFREE   8
#define TYPE_TEXT        9
#define TYPE_BASIC      10
#define TYPE_PVPIC      11


/*===========================================================================*/
/* check if given symentry is a TTArchive and if yes, try to extract its     */
/* description, otherwise use default description ("eBook")                  */
/*                                                                           */
/* returns 1 if okay,    0 otherwise                                         */
/*===========================================================================*/
short CheckForEbook(unsigned char* src, file_t* buffer) {
    src+=2;  // skip "length of variable" bytes

    if (!ttarchive_valid(src)) return 0;

    src = ttarchive_info(src);

    if (src) strncpy(buffer->comment,src,TTARCHIVE_INFOLENGTH);
    else     strcpy(buffer->comment,MSGDIRECT_COMMENT_EBOOK);

    buffer->type = TYPE_EBOOK;
    return 1;
}


/*===========================================================================*/
/* checks executable if it is a quill file                                   */
/* returns 1 if quill marker found (sets type and comment)                   */
/*===========================================================================*/
short CheckForQuill(unsigned char* src,unsigned short size,file_t* buffer) {
    unsigned char*       end    = src + ((size < 1024) ? size : 1024);
    short                idx    = 0;
    const unsigned char* marker = "***QUILL***";

    while (src < end) {
        if (*src == marker[idx]) {
            idx++;
        }
        else if (*src == marker[0]) {
            idx = 1;
        }
        else {
            idx = 0;
        }
        src++;
        if (idx == 12) {
            strncpy(buffer->comment,src,MAX_COMMENT);
            buffer->type = TYPE_QUILL;
            return 1;
        }
    }

    return 0;
}


/*===========================================================================*/
/* check for RPGM variable ...                                               */
/*===========================================================================*/
short IsPRGMVar(unsigned char* src,unsigned char* end) {
    while (src > end && *src-- != 0xE5);

    if (*src == 0xE4 && *(src-1) == 0x19) {
        return 1;
    }

    return 0;
}



/*===========================================================================*/
/* setups file type                                                          */
/*===========================================================================*/
void SetFileType(unsigned char* src,file_t* buffer,unsigned short size) {
    const char* tmpext;
    char  sb[5];
    char* orig_src = src;

    src += size-1;

    memset(sb,0,5);

    switch(*src) {
        case STR_TAG:
            tmpext = "STR";
            strcpy(buffer->comment,MSGDIRECT_COMMENT_STRING);
            break;
        case TEXT_TAG:
            tmpext = "TEXT";
            buffer->type = TYPE_TEXT;
            strcpy(buffer->comment,MSGDIRECT_COMMENT_TEXT);
            break;
        case PIC_TAG:
            tmpext = "PIC";
            strcpy(buffer->comment,MSGDIRECT_COMMENT_PIC);
            buffer->type = TYPE_PIC;
            break;
        case GDB_TAG:
            tmpext = "GDB";
            strcpy(buffer->comment,MSGDIRECT_COMMENT_GDB);
            break;
        case FIG_TAG:
            tmpext = "FIG";
            strcpy(buffer->comment,MSGDIRECT_COMMENT_FIG);
            break;
        case MAC_TAG:
            tmpext = "MACR";
            strcpy(buffer->comment,MSGDIRECT_COMMENT_MACR);
            break;
        case DATA_TAG:
            tmpext = "DATA";
            strcpy(buffer->comment,MSGDIRECT_COMMENT_DATA);
            break;
        case LIST_TAG:
            if (*(src-1) == LIST_TAG) {
                tmpext = "MAT";
                strcpy(buffer->comment,MSGDIRECT_COMMENT_MAT);
            }
            else {
                tmpext = "LIST";
                strcpy(buffer->comment,MSGDIRECT_COMMENT_LIST);
            }
            break;
        case FUNC_TAG:
            if (IsPRGMVar(src,src-size+1)) {
                tmpext = "PRGM";
                strcpy(buffer->comment,MSGDIRECT_COMMENT_PRGM);
            }
            else {
                tmpext = "FUNC";
                strcpy(buffer->comment,MSGDIRECT_COMMENT_FUNC);
            }
            buffer->type = TYPE_BASIC;
            break;
        case OTH_TAG:
            src-=5;
            if (!*src) memcpy(sb,src+1,3);
            else       memcpy(sb,src,4);
            tmpext = sb;
            if (!strcmp(sb,"ppg")) {
                buffer->type = TYPE_PPG_EXE;
                strcpy(buffer->comment,MSGDIRECT_COMMENT_PPG);
            }
            else if (!strcmp(sb,"ebk")) {
                CheckForEbook(orig_src,buffer);
            }
            else if (!strcmp(sb,"SDBF")) {
                buffer->type = TYPE_SCOTTFREE;
                src-=2;
                while (*src--);
                strncpy(buffer->comment,src+2,MAX_COMMENT);
            }
            break;
        case ASM_TAG:
            if ((!strncmp(orig_src+6,"68kP",4)) || (!strncmp(orig_src+4,"68cA",4))) {
                tmpext       = "ASM";
                buffer->type = TYPE_DOORS_EXE;
                //---------------------------
                // try to extract comment ...
                //---------------------------
                if ((!strncmp(orig_src+6,"68kP",4)) && *(unsigned short*)(orig_src+12)) {
                //if (*(unsigned short*)(orig_src+12)) {
                    strncpy(buffer->comment,orig_src + 2 + *(unsigned short*)(orig_src+12),MAX_COMMENT);
                }
                else {
                    strcpy(buffer->comment,MSGDIRECT_COMMENT_ASM);
                }
                break;
            }
            else if (!strncmp(orig_src+6,"68kL",4)) {
                tmpext       = "LIB";
                buffer->type = TYPE_DOORS_LIB;
                strcpy(buffer->comment,MSGDIRECT_COMMENT_LIB);
                break;
            }
            else {
                tmpext        = "EXE";
                buffer->type  = TYPE_NOSTUB_EXE;
                if (!CheckForQuill(orig_src,size,buffer)) {
                    strcpy(buffer->comment,MSGDIRECT_COMMENT_EXE);
                }
                break;
            }
        case 0x00:  // may be a picture-view picture
            if (*(src-1) == 0x00) {
                PV_FILE_HEADER* fh = (PV_FILE_HEADER*)(orig_src+2);
                // better canditate for a picture-view picture
                if (fh->magic == PV_MAGIC) {
                    unsigned short grays = 2<<(fh->nr_planes-1);
                    tmpext = "PPIC";
                    buffer->type = TYPE_PVPIC;
                    sprintf(buffer->comment,"%s (%uG-%ux%u)",
                            (fh->nr_pics) ? MSGDIRECT_COMMENT_ANIM:MSGDIRECT_COMMENT_PIC,grays,
                            ((PV_PLANE_HEADER*)(orig_src+2+sizeof(PV_FILE_HEADER)))->x_dim << 3,
                            ((PV_PLANE_HEADER*)(orig_src+2+sizeof(PV_FILE_HEADER)))->y_dim);
                    break;
                }
            }

            // else fall through !!!
        default:
#if defined(DEBUG_SHOW_FILETAGS)
            tmpext = sb;
            sprintf(sb,"x%02X",*src);
            break;
#else
            tmpext = "EXPR";
            strcpy(buffer->comment,MSGDIRECT_COMMENT_EXPR);
            break;
#endif
    }

    memcpy(&(buffer->ext[0]),tmpext,4);
}


//=============================================================================
// Revision History
//=============================================================================
//
// $Log: extension.c,v $
// Revision 1.11  2002/09/23 09:36:35  tnussb
// PreOS's pack archives gets recognized now correctly
//
// Revision 1.10  2002/09/10 11:28:41  tnussb
// changes up to v1.30 Beta 4 / examine history.txt for details
//
// Revision 1.9  2002/03/15 15:05:26  tnussb
// path to ttarchive changed (now in subfolder tts)
//
// Revision 1.8  2002/02/07 18:01:17  tnussb
// generic commit
//
// Revision 1.7  2001/02/10 10:22:59  Thomas Nussbaumer
// more strings replaced by defines
//
// Revision 1.6  2001/02/04 13:11:27  Thomas Nussbaumer
// changes up to version 1.00 RC2 (see history.txt)
//
// Revision 1.5  2001/01/26 21:04:44  Thomas Nussbaumer
// changes for version 0.80 [see history.txt]
//
// Revision 1.4  2001/01/21 17:05:09  Thomas Nussbaumer
// changes related to v0.65
//
// Revision 1.3  2001/01/17 23:38:41  Thomas Nussbaumer
// changes related to version 0.50 (see history.txt)
//
// Revision 1.2  2001/01/15 00:08:19  Thomas Nussbaumer
// see history.txt (up to v0.34) for what's new
//
// Revision 1.1  2001/01/06 23:09:46  Thomas Nussbaumer
// initial version
//
//
