/*
 * Copyright (C) 2010 Joseph Adams <joeyadams3.14159@gmail.com>
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#include "lambda.h"
#include <tigcclib.h>

#define KEY_2ND     4096
#define KEY_CATALOG 278

Map(long, char*) bindings;

char prompt_buffer[1024],     // holds text currently being edited
     prompt_cur_buffer[1024]; // backs up text of the most recent history entry

typedef struct History History;

struct History {
	History *prev, *next;
	char    *str;
};

static History *history_tail;

static WINDOW    w[1];
static TEXT_EDIT te[1];

static Environment prompt_env;

static const char *curPos(void)
{
	const char *p = prompt_buffer + te->CursorOffset;
	assert(p >= prompt_buffer && p < prompt_buffer + sizeof(prompt_buffer));
	return p;
}

static const char *prevWord(const char *start, const char *p)
{
	while (p > start && p[-1] == ' ')
		p--;
	
	while (p > start && p[-1] != ' ')
		p--;
	
	return p;
}

static const char *nextWord(const char *p, const char *end)
{
	while (p < end && *p == ' ')
		p++;
	
	while (p < end && *p != ' ')
		p++;
	
	return p;
}

void add_binding(long key, char *string)
{
	assert(string != NULL);
	
	/* Prevent the user from setting a binding that
	   will keep them from exiting the program. */
	struct {
		short       key;
		const char *name;
	} blacklist[] = {
		{KEY_ENTER, "ENTER"},
		{KEY_QUIT,  "QUIT"},
		{KEY_ON,    "ON"},
		{KEY_OFF2,  "\x7F+OFF"}
	};
	
	unsigned short i;
	
	for (i = 0; i < sizeof(blacklist) / sizeof(*blacklist); i++) {
		if (key == blacklist[i].key)
			error("Cannot bind the %s key", blacklist[i].name);
	}
	
	bindings = avl_insert(bindings, (void*)key, string);
}

static CALLBACK void te_handler(EVENT *ev)
{
	if (ev->Type == CM_KEYPRESS) {
		unsigned short key = ev->extra.Key.Code;
		
		if (key == KEY_ENTER)
			ER_throw(1);
		if (key == KEY_QUIT)
			ER_throw(2);
		
		// The default handler for KEY_OFF (2ND+ON) takes you back to the AMS
		// without properly shutting down the program.
		if (key == KEY_OFF) {
			off();
			return;
		}
		
		// Check the custom bindings
		{
			char *string = avl_lookup(bindings, (void*)(long)key);
			
			if (string != NULL) {
				TE_pasteText(te, string, strlen(string));
				return;
			}
		}
		
		if (key == KEY_UP)
			ER_throw(3);
		if (key == KEY_DOWN)
			ER_throw(4);
		
		// Support diamond+arrow for moving between words
		//
		// FIXME: The cursor sometimes smears when this is used.
		if (key == (unsigned short)(KEY_LEFT + KEY_DIAMOND)) {
			const char *p = prevWord(prompt_buffer, curPos());
			CU_stop();
			TE_select(te, p - prompt_buffer, p - prompt_buffer);
			TE_focus(te);
			w->Flags |= WF_DIRTY;
			EV_paintOneWindow();
			return;
		}
		if (key == (unsigned short)(KEY_RIGHT + KEY_DIAMOND)) {
			const char *n = nextWord(curPos(), prompt_buffer + te->CurSize);
			CU_stop();
			TE_select(te, n - prompt_buffer, n - prompt_buffer);
			TE_focus(te);
			w->Flags |= WF_DIRTY;
			EV_paintOneWindow();
			return;
		}
		
		// KEY_APPS is nothing but trouble (for now)
		if (key == KEY_APPS ||
				key == KEY_APPS + KEY_2ND ||
				key == KEY_APPS + KEY_DIAMOND)
			return;
		
		// KEY_VARLNK is also nothing but trouble
		if (key == KEY_VARLNK)
			return;
		
		// Support the CATALOG key for showing a popup of currently defined identifiers
		if (key == KEY_CATALOG && prompt_env != NULL) {
			const char *txt = envMenu(prompt_env);
			
			if (txt != NULL)
				TE_pasteText(te, txt, strlen(txt));
			
			return;
		}
	}
	
	if (!TE_handleEvent(te, ev))
		EV_defaultHandler(ev);
}

void add_history(const char *str)
{
	History *h = alloc(History);
	h->prev = history_tail;
	h->next = NULL;
	h->str  = gc_strdup(str);
	
	if (history_tail != NULL)
		history_tail->next = h;
	
	history_tail = h;
}

const char *prompt(const char *start, bool use_history, Environment env)
{
	SCR_STATE screen;
	
	prompt_env = env;
	
	if (start) {
		/*
		 * Make sure cursor is at the beginning of the line.
		 * Otherwise, the user will see:
		 *
		 * > putStr "hello"
		 * hello>
		 *
		 * Not only that, but they will have less room to type more text.
		 */
		SaveScrState(&screen);
		if (screen.CurX != 0)
			putchar('\n');
		
		puts(start);
	}
	
	SaveScrState(&screen);
	
	short font = FontGetSys();
	
	SetCurClip(&(SCR_RECT){{0, 0, LCD_WIDTH-1, LCD_HEIGHT-1}});
	
	if (!WinOpen(w, &(WIN_RECT){screen.CurX, screen.CurY, LCD_WIDTH-1, screen.CurY + font*2 + 6}, WF_NOBORDER | WF_SAVE_SCR))
		ER_throw(ER_MEMORY);
	
	WinActivate(w);
	WinFont(w, font);
	
	History *hcursor = NULL;
	prompt_buffer[0] = 0;

	enum {DONE, QUIT, REPEAT, REFRESH} state;

refresh: ;
	unsigned short len = strlen(prompt_buffer);
	TE_openFixed(te, w, NULL, prompt_buffer, sizeof(prompt_buffer), TE_MORE_ELLIPSES);
	TE_select(te, len, len);
	
	CU_start();
	
repeat:
	state = DONE;
	EV_captureEvents(te_handler);
	TRY
		EV_eventLoop();
	ONERR
		EV_captureEvents(NULL);
		
		switch (errCode) {
			case 2:
				state = QUIT;
				break;
			
			case 3: // KEY_UP: go to earlier history entry
				if (!use_history) {
					state = REPEAT;
					break;
				}
				
				if (history_tail == NULL || (hcursor && hcursor->prev == NULL)) {
					// All the way at the top; nothing to do
					state = REPEAT;
				} else if (hcursor == NULL) {
					// Walking away from current prompt; save it, and load most recent history entry
					strcpy(prompt_cur_buffer, prompt_buffer);
					hcursor = history_tail;
					strcpy(prompt_buffer, hcursor->str);
					state = REFRESH;
				} else {
					// Walking up another entry; just throw out what we had before (fixme)
					hcursor = hcursor->prev;
					strcpy(prompt_buffer, hcursor->str);
					state = REFRESH;
				}
				break;
			
			case 4: // KEY_DOWN: go to next history entry
				if (!use_history) {
					state = REPEAT;
					break;
				}
				
				if (hcursor == NULL) {
					// All the way at the bottom; nothing to do
					state = REPEAT;
				} else if (hcursor->next == NULL) {
					// Going back to current prompt; restore it
					hcursor = NULL;
					strcpy(prompt_buffer, prompt_cur_buffer);
					state = REFRESH;
				} else {
					// Going down another entry; just throw out what we had before (fixme)
					hcursor = hcursor->next;
					strcpy(prompt_buffer, hcursor->str);
					state = REFRESH;
				}
				break;
			
			default: ;
		}
	ENDTRY
	
	if (state == REPEAT)
		goto repeat;
	
	TE_close(te);
	CU_start();
	
	if (state == REFRESH)
		goto refresh;
	
	WinClose(w);
	RestoreScrState(&screen);
	
	if (state == QUIT)
		return NULL;
	
	puts(prompt_buffer);
	putchar('\n');
	return prompt_buffer;
}