/*
 * Copyright (C) 2010 Joseph Adams <joeyadams3.14159@gmail.com>
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#include "lambda.h"

#include <tigcclib.h>
#include "maths.h"

Environment defaultEnv;

static long open_handles;


/*********************** IO Actions (no arguments) *******************/

static Value *concatMap_f(Value *f, Value *xss);

static Value *bind_f(Value *m, Value *k)
{
	if (!is_func(k))
		error("Second argument to >>= must be a function");
	
	switch (getType(m)) {
		case T_IO:
			return mkBindValue(m, k);
		
		case T_LIST:
			return concatMap_f(k, m);
		
		case T_FUNC:
			return apply(apply(bindFunc_v, m), k);
		
		case T_RETURN:
			return apply(k, m->return_); // (return x) >>= f = f x
		
		default:
			error(">>= not implemented for %s", typeStr(m));
	}
}

static Value *return_f(Value *x)
{
	Value *v = mkValue(V_RETURN);
	v->return_ = x;
	return v;
}

static const char *prompt_in_action(const char *label)
{
	const char *line;
	
	disable_break();
	line = prompt(label, false, NULL);
	enable_break();
	
	if (!line)
		error("quit");
	
	return line;
}

static Value *getLine_callback(void *ctx)
{
	(void) ctx;
	const char *line = prompt_in_action(NULL);
	
	return buildString(line);
}

static Value *readLn_callback(void *ctx)
{
	(void) ctx;
	const char *line = prompt_in_action(NULL);
	
	return evaluate(parseExpression(line), defaultEnv);
}

static Value *heapAvail_callback(void *ctx)
{
	(void) ctx;
	
	return mkIntValue(HeapAvail());
}

static Value *performGC_callback(void *ctx)
{
	(void) ctx;
	
	performGC();
	
	return unit_v;
}

static Value *ngetchx_callback(void *ctx)
{
	(void) ctx;
	
	disable_break();
	short n = ngetchx();
	enable_break();
	
	return mkIntValue(n);
}

static Value *getFont_callback(void *ctx)
{
	(void) ctx;
	
	unsigned char font = FontGetSys();
	
	return mkIntValue(font);
}

static Value *clrscr_callback(void *ctx)
{
	(void) ctx;
	clrscr();
	return unit_v;
}

static Value *openHandles_callback(void *ctx)
{
	(void) ctx;
	return mkIntValue(open_handles);
}

static Value *rand_callback(void *ctx)
{
	(void) ctx;
	
	return mkIntValue(rand());
}

struct {
	const char *name;
	IOFunc      callback;
	void       *ctx;
} nullary_io_ops[] = {
	{"getLine",     getLine_callback,     NULL},
	{"readLn",      readLn_callback,      NULL},
	{"heapAvail",   heapAvail_callback,   NULL},
	{"performGC",   performGC_callback,   NULL},
	{"ngetchx",     ngetchx_callback,     NULL},
	{"getFont",     getFont_callback,     NULL},
	{"clrscr",      clrscr_callback,      NULL},
	{"openHandles", openHandles_callback, NULL},
	{"rand",        rand_callback,        NULL}
};

/************************** Nullary values ***************************/

typedef struct {
	PrimeSieve    sieve;
	unsigned long cur;
	unsigned long max;
} PrimesCtx;

static Value *primes_callback(void *ctxp)
{
	PrimesCtx     *ctx   = ctxp;
	PrimeSieve     sieve = ctx->sieve;
	unsigned long  i;
	
	check_break();
	
	for (i = ctx->cur; ; i++) {
		if (i > ctx->max) {
			ctx->max = ctx->max * 2 + 1;
			gc_free(sieve);
			sieve = ctx->sieve = makePrimeSieve(ctx->max);
		}
		
		if (testPrimeSieve(sieve, i) == 0)
			break;
	}
	
	ctx->cur = i + 1;
	
	return mkConsValue(mkIntValue(i), deferCall(primes_callback, ctx));
}

static Value *primes_f(void)
{
	PrimesCtx *ctx = alloc(PrimesCtx);
	ctx->cur   = 2;
	ctx->max   = 63;
	ctx->sieve = makePrimeSieve(ctx->max);
	
	return primes_callback(ctx);
}

/**************************** Unary functions ************************/

static Value *concat_f(Value *xss);
static Value *nub_f(Value *xs);
static Value *remove_binding_f(Value *key);

static Value *type_f(Value *v)
{
	return mkIntValue(getType(v));
}

static Value *not_f(Value *v)
{
	if (force(v)->tag != V_BOOL)
		error("not: invalid argument");
	
	return v->b ? false_v : true_v;
}

static Value *id_f(Value *x)
{
	return x;
}

static Value *null_f(Value *xs)
{
	if (force(xs)->tag == V_CONS)
		return false_v;
	else if (xs->tag == V_NIL)
		return true_v;
	else if (xs->tag == V_RETURN)
		return false_v;
	else
		error("null: invalid argument (%s)", typeStr(xs));
}

static Value *head_f(Value *xs)
{
	if (force(xs)->tag == V_CONS)
		return xs->cons.x;
	else if (xs->tag == V_NIL)
		error("head: empty list");
	else if (xs->tag == V_RETURN)
		return xs->return_;
	else
		error("head: invalid argument (%s)", typeStr(xs));
}

static Value *tail_f(Value *xs)
{
	if (force(xs)->tag == V_CONS)
		return xs->cons.xs;
	else if (xs->tag == V_NIL)
		error("tail: empty list");
	else if (xs->tag == V_RETURN)
		return nil_v;
	else
		error("tail: invalid argument (%s)", typeStr(xs));
}

static Value *last_f(Value *xs)
{
	Value *last = NULL, *x;
	
	if (!is_list(xs))
		error("last: invalid argument");
	
	list_foreach(xs, x)
		last = x;
	
	if (last == NULL)
		error("last: empty list");
	
	return last;
}

static Value *init_callback(void *ctxp)
{
	Value *xs = ctxp, *xs_tail = list_tail(xs);
	
	if (list_head(xs_tail) == NULL)
		return nil_v;
	
	return mkConsValue(list_head(xs), deferCall(init_callback, xs_tail));
}

static Value *init_f(Value *xs)
{
	if (!is_list(xs))
		error("init: invalid argument");
	
	Value *h = list_head(xs);
	if (h == NULL)
		error("init: empty list");
	
	return init_callback(xs);
}

static Value *length_f(Value *i)
{
	long long  ret = 0;
	Value     *x;
	
	if (!is_list(i))
		error("length: invalid argument (%s)", typeStr(i));
	
	list_foreach(i, x)
		ret++;
	
	return mkIntValue(ret);
}

static Value *fix_f(Value *f)
{
	return apply(f, deferApply(fix_v, f));
}

static Value *printLn_callback(void *x)
{
	printLn(force(x));
	return unit_v;
}

static Value *printLn_f(Value *x)
{
	return mkIOValue(printLn_callback, x);
}

static Value *print_callback(void *x)
{
	print(force(x));
	return unit_v;
}

static Value *print_f(Value *x)
{
	return mkIOValue(print_callback, x);
}

static Value *printDebug_callback(void *x)
{
	printValueDebug(force(x));
	return unit_v;
}

static Value *printDebug_f(Value *x)
{
	return mkIOValue(printDebug_callback, x);
}

static Value *putChar_callback(void *x)
{
	Value *c = x;
	
	if (force(c)->tag != V_CHAR)
		error("putChar passed a value of type %s", typeStr(c));
	
	putchar(c->c);
	return unit_v;
}

static Value *putChar_f(Value *c)
{
	return mkIOValue(putChar_callback, c);
}

static Value *putStr_callback(void *ptr)
{
	Value *i = ptr,
	      *x;
	
	if (!is_list(i))
		error("putStr: expected string, but passed %s", typeStr(i));
	
	list_foreach(i, x) {
		if (force(x)->tag != V_CHAR)
			error("Expected string, but list has an item of type %s", typeStr(x));
		putchar(x->c);
	}
	
	return unit_v;
}

static Value *putStr_f(Value *c)
{
	return mkIOValue(putStr_callback, c);
}

static Value *prompt_callback(void *ptr)
{
	Value *label = ptr;
	char   buffer[256];
	const char *line;
	
	readString(label, buffer, sizeof(buffer));
	line = prompt_in_action(buffer);
	
	return evaluate(parseExpression(line), defaultEnv);
}

static Value *prompt_f(Value *label)
{
	return mkIOValue(prompt_callback, label);
}

static Value *promptStr_callback(void *ptr)
{
	Value *label = ptr;
	char   buffer[256];
	const char *line;
	
	readString(label, buffer, sizeof(buffer));
	line = prompt_in_action(buffer);
	
	return buildString(line);
}

static Value *promptStr_f(Value *label)
{
	return mkIOValue(promptStr_callback, label);
}

static Value *verboseGC_callback(void *x)
{
	Value *v = x;
	
	if (force(v)->tag != V_INT)
		error("verboseGC must be passed an Int");
	
	gc_set_verbose_level(v->i);
	return unit_v;
}

static Value *verboseGC_f(Value *v)
{
	return mkIOValue(verboseGC_callback, v);
}

static Value *setFont_callback(void *x)
{
	Value *v = x;
	
	if (force(v)->tag != V_INT)
		error("setFont must be passed an Int");
	
	if (v->i >= F_4x6 && v->i <= F_8x10)
		FontSetSys(v->i);
	else
		error("setFont: font index out of range");
	
	return unit_v;
}

static Value *setFont_f(Value *v)
{
	return mkIOValue(setFont_callback, v);
}

static Value *chr_f(Value *v)
{
	if (force(v)->tag != V_INT)
		error("chr must be passed an Int");
	
	return mkCharValue(v->i);
}

static Value *ord_f(Value *v)
{
	if (force(v)->tag != V_CHAR)
		error("ord must be passed a Char");
	
	return mkIntValue((unsigned char) v->c);
}

static Value *isSpace_f(Value *v)
{
	if (force(v)->tag != V_CHAR)
		error("isSpace passed value of type %s", typeStr(v));
	
	return is_space(v->c) ? true_v : false_v;
}
static Value *isDigit_f(Value *v)
{
	if (force(v)->tag != V_CHAR)
		error("isDigit passed value of type %s", typeStr(v));
	
	return is_digit(v->c) ? true_v : false_v;
}
static Value *isHex_f(Value *v)
{
	if (force(v)->tag != V_CHAR)
		error("isHex passed value of type %s", typeStr(v));
	
	return is_hex(v->c) ? true_v : false_v;
}
static Value *isUpper_f(Value *v)
{
	if (force(v)->tag != V_CHAR)
		error("isUpper passed value of type %s", typeStr(v));
	
	return is_upper(v->c) ? true_v : false_v;
}
static Value *isLower_f(Value *v)
{
	if (force(v)->tag != V_CHAR)
		error("isLower passed value of type %s", typeStr(v));
	
	return is_lower(v->c) ? true_v : false_v;
}
static Value *isSymbol_f(Value *v)
{
	if (force(v)->tag != V_CHAR)
		error("isSymbol passed value of type %s", typeStr(v));
	
	return is_symbol(v->c) ? true_v : false_v;
}
static Value *isAlpha_f(Value *v)
{
	if (force(v)->tag != V_CHAR)
		error("isAlpha passed value of type %s", typeStr(v));
	
	return is_alpha(v->c) ? true_v : false_v;
}

static Value *error_f(Value *v)
{
	puts("error: ");
	putStr_callback(v);
	putchar('\n');
	ER_throw(ER_NO_MSG);
}

typedef struct {
	FILE       *f;
	const char *name;
	bool        text_mode;
} ReadFile;

void readFile_finalizer(ReadFile *ctx)
{
	fclose(ctx->f);
	open_handles--;
}

Value *readFile_callback2(void *ctxp)
{
	ReadFile *ctx = ctxp;
	int   c;
	
	check_break();
	
	c = getc(ctx->f);
	if (c == EOF) {
		bool e  = ferror(ctx->f);
		bool ec = fclose(ctx->f) != 0;
		
		open_handles--;
		
		if (e || ec)
			error("Error reading file '%s'", ctx->name);
		
		gc_free_no_finalize(ctx);
		
		return nil_v;
	}
	
	if (c == '\r' && ctx->text_mode)
		c = '\n';
	
	return mkConsValue(mkCharValue(c), deferCall(readFile_callback2, ctx));
}

static Value *readFileBase(const char *name, const char *mode, bool text_mode)
{
	FILE *f = fopen(name, mode);
	if (!f)
		error("Could not read file '%s'", name);
	open_handles++;
	
	ReadFile *ctx = gc_alloc_type(sizeof(*ctx), GC_READFILE);
	ctx->f    = f;
	ctx->name = name;
	ctx->text_mode = text_mode;
	
	return deferCall(readFile_callback2, ctx);
}

static Value *readFile_callback(void *ctxp)
{
	return readFileBase(ctxp, "rt", true);
}

static Value *readFile_f(Value *name)
{
	char buffer[18];
	readString(name, buffer, sizeof(buffer));
	return mkIOValue(readFile_callback, gc_strdup(buffer));
}

static Value *readBinaryFile_callback(void *ctxp)
{
	return readFileBase(ctxp, "rb", false);
}

static Value *readBinaryFile_f(Value *name)
{
	char buffer[18];
	readString(name, buffer, sizeof(buffer));
	return mkIOValue(readBinaryFile_callback, gc_strdup(buffer));
}

typedef struct {
	const char *name;
	const char *mode;
	Value      *content;
	const char *func_name;
} WriteFile;

static Value *writeFile_callback(void *ctxp)
{
	WriteFile *ctx = ctxp;
	Value     *c;
	
	FILE *f = fopen(ctx->name, ctx->mode);
	if (!f)
		error("Could not write to file '%s'", ctx->name);
	open_handles++;
	
	list_foreach(ctx->content, c) {
		if (break_pressed) {
			fclose(f);
			ER_throw(ER_BREAK);
		}
		
		if (force(c)->tag != V_CHAR)
			error("%s: Expected string, but list has an item of type %s", ctx->func_name, typeStr(c));
		
		if (putc(c->c, f) == EOF) {
			fclose(f);
			goto writeError;
		}
	}
	
	if (fclose(f) != 0)
		goto writeError;
	open_handles--;
	
	return unit_v;

writeError:
	error("Error writing to file '%s'", ctx->name);
}

static Value *writeFile_base(Value *f, Value *content, const char *mode, const char *func_name)
{
	if (!is_list(content))
		error("%s: second argument not a string", func_name);
	
	char buffer[18];
	readString(f, buffer, sizeof(buffer));
	
	WriteFile *ctx = alloc(WriteFile);
	ctx->name      = gc_strdup(buffer);
	ctx->mode      = mode;
	ctx->content   = content;
	ctx->func_name = func_name;
	
	return mkIOValue(writeFile_callback, ctx);
}

static Value *writeFile_f(Value *f, Value *content)
{
	return writeFile_base(f, content, "wt", "writeFile");
}

static Value *writeBinaryFile_f(Value *f, Value *content)
{
	return writeFile_base(f, content, "wb", "writeBinaryFile");
}

static Value *appendFile_f(Value *f, Value *content)
{
	return writeFile_base(f, content, "at", "appendFile");
}

static Value *appendBinaryFile_f(Value *f, Value *content)
{
	return writeFile_base(f, content, "ab", "appendBinaryFile");
}

static Value *perform_f(Value *a)
{
	return perform(force(a));
}

static Value *interleave_callback2(void *ctx)
{
	return perform(ctx);
}

static Value *interleave_callback(void *ctx)
{
	return deferCall(interleave_callback2, ctx);
}

static Value *interleave_f(Value *a)
{
	if (getType(a) != T_IO)
		error("interleave passed value of type %s", typeStr(a));
	
	return mkIOValue(interleave_callback, a);
}

static Value *even_f(Value *a)
{
	if (force(a)->tag != V_INT)
		error("even: invalid argument");
	
	return (a->i & 1) ? false_v : true_v;
}

static Value *odd_f(Value *a)
{
	if (force(a)->tag != V_INT)
		error("odd: invalid argument");
	
	return (a->i & 1) ? true_v : false_v;
}

static Value *random_callback(void *ctx)
{
	Value *n = ctx;
	
	if (force(n)->tag != V_INT)
		error("random: invalid argument");
	
	return mkIntValue(random(n->i));
}

static Value *random_f(Value *num)
{
	return mkIOValue(random_callback, num);
}

struct {
	const char *name;
	UnaryFunc   func;
} unary_ops[] = {
	{"type",      type_f},
	{"not",       not_f},
	{"id",        id_f},
	{"null",      null_f},
	{"head",      head_f},
	{"tail",      tail_f},
	{"last",      last_f},
	{"init",      init_f},
	{"length",    length_f},
	{"concat",    concat_f},
	{"nub",       nub_f},
	{"fix",       fix_f},
	{"return",    return_f},
	{"print",     print_f},
	{"printLn",   printLn_f},
	{"printDebug", printDebug_f},
	{"putChar",   putChar_f},
	{"putStr",    putStr_f},
	{"prompt",    prompt_f},
	{"promptStr", promptStr_f},
	{"verboseGC", verboseGC_f},
	{"setFont",   setFont_f},
	{"chr",       chr_f},
	{"ord",       ord_f},
	{"isSpace",   isSpace_f},
	{"isDigit",   isDigit_f},
	{"isHex",     isHex_f},
	{"isUpper",   isUpper_f},
	{"isLower",   isLower_f},
	{"isSymbol",  isSymbol_f},
	{"isAlpha",   isAlpha_f},
	{"error",     error_f},
	{"even",      even_f},
	{"odd",       odd_f},
	{"readFile",        readFile_f},
	{"readBinaryFile",  readBinaryFile_f},
	{"perform",            perform_f},
	{"unsafePerformIO",    perform_f},
	{"interleave",         interleave_f},
	{"unsafeInterleaveIO", interleave_f},
	{"random",    random_f},
	{"unbind",    remove_binding_f}
};


/*************************** Binary functions ************************/

static Value *typeEq_f(Value *a, Value *b)
{
	ValueType ta = getType(a),
	          tb = getType(b);
	
	if (ta == tb)
		return true_v;
	
	if ((ta == T_LIST && tb == T_RETURN) ||
	    (ta == T_RETURN && tb == T_LIST))
		return true_v;
	
	return false_v;
}

static Value *seq_f(Value *a, Value *b)
{
	force(a);
	return b;
}

static Value *apply_f(Value *f, Value *x)
{
	return apply(f, x);
}

static Value *forceApply_f(Value *f, Value *x)
{
	return apply(f, force(x));
}

static Value *cons_f(Value *x, Value *xs)
{
	return mkConsValue(x, xs);
}

static Value *or_f(Value *a, Value *b)
{
	if (force(a)->tag != V_BOOL)
		goto invalid;
	
	if (a->b)
		return true_v;
	
	if (force(b)->tag != V_BOOL)
		goto invalid;
	
	return b->b ? true_v : false_v;
	
invalid:
	error("invalid arguments to (||)");
}

static Value *and_f(Value *a, Value *b)
{
	if (force(a)->tag != V_BOOL)
		goto invalid;
	
	if (!a->b)
		return false_v;
	
	if (force(b)->tag != V_BOOL)
		goto invalid;
	
	return b->b ? true_v : false_v;
	
invalid:
	error("invalid arguments to (||)");
}

static Value *plus_f(Value *a, Value *b)
{
	ValueTag ta = force(a)->tag,
	         tb = force(b)->tag;
	
	if (ta == V_INT && tb == V_INT)
		return mkIntValue(a->i + b->i);
	
	error("Invalid arguments to (+)");
}

static Value *minus_f(Value *a, Value *b)
{
	ValueTag ta = force(a)->tag,
	         tb = force(b)->tag;
	
	if (ta == V_INT && tb == V_INT)
		return mkIntValue(a->i - b->i);
	
	error("Invalid arguments to (-)");
}

static Value *times_f(Value *a, Value *b)
{
	ValueTag ta = force(a)->tag,
	         tb = force(b)->tag;
	
	if (ta == V_INT && tb == V_INT)
		return mkIntValue(a->i * b->i);
	
	error("Invalid arguments to (*)");
}

enum div_op {DIV = 0, MOD = 1, QUOT = 2, REM = 3};

static const char *div_op_name[] = {
	"div",
	"mod",
	"quot",
	"rem"
};

static Value *divBase(Value *a, Value *b, enum div_op op)
{
	long long ret, n, d;
	
	if (force(a)->tag != V_INT || force(b)->tag != V_INT)
		error("Invalid arguments to %s", div_op_name[op]);
	
	n = a->i;
	d = b->i;
	
	if (d == 0)
		error("Divide by zero");
	if (n == 0)
		return mkIntValue(0);
	
	if (op & 1) // MOD or REM
		ret = n % d;
	else
		ret = n / d;
	
	if (op < QUOT) // DIV or MOD
	{
		long long r = n % d;
		
		if ((r < 0 && d > 0) || (r > 0 && d < 0)) {
			if (op == DIV)
				ret--;
			else
				ret += d;
		}
	}
	
	return mkIntValue(ret);
}

static Value *div_f(Value *a, Value *b)
{
	return divBase(a, b, DIV);
}
static Value *mod_f(Value *a, Value *b)
{
	return divBase(a, b, MOD);
}
static Value *quot_f(Value *a, Value *b)
{
	return divBase(a, b, QUOT);
}
static Value *rem_f(Value *a, Value *b)
{
	return divBase(a, b, REM);
}

static Value *pow_f(Value *a, Value *b)
{
	if (force(a)->tag != V_INT || force(b)->tag != V_INT)
		error("Invalid arguments to (^)");
	
	long long base = a->i,
	          exp  = b->i,
	          ret;
	
	if (exp < 0)
		error("Negative exponent");
	
	if (exp == 0) {
		ret = 1;
	} else {
		for (ret = base; exp != 1; exp--) {
			ret *= base;
			check_break();
		}
	}
	
	return mkIntValue(ret);
}

static Value *logBase_f(Value *a, Value *b)
{
	if (force(a)->tag != V_INT || force(b)->tag != V_INT ||
	    a->i < 2 || b->i < 1)
		error("Invalid argument(s) to log");
	
	unsigned long long base = a->i, num = b->i;
	int                ret = 0;
	
	while (num >= base) {
		num /= base;
		ret++;
	}
	
	return mkIntValue(ret);
}

static int compare(Value *a, Value *b);

static int compareLists(Value *a, Value *b)
{
	for (;;) {
		Value *ha = list_head(a),
		      *hb = list_head(b);
		
		if (ha == NULL && hb == NULL)
			return 0;
	
		if (ha == NULL)
			return -1;
		if (hb == NULL)
			return 1;
		
		int c = compare(ha, hb);
		if (c != 0)
			return c;
		
		a = list_tail(a);
		b = list_tail(b);
	}
}

static int compare(Value *a, Value *b)
{
	ValueTag ta = force(a)->tag,
	         tb = force(b)->tag;
	
	if (ta == V_INT && tb == V_INT)
		return a->i < b->i ? -1 : a->i == b->i ? 0 : 1;
	else if (ta == V_CHAR && tb == V_CHAR)
		return (int)(unsigned char)a->c - (int)(unsigned char)b->c;
	else if (ta == V_UNIT && tb == V_UNIT)
		return 0;
	else if (is_list_(a) && is_list_(b))
		return compareLists(a, b);
	
	error("Cannot compare %s and %s", typeStr(a), typeStr(b));
}

static Value *compare_f(Value *a, Value *b)
{
	return mkIntValue(compare(a, b));
}
static Value *compareR_f(Value *a, Value *b)
{
	return mkIntValue(compare(b, a));
}
static Value *eq_f(Value *a, Value *b)
{
	return compare(a, b) == 0 ? true_v : false_v;
}
static Value *ne_f(Value *a, Value *b)
{
	return compare(a, b) != 0 ? true_v : false_v;
}
static Value *lt_f(Value *a, Value *b)
{
	return compare(a, b) < 0 ? true_v : false_v;
}
static Value *le_f(Value *a, Value *b)
{
	return compare(a, b) <= 0 ? true_v : false_v;
}
static Value *gt_f(Value *a, Value *b)
{
	return compare(a, b) > 0 ? true_v : false_v;
}
static Value *ge_f(Value *a, Value *b)
{
	return compare(a, b) >= 0 ? true_v : false_v;
}

typedef struct {
	Value *f;
	Value *xs;
} MapCtx;

static Value *map_callback(void *ctxp)
{
	MapCtx *ctx = ctxp;
	Value  *x   = list_pop(&ctx->xs);
	
	if (x == NULL) {
		gc_free(ctx);
		return nil_v;
	}
	
	return mkConsValue(deferApply(ctx->f, x), deferCall(map_callback, ctx));
}

static Value *map_f(Value *f, Value *xs)
{
	if (!is_func(f))
		error("First argument to map must be a function");
	if (!is_list(xs))
		error("Second argument to map must be a list");
	
	MapCtx *ctx = alloc(MapCtx);
	ctx->f = f;
	ctx->xs = xs;
	
	return map_callback(ctx);
}

typedef struct {
	Value *f;
	Value *xs;
	Value *xss;
} ConcatMapCtx;

static Value *concatMap_callback(void *ctxp)
{
	ConcatMapCtx *ctx = ctxp;
	
	// Use temporary copies of these in case break is pressed.
	Value *xs  = ctx->xs,
	      *xss = ctx->xss;
	
	for (;;) {
		Value *x = list_pop(&xs);
		
		if (x == NULL) {
			// Current list ran out; get another one
			Value *a = list_pop(&xss);
			
			if (a == NULL) {
				// No more lists to concatenate
				gc_free(ctx);
				return nil_v;
			}
			
			xs = ctx->f ? apply(ctx->f, a) : a;
			if (!is_list(xs)) {
				if (ctx->f)
					error("concatMap: function returned %s rather than a list", typeStr(xs));
				else
					error("concat: list contained a(n) %s (must contain lists)", typeStr(xs));
			}
			
			continue;
		}
		
		ctx->xs = xs;
		ctx->xss = xss;
		return mkConsValue(x, deferCall(concatMap_callback, ctx));
	}
}

static Value *concatMap_f(Value *f, Value *xss)
{
	if (!is_func(f))
		error("First argument to concatMap must be a function");
	if (!is_list(xss))
		error("Second argument to concatMap must be a list");
	
	ConcatMapCtx *ctx = alloc(ConcatMapCtx);
	ctx->f = f;
	ctx->xs = nil_v;
	ctx->xss = xss;
	
	return concatMap_callback(ctx);
}

static Value *concat_f(Value *xss)
{
	if (!is_list(xss))
		error("Argument to concat must be a list");
	
	ConcatMapCtx *ctx = alloc(ConcatMapCtx);
	ctx->f = NULL;
	ctx->xs = nil_v;
	ctx->xss = xss;
	
	return concatMap_callback(ctx);
}

typedef struct {
	AVL    set;
	Value *xs;
} NubCtx;

static Value *nub_callback(void *ctxp)
{
	NubCtx *ctx = ctxp;
	
	for (;;) {
		Value *x = list_pop(&ctx->xs);
		
		if (x == NULL) {
			gc_free(ctx);
			return nil_v;
		}
		
		if (avl_member(ctx->set, x))
			continue;
		ctx->set = avl_insert(ctx->set, x, NULL);
		
		return mkConsValue(x, deferCall(nub_callback, ctx));
	}
}

static Value *nub_f(Value *xs)
{
	if (!is_list(xs))
		error("Second argument to nub must be a list");
	
	NubCtx *ctx = alloc(NubCtx);
	ctx->set = avl_new((AvlCompare)compare);
	ctx->xs  = xs;
	
	return nub_callback(ctx);
}

static Value *subscript_f(Value *xs, Value *idx)
{
	unsigned long long i;
	Value *x;
	
	if (!is_list(xs) || force(idx)->tag != V_INT)
		error("(!!): invalid arguments");
	
	if (idx->i < 0)
		error("(!!): negative index");
	
	i = idx->i;
	for (;;) {
		x = list_head(xs);
		if (x == NULL)
			error("(!!): index too large");
		
		if (i-- == 0)
			break;
		
		xs = list_tail(xs);
	}
	
	return x;
}

typedef struct {
	long long cur, end;
	long long step;
	bool forever;
} Range;

static Range *mkRange(long long start, long long end, long long step, bool forever)
{
	Range *r = alloc(Range);
	r->cur = start;
	r->end = end;
	r->step = step;
	r->forever = forever;
	return r;
}

static Value *range_callback(void *ctxp)
{
	Range     *range = ctxp;
	long long  end   = range->end;
	long long  step  = range->step;
	
	check_break();
	
	if (!range->forever) {
		if ((step > 0 && range->cur > end) ||
				(step < 0 && range->cur < end))
		{
			gc_free(range);
			return nil_v;
		}
	}
	
	Value *v = mkIntValue(range->cur);
	range->cur += step;
	
	return mkConsValue(v, deferCall(range_callback, range));
}

static Value *stepInf_f(Value *step, Value *start)
{
	if (force(step)->tag != V_INT || force(start)->tag != V_INT)
		error("stepInf: invalid arguments");
	
	return range_callback(mkRange(start->i, 0, step->i, true));
}

static Value *indexOf_f(Value *k, Value *xs)
{
	if (!is_list(xs))
		error("Second argument to indexOf must be a list (but it's %s)", typeStr(xs));
	
	long long  i = 0;
	Value     *x;
	
	list_foreach(xs, x) {
		if (compare(k, x) == 0)
			return mkIntValue(i);
		i++;
	}
	
	return mkIntValue(-1);
}

static long long gcd(long long a, long long b)
{
	if (a == 0 && b == 0)
		error("gcd 0 0 is undefined");
	
	if (a < 0)
		a = -a;
	if (b < 0)
		b = -b;
	
	while (b != 0) {
		long long tmp = a % b;
		a = b;
		b = tmp;
	}
	
	return a;
}

static long long lcm(long long a, long long b)
{
	if (a == 0 || b == 0)
		return 0;
	
	return abs(a / gcd(a, b) * b);
}

static Value *gcd_f(Value *a, Value *b)
{
	if (force(a)->tag != V_INT || force(b)->tag != V_INT)
		error("gcd: invalid arguments");
	
	return mkIntValue(gcd(a->i, b->i));
}

static Value *lcm_f(Value *a, Value *b)
{
	if (force(a)->tag != V_INT || force(b)->tag != V_INT)
		error("lcm: invalid arguments");
	
	return mkIntValue(lcm(a->i, b->i));
}

typedef struct {
	long  key;
	char *string;
} BindCtx;

static Value *add_binding_callback(void *ctxp)
{
	BindCtx *ctx = ctxp;
	add_binding(ctx->key, ctx->string);
	return unit_v;
}

static Value *add_binding_f(Value *key, Value *str)
{
	char buffer[256];
	
	force(key);
	if (key->tag != V_INT && key->tag != V_CHAR)
		error("bind: first argument must be a key code or a character");
	
	long k = key->tag == V_INT ? key->i : (unsigned char)key->c;
	
	suppressError = true;
	TRY
		readString(str, buffer, sizeof(buffer));
	ONERR
		suppressError = false;
		error("bind: second argument must be a string");
	ENDTRY
	suppressError = false;
	
	BindCtx *ctx = alloc(BindCtx);
	ctx->key = k;
	ctx->string = gc_strdup(buffer);
	
	return mkIOValue(add_binding_callback, ctx);
}

static Value *remove_binding_callback(void *ctxp)
{
	remove_binding((long)ctxp);
	
	return unit_v;
}

static Value *remove_binding_f(Value *key)
{
	force(key);
	if (key->tag != V_INT && key->tag != V_CHAR)
		error("bind: first argument must be a key code or a character");
	
	long k = key->tag == V_INT ? key->i : (unsigned char)key->c;
	
	return mkIOValue(remove_binding_callback, (void*)k);
}

struct {
	const char *name;
	BinaryFunc  func;
} binary_ops[] = {
	{"typeEq",    typeEq_f},
	{"~",         typeEq_f},
	{"seq",       seq_f},
	{"$",         apply_f},
	{"$!",        forceApply_f},
	{":",         cons_f},
	{"||",        or_f},
	{"&&",        and_f},
	{"+",         plus_f},
	{"-",         minus_f},
	{"*",         times_f},
	{"/",         div_f},
	{"^",         pow_f},
	{"logBase",   logBase_f},
	{"div",       div_f},
	{"mod",       mod_f},
	{"quot",      quot_f},
	{"rem",       rem_f},
	{"gcd",       gcd_f},
	{"lcm",       lcm_f},
	{"compare",   compare_f},
	{"compareR",  compareR_f},
	{"==",        eq_f},
	{"/=",        ne_f},
	{"\x9d",      ne_f},
	{"<",         lt_f},
	{"<=",        le_f},
	{"\x9c",      le_f},
	{">",         gt_f},
	{">=",        ge_f},
	{"\x9e",      ge_f},
	{">>=",       bind_f},
	{"map",       map_f},
	{"concatMap", concatMap_f},
	{"!!",        subscript_f},
	{"stepInf",   stepInf_f},
	{"indexOf",   indexOf_f},
	
	{"writeFile",        writeFile_f},
	{"writeBinaryFile",  writeBinaryFile_f},
	{"appendFile",       appendFile_f},
	{"appendBinaryFile", appendBinaryFile_f},
	
	{"bind",      add_binding_f}
};


/*************************** N-ary functions *************************/

struct drawLine {
	short x0, y0, x1, y1;
	short attr;
};

static Value *drawLine_callback(void *ctxp)
{
	struct drawLine *ctx = ctxp;
	
	DrawClipLine(
		&(WIN_RECT){ctx->x0, ctx->y0, ctx->x1, ctx->y1},
		&(SCR_RECT){{0,0,LCD_WIDTH-1,LCD_HEIGHT-1}},
		ctx->attr);
	
	return unit_v;
}

static Value *drawLineAttr_f(List(Value) *vs)
{
	Value *attr = vs->next->next->next->next->item,
	      *x0   = vs->next->next->next->item,
	      *y0   = vs->next->next->item,
	      *x1   = vs->next->item,
	      *y1   = vs->item;
	
	if (force(attr)->tag != V_INT ||
	    force(x0)->tag   != V_INT ||
	    force(y0)->tag   != V_INT ||
	    force(x1)->tag   != V_INT ||
	    force(y1)->tag   != V_INT)
		error("Invalid arguments to drawLine");
	
	struct drawLine *ctx = alloc(struct drawLine);
	ctx->x0 = x0->i;
	ctx->x1 = x1->i;
	ctx->y0 = y0->i;
	ctx->y1 = y1->i;
	ctx->attr = attr->i;
	
	return mkIOValue(drawLine_callback, ctx);
}

static Value *foldl_f(List(Value) *vs)
{
	Value *f  = vs->next->next->item,
	      *z  = vs->next->item,
	      *xs = vs->item;
	Value *x;
	
	if (!is_list(xs))
		error("Third argument to foldl must be a list (but it's %s)", typeStr(xs));
	
	list_foreach(xs, x)
		z = forceApply(forceApply(f, z), x);
	
	return z;
}

static Value *foldr_f(List(Value) *vs)
{
	Value *f  = vs->next->next->item,
	      *z  = vs->next->item,
	      *xs = vs->item;
	
	if (!is_list(xs))
		error("Third argument to foldr must be a list (but it's %s)", typeStr(xs));
	
	Value *head = list_head(xs);
	
	if (head == NULL)
		return z;
	
	Value *foldr_fz = mkValue(V_NARY);
	foldr_fz->nary.func      = foldr_f;
	foldr_fz->nary.applied   = vs->next;
	foldr_fz->nary.remaining = 1;
	
	return deferApply(deferApply(f, head), deferApply(foldr_fz, list_tail(xs)));
}

static Value *step_f(List(Value) *vs)
{
	Value *step  = vs->next->next->item,
	      *start = vs->next->item,
	      *end   = vs->item;
	
	if (force(step)->tag  != V_INT ||
	    force(start)->tag != V_INT ||
	    force(end)->tag   != V_INT)
		error("step: invalid arguments");
	
	return range_callback(mkRange(start->i, end->i, step->i, false));
}

struct zipWith {
	Value *f, *xs, *ys;
};

static Value *zipWith_callback(void *ctxp)
{
	struct zipWith *ctx = ctxp;
	
	Value *xs = ctx->xs,
	      *ys = ctx->ys;
	
	Value  *x   = list_pop(&xs),
	       *y   = list_pop(&ys);
	
	if (x == NULL || y == NULL) {
		gc_free(ctx);
		return nil_v;
	}
	
	ctx->xs = xs;
	ctx->ys = ys;
	return mkConsValue(deferApply(deferApply(ctx->f, x), y), deferCall(zipWith_callback, ctx));
}

static Value *zipWith_f(List(Value) *vs)
{
	Value *f  = vs->next->next->item,
	      *xs = vs->next->item,
	      *ys = vs->item;
	
	if (!is_func(f))
		error("First argument to zipWith must be a function");
	if (!is_list(xs))
		error("Second argument to zipWith must be a list");
	if (!is_list(ys))
		error("Third argument to zipWith must be a list");
	
	struct zipWith *ctx = alloc(struct zipWith);
	ctx->f = f;
	ctx->xs = xs;
	ctx->ys = ys;
	
	return zipWith_callback(ctx);
}

struct {
	const char   *name;
	NaryFunc      func;
	unsigned int  argc;
} nary_ops[] = {
	{"drawLineAttr", drawLineAttr_f, 5},
	{"foldl",        foldl_f,        3},
	{"foldr",        foldr_f,        3},
	{"step",         step_f,         3},
	{"zipWith",      zipWith_f,      3}
};


/*************************** Let bindings ****************************/

static struct {
	const char *f;
	const char *expr;
} let_items[] = {
	{".",     "\\f\\g\\x f (g x)"},
	{"id",    "\\x x"},
	{"const", "\\x \\_ x"},
	
	// {"foldl2", "\\f\\z\\xs if null xs then z else foldl2 f (f z (head xs)) (tail xs)"},
	// {"foldr2", "\\f\\z\\xs if null xs then z else f (head xs) (foldr2 f z (tail xs))"},
	
	// {"zipWith",   "\\f \\xs \\ys if null xs then [] else if null ys then [] else f (head xs) (head ys) : zipWith f (tail xs) (tail ys)"},
	
	{"pair",    "\\x \\y [x,y]"},
	{"fst",     "head"},
	{"snd",     "\\p head (tail p)"},
	{"zip",     "zipWith pair"},
	{"unzip",   "foldr (\\p \\ps [fst p : fst ps, snd p : snd ps]) [[],[]]"},
	{"curry",   "\\f \\x \\y f [x,y]"},
	{"uncurry", "\\f \\p f (fst p) (snd p)"},
	
	{"foldl1",  "\\f \\xs if null xs then error \"foldl1: empty list\" else foldl f (head xs) (tail xs)"},
	{"foldr1",  "\\f \\xs if null xs then error \"foldr1: empty list\" else if null (tail xs) then head xs else f (head xs) (foldr1 f (tail xs))"},
	{"scanl",   "\\f \\z \\xs z : if null xs then [] else scanl f (f z (head xs)) (tail xs)"},
	{"scanl1",  "\\f \\xs if null xs then [] else scanl f (head xs) (tail xs)"},
	{"scanr",   "\\f \\z \\xs if null xs then [z] else (\\qs f (head xs) (head qs) : qs) (scanr f z (tail xs))"},
	{"scanr1",  "\\f \\xs if null xs then [] else if null (tail xs) then xs else (\\qs f (head xs) (head qs) : qs) (scanr1 f (tail xs))"},
	{"iterate", "\\f\\z z : iterate f (f z)"},
	{"repeat",  "\\x fix (x:)"},
	{"until",   "\\p \\f \\x if p x then x else until p f (f x)"},
	{"flip",    "\\f \\x \\y f y x"},
	
	// {"map",       "\\f foldr (\\x \\xs f x : xs) []"},
	// {"++",        "\\xs \\ys if null xs then ys else head xs : tail xs ++ ys"},
	{"++",        "\\xs \\ys foldr (:) ys xs"},
	{"replicate", "\\n\\x if n < 1 then [] else x : replicate (n - 1) x"},
	{"cycle",     "\\xs if null xs then undefined else fix (xs ++)"},
	// {"concat",    "foldr (++) []"},
	// {"concatMap", "\\f concat . map f"},
	// {"concat",    "concatMap id"},
	{"reverse",   "foldl (flip (:)) []"},
	{"nest",      "\\x [x]"},
	
	{"filter",    "\\pred foldr (\\x \\xs if pred x then x : xs else xs) []"},
	{"item",      "fix \\item \\n \\xs if n < 1 then head xs else item (n - 1) (tail xs)"},
	{"take",      "\\n \\xs if n < 1 then [] else if null xs then [] else head xs : take (n-1) (tail xs)"},
	{"drop",      "\\n \\xs if n < 1 then xs else if null xs then xs else drop (n-1) (tail xs)"},
	{"drop1",     "\\xs if null xs then [] else tail xs"},
	{"splitAt",   "\\n \\xs [take n xs, drop n xs]"},
	
	{"takeWhile", "\\p \\xs if null xs then [] else if p (head xs) then head xs : takeWhile p (tail xs) else []"},
	{"dropWhile", "\\p \\xs if null xs then [] else if p (head xs) then dropWhile p (tail xs) else xs"},
	
	// These take a (\x \y) return function rather than returning [x,y]
	// {"span_",     "\\pred \\xs \\ret if null xs then ret [] [] else if pred (head xs) then span_ pred (tail xs) (\\ys \\zs ret (head xs : ys) zs) else ret [] xs"},
	{"span_",     "\\p \\xs \\ret ret (takeWhile p xs) (dropWhile p xs)"},
	{"break_",    "\\p span_ (not . p)"},
	
	{"span",      "\\pred \\xs span_ pred xs pair"},
	{"break",     "\\pred \\xs break_ pred xs pair"},
	
	{"intersperse", "\\d \\xs if null xs then [] else if null (tail xs) then xs else head xs : d : intersperse d (tail xs)"},
	{"intercalate", "\\d concat . intersperse d"},
	
	{"groupBy",     "\\eq \\xs if null xs then [] else span_ (eq (head xs)) (tail xs) (\\ys \\zs (head xs : ys) : groupBy eq zs)"},
	{"group",       "groupBy (==)"},
	
	// {"elem",    "\\k \\xs if null xs then false else if head xs == k then true else k elem tail xs"},
	// {"elem",    "\\k foldr (\\a \\b a == k || b) false"},
	// {"notElem", "\\k \\xs not (k elem xs)"},
	{"elem",    "\\k \\xs indexOf k xs /= \xAD" "1"},
	{"notElem", "\\k \\xs indexOf k xs == \xAD" "1"},
	
	{"merge", "\\xs \\ys if null xs then ys else if null ys then xs else if head xs <= head ys then head xs : merge (tail xs) ys else head ys : merge xs (tail ys)"},
	{"sort",  "\\xs (\\len if len <= 1 then xs else merge (sort (take (len/2) xs)) (sort (drop (len/2) xs))) (length xs)"},
	
	{"min",    "\\x \\y if x <= y then x else y"},
	{"max",    "\\x \\y if x <= y then y else x"},
	
	{"minimum", "\\xs if null xs then error \"minimum: empty list\" else foldl1 min xs"},
	{"maximum", "\\xs if null xs then error \"maximum: empty list\" else foldl1 max xs"},
	
	{"negate", "\\n 0-n"},
	{"abs",    "\\n if n<0 then negate n else n"},
	// {"gcd_",   "\\x \\y if y==0 then x else gcd_ y (x rem y)"},
	// {"gcd",    "\\x \\y if x==0 && y==0 then error \"gcd 0 0 is undefined\" else gcd_ (abs x) (abs y)"},
	// {"lcm",    "\\x \\y if x==0 || y==0 then 0 else abs ((x quot (gcd x y)) * y)"},
	
	{"and", "foldr (&&) true"},
	{"or",  "foldr (||) false"},
	
	{"all", "\\p foldr (\\x \\b p x && b) true"},
	{"any", "\\p foldr (\\x \\b p x || b) false"},
	
	{">>", "\\m1 \\m2 m1 >>= \\x m2"},
	{"=<<", "flip (>>=)"},
	{"<<", "\\m2 \\m1 m1 >>= \\x m2"},
	{"join",    "\\z z >>= \\m m"},
	{"forever", "\\m m >> forever m"},
	
	{"sequence_rev", "\\ms \\xs if null ms then return xs else head ms >>= \\x sequence_rev (tail ms) (x : xs)"},
	{"sequence",     "\\ms sequence_rev ms [] >>= return . reverse"},
	{"sequence_",    "foldr (\\x \\xs x >> xs) (return ())"},
	{"mapM_",        "\\f foldr (\\x \\xs f x >> xs) (return ())"},
	{"mapM",         "\\f sequence . map f"},
	{"forM",         "flip mapM"},
	{"replicateM",   "\\n sequence . replicate n"},
	{"replicateM_",  "\\n \\a if n < 1 then return () else a >> replicateM_ (n-1) a"},
	
	{"fmap", "\\f \\m m >>= return . f"},
	{"<$>", "fmap"},
	{"ap", "\\mf \\mx mf >>= \\f mx >>= \\x return (f x)"},
	{"<*>", "ap"},
	
	{"sum",     "foldl (+) 0"},
	{"product", "foldl (*) 0"},
	
	// {"range",     "\\lo \\hi if lo > hi then [] else lo : range (lo+1) hi"},
	// {"countFrom", "\\n n : countFrom (n+1)"},
	{"range",      "step 1"},
	{"rangeR",     "step \xAD" "1"},
	{"countFrom",  "stepInf 1"},
	{"countFromR", "stepInf \xAD" "1"},
	
	{"putStrLn", "\\str putStr str >> putChar '\\n'"},
	{"putStrSp", "\\s putStr s >> putChar ' '"},
	{"printSp",  "\\x print x >> putChar ' '"},
	
	{"isNewline",  "\\c c == '\\n' || c == '\\r'"},
	{"notNewline", "\\c c /= '\\n' && c /= '\\r'"},
	{"lines",      "\\str if null str then [] else takeWhile notNewline str : lines (drop1 (dropWhile notNewline str))"},
	{"words",      "\\s (\\s' if null s' then [] else break_ isSpace s' (\\w \\s'' w : words s'')) (dropWhile isSpace s)"},
	{"unlines",    "concatMap (++ \"\\n\")"},
	{"unwords",    "\\ws if null ws then \"\" else foldr1 (\\w \\s w ++ ' ':s) ws"},
	
	{"A_REVERSE",  "0"},
	{"A_NORMAL",   "1"},
	{"A_XOR",      "2"},
	{"A_SHADED",   "3"},
	{"A_REPLACE",  "4"},
	{"A_OR",       "5"},
	{"A_AND",      "6"},
	{"A_THICK1",   "7"},
	{"A_SHADE_V",  "8"},
	{"A_SHADE_H",  "9"},
	{"A_SHADE_NS", "10"},
	{"A_SHADE_PS", "11"},
	
	{"drawLine",   "drawLineAttr A_NORMAL"},
	{"withFont",   "\\new \\a getFont >>= \\old setFont new >> a >>= \\result setFont old >> return result"}
};


/****************************** Aliases ******************************/
static struct {
	const char *name;
	const char *orig;
} aliases[] = {
	{"\xb7",   "."},
	{"\x1f",   "elem"},
	{"/\x1f",  "notElem"},
	{"..",     "range"},
	{"...",    "countFrom"},
	{"\xa0",   "countFrom"},
	{"each",   "mapM_"},
	{"True",   "true"},
	{"False",  "false"}
};


/****************************** newEnv() *****************************/

Value *false_v, *true_v;
Value *fix_v, *foldr_v;
Value *step_v, *stepInf_v;
Value *unit_v, *nil_v;

Value *undefined_v, *infinite_loop_v;

Value *bindFunc_v;

Environment newEnv(void)
{
	Environment  env = avl_new((AvlCompare)strcmp);
	unsigned int i;
	
	unit_v      = mkUnit();
	nil_v       = mkValue(V_NIL);
	
	false_v     = mkBoolValue(false);
	true_v      = mkBoolValue(true);
	undefined_v = mkValue(V_UNDEFINED);
	infinite_loop_v = mkValue(V_INFINITE_LOOP);
	
	env = addEnv(env, "false",     false_v);
	env = addEnv(env, "true" ,     true_v);
	env = addEnv(env, "undefined", undefined_v);
	
	env = addEnv(env, "lcdWidth", mkIntValue(LCD_WIDTH));
	env = addEnv(env, "lcdHeight", mkIntValue(LCD_HEIGHT));
	
	env = addEnv(env, "small",  mkIntValue(F_4x6));
	env = addEnv(env, "medium", mkIntValue(F_6x8));
	env = addEnv(env, "large",  mkIntValue(F_8x10));
	
	env = addEnv(env, "primes", primes_f());
	
	for (i = 0; i < sizeof(nullary_io_ops) / sizeof(*nullary_io_ops); i++)
		env = addEnv(env, nullary_io_ops[i].name, mkIOValue(nullary_io_ops[i].callback, nullary_io_ops[i].ctx));
	
	for (i = 0; i < sizeof(unary_ops) / sizeof(*unary_ops); i++) {
		Value *v = mkValue(V_UNARY);
		v->unary = unary_ops[i].func;
		env = addEnv(env, unary_ops[i].name, v);
	}
	
	for (i = 0; i < sizeof(binary_ops) / sizeof(*binary_ops); i++) {
		Value *v = mkValue(V_BINARY);
		v->binary.func = binary_ops[i].func;
		env = addEnv(env, binary_ops[i].name, v);
	}
	
	for (i = 0; i < sizeof(nary_ops) / sizeof(*nary_ops); i++) {
		Value *v = mkValue(V_NARY);
		
		v->nary.func      = nary_ops[i].func;
		v->nary.applied   = NULL;
		v->nary.remaining = nary_ops[i].argc;
		
		env = addEnv(env, nary_ops[i].name, v);
	}
	
	for (i = 0; i < sizeof(let_items) / sizeof(*let_items); i++) {
		TRY
			env = addLet(env, let_items[i].f, let_items[i].expr);
		ONERR
			printf("   in let %s = \xa0\n", let_items[i].f);
			ngetchx();
			
			if (errCode == ER_MEMORY)
				PASS;
		ENDTRY
	}
	
	bindFunc_v = evaluate(parseExpression("\\m \\k \\x k (m x) x"), NULL);
	
	for (i = 0; i < sizeof(aliases) / sizeof(*aliases); i++)
		env = addEnv(env, aliases[i].name, lookupEnv(env, aliases[i].orig));
	
	fix_v       = lookupEnv(env, "fix");
	foldr_v     = lookupEnv(env, "foldr");
	step_v      = lookupEnv(env, "step");
	stepInf_v   = lookupEnv(env, "stepInf");
	
	defaultEnv = env;
	
	return env;
}
