/*
 * Copyright (C) 2010 Joseph Adams <joeyadams3.14159@gmail.com>
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#include "lambda.h"

// Input value must be forced in advance.
Value *perform(Value *m)
{
	while (m->tag == V_BIND)
		m = forceApply(m->bind.k, perform(m->bind.m));
	
	if (m->tag == V_IO)
		return m->io.func(m->io.ctx);
	else if (m->tag == V_RETURN)
		return m->return_;
	else
		error("Cannot perform %s", typeStr(m));
}

void print(Value *v)
{
	char buffer[22];
	
	switch (v->tag) {
		case V_INT:
			puts(showInt(v->i, buffer));
			break;
		
		case V_CHAR:
			puts(showChar(v->c, buffer));
			break;
		
		case V_UNIT:
			putchar('(');
			putchar(')');
			break;
		
		case V_NIL:
			printf("[]");
			break;
		
		case V_CONS:
			{
				Value *x;
				bool   prev = false;
				
				putchar('[');
				list_foreach(v, x) {
					if (prev)
						putchar(',');
					else
						prev = true;
					
					print(force(x));
				}
				putchar(']');
			}
			break;
		
		case V_LAMBDA:
		case V_UNARY:
		case V_BINARY:
		case V_BINARY1:
		case V_NARY:
			printf("<closure>");
			break;
		
		case V_UNDEFINED:
			printf("<undefined>");
			break;
		
		case V_INFINITE_LOOP:
			printf("<infinite loop>");
			break;
		
		case V_IO:
		case V_BIND:
			printf("<IO>");
			break;
		
		case V_RETURN:
			puts("return ");
			print(v->return_);
			break;
		
		case V_BOOL:
			puts(v->b ? "true" : "false");
			break;
		
		default:
			if (IS_THUNK(v)) {
				printf("<thunk>");
				break;
			}
			
			error("Corrupt data structure passed to print()");
	}
}

void printValueDebug(Value *v)
{
	assert(v != NULL);
	
	char buf[64], buf2[64];
	
	switch (v->tag) {
		case V_LAMBDA:
			puts("(\\$ ");
			printLExpr(v->lambda.expr);
			putchar(')');
			break;
		
		case V_UNARY:
			printf("{%s}", show_ptr(v->unary, buf));
			break;
		
		case V_BINARY:
			printf("{%s}", show_ptr(v->binary.func, buf));
			break;
		
		case V_BINARY1:
			printf("{%s %s}", show_ptr(v->binary.func, buf), gc_show_ptr(v->binary.v1, buf2));
			break;
		
		case V_NARY:
			if (v->nary.applied == NULL) {
				printf("{%s}", show_ptr(v->nary.func, buf));
			} else {
				void loop(List(Value) *i) {
					char buf[64];
					if (i->next) {
						loop(i->next);
						putchar(' ');
					}
					printf("%s", gc_show_ptr(i->item, buf));
				}
				
				printf("{%s ", show_ptr(v->nary.func, buf));
				loop(v->nary.applied);
				putchar('}');
			}
			break;
		
		default:
			print(v);
	}
}

void printLExpr(LExpr *expr)
{
	assert(expr != NULL);
	
	switch (expr->tag) {
		case L_VAR:
			printf("#%u", expr->var);
			break;
		
		case L_VALUE:
			printValueDebug(expr->value);
			break;
		
		case L_LIST:
			{
				void pl(List *i) {
					if (i->next) {
						pl(i->next);
						putchar(',');
					}
					printLExpr(i->item);
				}
				
				putchar('[');
				if (expr->list)
					pl(expr->list);
				putchar(']');
			}
			break;
		
		case L_LAMBDA:
			puts("(\\# ");
			printLExpr(expr->lambda);
			putchar(')');
			break;
		
		case L_AP:
			if (expr->ap.f->tag == L_IF_THEN) {
				putchar('(');
				printLExpr(expr->ap.f);
				putchar(')');
			} else {
				printLExpr(expr->ap.f);
			}
			putchar(' ');
			if (expr->ap.x->tag == L_AP || expr->ap.x->tag == L_IF_THEN) {
				putchar('(');
				printLExpr(expr->ap.x);
				putchar(')');
			} else {
				printLExpr(expr->ap.x);
			}
			break;
		
		case L_IF_THEN:
			puts("if ");
			if (expr->if_then.pred->tag == L_IF_THEN) {
				putchar('(');
				printLExpr(expr->if_then.pred);
				putchar(')');
			} else {
				printLExpr(expr->if_then.pred);
			}
			puts(" then ");
			printLExpr(expr->if_then.on_true);
			puts(" else ");
			printLExpr(expr->if_then.on_false);
			break;
		
		default:
			error("Corrupt data structure passed to printLExpr()");
	}
}

const char *typeStr(Value *v)
{
	switch (v->tag) {
		case V_INT:
			return "Int";
		
		case V_CHAR:
			return "Char";
		
		case V_BOOL:
			return "Bool";
		
		case V_UNIT:
			return "()";
		
		case V_NIL:
			return "[a]";
		
		case V_CONS:
			return "[a]";
		
		case V_LAMBDA:
		case V_UNARY:
		case V_BINARY:
		case V_BINARY1:
		case V_NARY:
			return "<closure>";
		
		case V_UNDEFINED:
			return "<undefined>";
		
		case V_INFINITE_LOOP:
			return "<infinite loop>";
		
		case V_IO:
		case V_BIND:
			return "IO";
		
		case V_RETURN:
			return "Monad m => m";
		
		default:
			if (IS_THUNK(v))
				return "<thunk>";
			
			error("Corrupt data structure passed to typeStr()");
	}
}

static const unsigned char typeID[] = {
	[V_INT]       = T_INT,
	[V_CHAR]      = T_CHAR,
	[V_BOOL]      = T_BOOL,
	[V_UNIT]      = T_UNIT,
	
	[V_NIL]       = T_LIST,
	[V_CONS]      = T_LIST,
	
	[V_LAMBDA]    = T_FUNC,
	[V_UNARY]     = T_FUNC,
	[V_BINARY]    = T_FUNC,
	[V_BINARY1]   = T_FUNC,
	[V_NARY]      = T_FUNC,
	
	[V_IO]        = T_IO,
	[V_BIND]      = T_IO,
	
	[V_RETURN]    = T_RETURN
};

ValueType getType(Value *v)
{
	unsigned int tag = force(v)->tag;
	
	if (tag >= sizeof(typeID) / sizeof(*typeID) ||
	    typeID[tag] == 0)
		error("corrupt data structure passed to getType");
	
	return typeID[tag];
}

void readString(Value *str, char *out, size_t max)
{
	Value *x;
	
	assert(max > 0);
	
	if (!is_list(str))
		error("Expected string argument");
	
	list_foreach(str, x) {
		if (force(x)->tag != V_CHAR)
			error("Expected string argument, but list has an item of type %s", typeStr(x));
		
		if (max > 1) {
			*out++ = x->c;
			max--;
		} else {
			error("String too long");
		}
	}
	
	*out = 0;
}

Value *buildString(const char *str)
{
	Value *head = nil_v, **tail = &head;
	
	for (; *str != '\0'; str++) {
		*tail = mkConsValue(mkCharValue(*str), *tail);
		tail = &(*tail)->cons.xs;
	}
	
	return head;
}

void chomp(char *buffer)
{
	if (!*buffer)
		return;
	
	while (*buffer)
		buffer++;
	
	if (buffer[-1] == '\n')
		buffer[-1] = 0;
}
